package com.exner.tools.jkbikemechanicaldisasterprevention.database.tools

import androidx.room.ColumnInfo
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.convertPotentiallyEmptyStringToLocalDate
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.potentiallyEmptyLocalDateToString
import kotlinx.datetime.LocalDate
import kotlinx.serialization.Serializable

@Serializable
data class ComponentCsv(
    val name: String,
    val description: String?,
    @ColumnInfo(name = "acquisition_date")
    val acquisitionDate: LocalDate?,
    @ColumnInfo(name = "first_use_date")
    val firstUseDate: LocalDate?,
    @ColumnInfo(name = "last_check_date")
    val lastCheckDate: LocalDate?,
    @ColumnInfo(name = "last_check_mileage")
    val lastCheckMileage: Int?,
    @ColumnInfo(name = "current_mileage")
    val currentMileage: Int?,
    @ColumnInfo(name = "mileage_offset")
    val mileageOffset: Int?,
    @ColumnInfo(name = "bike_uid")
    val bikeUid: Long?,
    @ColumnInfo(name = "title_for_automatic_activities")
    val titleForAutomaticActivities: String?,
    @ColumnInfo(name = "wear_level")
    val wearLevel: WearLevel?,
    @ColumnInfo(name = "retirement_date")
    val retirementDate: LocalDate?,
    @ColumnInfo(name = "retirement_reason")
    val retirementReason: RetirementReason?,
    @ColumnInfo(name = "check_interval_miles")
    val checkIntervalMiles: Int?,
    @ColumnInfo(name = "check_interval_days")
    val checkIntervalDays: Int?,
    @ColumnInfo(name = "notify_when_activity_created")
    val notifyWhenActivityCreated: Boolean?,
    val uid: Long? = 0L
) {
    companion object {
        fun convertPotentiallyEmptyStringToInt(input: String?): Int? {
            if (input.isNullOrEmpty()) {
                return null
            }
            return input.toInt()
        }

        fun convertPotentiallyEmptyStringToLong(input: String?): Long? {
            if (input.isNullOrEmpty()) {
                return null
            }
            return input.toLong()
        }

        fun convertPotentiallyEmptyStringToBoolean(input: String?): Boolean? {
            if (input.isNullOrEmpty()) {
                return null
            }
            return input.toBoolean()
        }

        fun createFromMap(valueMap: Map<String, String>): ComponentCsv {
            return ComponentCsv(
                name = if (valueMap["Name"].isNullOrEmpty()) { "No name!" } else { valueMap["Name"]!! },
                description = if (valueMap["Description"].isNullOrEmpty()) { null } else { valueMap["Description"] },
                acquisitionDate = convertPotentiallyEmptyStringToLocalDate(valueMap["AcquisitionDate"]),
                firstUseDate = convertPotentiallyEmptyStringToLocalDate(valueMap["FirstUseDate"]),
                lastCheckDate = convertPotentiallyEmptyStringToLocalDate(valueMap["LastCheckDate"]),
                lastCheckMileage = convertPotentiallyEmptyStringToInt(valueMap["LastCheckMileage"]),
                currentMileage = convertPotentiallyEmptyStringToInt(valueMap["CurrentMileage"]),
                mileageOffset = convertPotentiallyEmptyStringToInt(valueMap["MileageOffset"]),
                bikeUid = convertPotentiallyEmptyStringToLong(valueMap["BikeUid"]),
                titleForAutomaticActivities = if (valueMap["TitleForAutomaticActivities"].isNullOrEmpty()) { null } else {  valueMap["TitleForAutomaticActivities"]!! },
                wearLevel = wearLevelFromString(valueMap["WearLevel"]),
                retirementDate = convertPotentiallyEmptyStringToLocalDate(valueMap["RetirementDate"]),
                retirementReason = retirementReasonFromString(valueMap["RetirementReason"]),
                checkIntervalMiles = convertPotentiallyEmptyStringToInt(valueMap["CheckIntervalMiles"]),
                checkIntervalDays = convertPotentiallyEmptyStringToInt(valueMap["CheckIntervalDays"]),
                notifyWhenActivityCreated = convertPotentiallyEmptyStringToBoolean(valueMap["NotifyWhenActivityCreated"]),
                uid = convertPotentiallyEmptyStringToLong(valueMap["Uid"]) ?: 0L
            )
        }

        fun createFromComponent(component: Component): ComponentCsv {
            return ComponentCsv(
                name = component.name,
                description = component.description,
                acquisitionDate = component.acquisitionDate,
                firstUseDate = component.firstUseDate,
                lastCheckDate = component.lastCheckDate,
                lastCheckMileage = component.lastCheckMileage,
                currentMileage = component.currentMileage,
                mileageOffset = component.mileageOffset,
                bikeUid = component.bikeUid,
                titleForAutomaticActivities = component.titleForAutomaticActivities,
                wearLevel = component.wearLevel,
                retirementDate = component.retirementDate,
                retirementReason = component.retirementReason,
                checkIntervalMiles = component.checkIntervalMiles,
                checkIntervalDays = component.checkIntervalDays,
                notifyWhenActivityCreated = component.notifyWhenActivityCreated,
                uid = component.uid
            )
        }

        fun getHeaders(): List<String> {
            return listOf(
                "Name",
                "Description",
                "AcquisitionDate",
                "FirstUseDate",
                "LastCheckDate",
                "LastCheckMileage",
                "CurrentMileage",
                "MileageOffset",
                "BikeUid",
                "TitleForAutomaticActivities",
                "WearLevel",
                "RetirementDate",
                "RetirementReason",
                "CheckIntervalMiles",
                "CheckIntervalDays",
                "NotifyWhenActivityCreated",
                "Uid"
            )
        }
    }
}

fun ComponentCsv.asStringList(): List<Any?> {
    val result: MutableList<Any?> = mutableListOf()
    result.add(this.name)
    result.add(this.description)
    result.add(potentiallyEmptyLocalDateToString(this.acquisitionDate))
    result.add(potentiallyEmptyLocalDateToString(this.firstUseDate))
    result.add(potentiallyEmptyLocalDateToString(this.lastCheckDate))
    result.add(this.lastCheckMileage)
    result.add(this.currentMileage)
    result.add(this.mileageOffset)
    result.add(this.bikeUid)
    result.add(this.titleForAutomaticActivities)
    result.add(if (this.wearLevel != null) { this.wearLevel.toString() } else { null })
    result.add(potentiallyEmptyLocalDateToString(this.retirementDate))
    result.add(if (this.retirementReason != null) { this.retirementReason.toString() } else { null })
    result.add(this.checkIntervalMiles)
    result.add(this.checkIntervalDays)
    result.add(this.notifyWhenActivityCreated.toString())
    result.add(this.uid)
    return result
}

fun ComponentCsv.asComponent(): Component {
    return Component(
        name = this.name,
        description = this.description ?: "No description",
        acquisitionDate = this.acquisitionDate,
        firstUseDate = this.firstUseDate,
        lastCheckDate = this.lastCheckDate,
        lastCheckMileage = this.lastCheckMileage,
        currentMileage = this.currentMileage,
        mileageOffset = this.mileageOffset,
        bikeUid = this.bikeUid,
        titleForAutomaticActivities = this.titleForAutomaticActivities,
        wearLevel = this.wearLevel,
        retirementDate = this.retirementDate,
        retirementReason = this.retirementReason,
        checkIntervalMiles = this.checkIntervalMiles,
        checkIntervalDays = this.checkIntervalDays,
        notifyWhenActivityCreated = this.notifyWhenActivityCreated ?: false,
        lastCheckNotificationMileage = null,
        lastCheckNotificationDate = null,
        uid = this.uid ?: 0L,
    )
}