package com.exner.tools.jkbikemechanicaldisasterprevention.database

import androidx.annotation.WorkerThread
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.BackgroundWorkLog
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.IntervalsSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.StravaSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ActivityWithBikeAndRideData
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.RetiredComponents
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.RideWithBikeAndActivityTotals
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ShelvedComponents
import kotlinx.coroutines.flow.Flow
import javax.inject.Inject

class KJsRepository @Inject constructor(private val kjsDAO: KJsDAO) {

    // Room executes all queries on a separate thread.
    // Observed Flow will notify the observer when the data has changed.
    val observeBikes: Flow<List<Bike>> = kjsDAO.observeBikesOrderedByLastUsedDate()

    val observeAllBikes: Flow<List<Bike>> = kjsDAO.observeAllBikesOrderedByLastUsedDate()

    val observerNumberOfBikes: Flow<Int> = kjsDAO.observeNumberOfBikes()

    val observeNumberOfAllBikes: Flow<Int> = kjsDAO.observeNumberOfAllBikes()

    val observeNumberOfStravaConnectedBikes: Flow<Int> =
        kjsDAO.observeNumberOfStravaConnectedBikes()

    val observeStravaSummaryGear: Flow<List<StravaSummaryGear>> = kjsDAO.observeStravaSummaryGear()

    val observeNumberOfIntervalsConnectedBikes: Flow<Int> =
        kjsDAO.observeNumberOfIntervalsConnectedBikes()

    val observeIntervalsSummaryGear: Flow<List<IntervalsSummaryGear>> =
        kjsDAO.observeIntervalsSummaryGear()

    val observeActivities: Flow<List<Activity>> = kjsDAO.observeActivitiesOrderedByTitle()
    val observeActivitiesButNotForCompletedRides: Flow<List<Activity>> = kjsDAO.observeActivitiesButNotForCompletedRides()

    val observeActivityWithBikeAndRideDataOrderedByDueDate: Flow<List<ActivityWithBikeAndRideData>> =
        kjsDAO.observeActivitiesWithBikeDataOrderedByDueDate()

    val observeActivityWithBikeAndRideDataAndDueDateOrderedByDueDate: Flow<List<ActivityWithBikeAndRideData>> =
        kjsDAO.observeActivitiesWithBikeDataAndDueDateOrderedByDueDate()

    val observeNumberOfActivities: Flow<Int> = kjsDAO.observeNumberOfActivities()

    val observeTemplateActivity: Flow<List<TemplateActivity>> = kjsDAO.observeTemplateActivities()
    val observeCustomTemplateActivity: Flow<List<TemplateActivity>> =
        kjsDAO.observeCustomTemplateActivities()
    val observeNumberOfCustomTemplateActivities: Flow<Int> =
        kjsDAO.observeNumberOfCustomTemplateActivities()

    val observeComponents: Flow<List<Component>> = kjsDAO.observeComponentsOrderedByName()

    val observeNonRetiredComponents: Flow<List<Component>> =
        kjsDAO.observeNonRetiredComponentsOrderedByName()

    val observeNumberOfNonRetiredComponents: Flow<Int> =
        kjsDAO.observeNumberOfNonRetiredComponents()

    val observeRetiredComponents: Flow<List<RetiredComponents>> = kjsDAO.observeRetiredComponents()

    val observeNumberOfRetiredComponents: Flow<Int> = kjsDAO.observeNumberOfRetiredComponents()

    val observerShelvedComponents: Flow<List<ShelvedComponents>> = kjsDAO.observeShelvedComponents()

    val observeBackgroundWorkLog: Flow<List<BackgroundWorkLog>> =
        kjsDAO.observeBackgroundWorkLog()

    val observeRides: Flow<List<Ride>> = kjsDAO.observeRides()
    val observeRidesWithBikeAndActivityTotals: Flow<List<RideWithBikeAndActivityTotals>> =
        kjsDAO.observeRidesWithBikeAndActivityTotals()
    val observeRidesInPreparation: Flow<List<Ride>> = kjsDAO.observeRidesInPreparation()

    @WorkerThread
    suspend fun getActivityByUid(activityUid: Long): Activity? {
        return kjsDAO.getActivityByUid(activityUid)
    }

    @WorkerThread
    suspend fun insertActivity(activity: Activity) {
        kjsDAO.insertActivity(activity)
    }

    @WorkerThread
    suspend fun updateActivity(activity: Activity) {
        kjsDAO.updateActivity(activity)
    }

    @WorkerThread
    suspend fun deleteActivityByUid(activityUid: Long) {
        kjsDAO.deleteActivityByUid(activityUid)
    }

    @WorkerThread
    suspend fun deleteActivitiesForRide(rideUid: Long) {
        kjsDAO.deleteActivitiesForRide(rideUid)
    }

    @WorkerThread
    suspend fun getNumberOfActivitiesForRideLevel(rideLevel: Int): Int {
        return kjsDAO.getNumberOfActivitiesForRideLevel(rideLevel)
    }

    @WorkerThread
    suspend fun getAllActivities(): List<Activity> {
        return kjsDAO.getAllActivitiesOrderedByTitle()
    }

    @WorkerThread
    suspend fun getAllActivitiesButNotForClosedRides(): List<Activity> {
        return kjsDAO.getAllActivitiesButNotForClosedRides()
    }

    @WorkerThread
    suspend fun deleteAllActivities() {
        kjsDAO.deleteAllActivities()
    }

    @WorkerThread
    suspend fun getTemplateActivity(templateActivityUid: Long): TemplateActivity? {
        return kjsDAO.getTemplateActivityByUid(templateActivityUid)
    }

    @WorkerThread
    suspend fun getTemplateActivitiesForRideLevel(rideLevel: Int): List<TemplateActivity> {
        return kjsDAO.getTemplateActivityForRideLevelOrNoRideLevel(rideLevel)
    }

    @WorkerThread
    suspend fun getTemplateActivityByUid(uid: Long): TemplateActivity? {
        return kjsDAO.getTemplateActivityByUid(uid)
    }

    @WorkerThread
    suspend fun insertTemplateActivity(templateActivity: TemplateActivity) {
        kjsDAO.insertTemplateActivity(templateActivity)
    }

    @WorkerThread
    suspend fun updateTemplateActivity(templateActivity: TemplateActivity) {
        kjsDAO.updateTemplateActivity(templateActivity)
    }

    @WorkerThread
    suspend fun deleteTemplateActivityByUid(templateActivityUid: Long) {
        kjsDAO.deleteTemplateActivityByUid(templateActivityUid)
    }

    @WorkerThread
    suspend fun getAllTemplateActivities(): List<TemplateActivity> {
        return kjsDAO.getAllTemplateActivities()
    }

    @WorkerThread
    suspend fun deleteAllBuiltInTemplateActivities() {
        kjsDAO.deleteAllBuiltInTemplateActivities()
    }

    @WorkerThread
    suspend fun getBikeByUid(uid: Long): Bike? {
        return kjsDAO.getBikeByUid(uid)
    }

    @WorkerThread
    suspend fun getBikeForStravaGearId(gearId: String): Bike? {
        return kjsDAO.getBikeForStravaGearId(gearId)
    }

    @WorkerThread
    suspend fun getBikeForIntervalsGearId(gearId: String): Bike? {
        return kjsDAO.getBikeForIntervalsGearId(gearId)
    }

    @WorkerThread
    suspend fun insertBike(bike: Bike): Long {
        return kjsDAO.insertBike(bike)
    }

    @WorkerThread
    suspend fun updateBike(bike: Bike) {
        kjsDAO.updateBike(bike)
    }

    @WorkerThread
    suspend fun deleteBike(bike: Bike) {
        kjsDAO.deleteBike(bike)
    }

    @WorkerThread
    suspend fun getAllBikes(): List<Bike> {
        return kjsDAO.getAllBikesOrderedByLastUsedDate()
    }

    @WorkerThread
    suspend fun deleteAllBikes() {
        kjsDAO.deleteAllBikes()
    }

    @WorkerThread
    suspend fun getAllStravaSummaryGear(): List<StravaSummaryGear> {
        return kjsDAO.getAllStravaSummaryGearByDistance()
    }

    @WorkerThread
    suspend fun getStravaSummaryGearById(id: String): StravaSummaryGear? {
        return kjsDAO.getStravaSummaryGearById(id)
    }

    @WorkerThread
    suspend fun insertStravaSummaryGear(gear: StravaSummaryGear) {
        kjsDAO.insertStravaSummaryGear(gear)
    }

    @WorkerThread
    suspend fun updateStravaSummaryGear(gear: StravaSummaryGear) {
        kjsDAO.updateStravaSummaryGear(gear)
    }

    @WorkerThread
    suspend fun getAllIntervalsSummaryGear(): List<IntervalsSummaryGear> {
        return kjsDAO.getAllIntervalsSummaryGearByDistance()
    }

    @WorkerThread
    suspend fun getIntervalsSummaryGearById(id: String): IntervalsSummaryGear? {
        return kjsDAO.getIntervalsSummaryGearById(id)
    }

    @WorkerThread
    suspend fun insertIntervalsSummaryGear(gear: IntervalsSummaryGear) {
        kjsDAO.insertIntervalsSummaryGear(gear)
    }

    @WorkerThread
    suspend fun updateIntervalsSummaryGear(gear: IntervalsSummaryGear) {
        kjsDAO.updateIntervalsSummaryGear(gear)
    }

    @WorkerThread
    suspend fun insertRide(ride: Ride): Long {
        return kjsDAO.insertRide(ride)
    }

    @WorkerThread
    suspend fun updateRide(ride: Ride) {
        kjsDAO.updateRide(ride)
    }

    @WorkerThread
    suspend fun getAllRides(): List<Ride> {
        return kjsDAO.getAllRides()
    }

    @WorkerThread
    suspend fun deleteAllRides() {
        kjsDAO.deleteAllRides()
    }

    @WorkerThread
    suspend fun getRideByUid(rideUid: Long): Ride? {
        return kjsDAO.getRideByUid(rideUid)
    }

    @WorkerThread
    suspend fun insertComponent(component: Component): Long {
        return kjsDAO.insertComponent(component)
    }

    @WorkerThread
    suspend fun getAllComponents(): List<Component> {
        return kjsDAO.getAllComponents()
    }

    @WorkerThread
    suspend fun getComponentByUid(uid: Long): Component? {
        return kjsDAO.getComponentByUid(uid)
    }

    @WorkerThread
    suspend fun deleteAllComponents() {
        kjsDAO.deleteAllComponents()
    }

    @WorkerThread
    suspend fun updateComponent(component: Component) {
        kjsDAO.updateComponent(component)
    }

    @WorkerThread
    suspend fun deleteComponent(component: Component) {
        kjsDAO.deleteComponent(component)
    }

    @WorkerThread
    suspend fun getAllComponentsForBike(bikeUid: Long): List<Component> {
        return kjsDAO.getAllComponentsForBike(bikeUid)
    }

    @WorkerThread
    suspend fun getNumberOfComponentsForBike(bikeUid: Long): Int {
        return kjsDAO.getNumberOfComponentsForBike(bikeUid)
    }

    @WorkerThread
    suspend fun insertBackgroundWorkLog(logEntry: BackgroundWorkLog) {
        kjsDAO.insertAAGLogEntry(logEntry)
    }

    @WorkerThread
    suspend fun getActivitiesForRide(rideUid: Long): List<Activity> {
        return kjsDAO.getActivitiesForRide(rideUid)
    }

    // stuff for cleaning up 3rd-party data

    @WorkerThread
    suspend fun getNumberOfBikesWithThirdPartyData(): Int {
        return kjsDAO.getNumberOfBikesWithThirdPartyData()
    }

    @WorkerThread
    suspend fun getNumberOfImportedGearStrava(): Int {
        return kjsDAO.getNumberOfImportedGearStrava()
    }

    @WorkerThread
    suspend fun getNumberOfImportedGearIntervals(): Int {
        return kjsDAO.getNumberOfImportedGearIntervals()
    }

    @WorkerThread
    suspend fun deleteAllIntervalsSummaryGear() {
        kjsDAO.deleteAllIntervalsSummaryGear()
    }

    @WorkerThread
    suspend fun deleteAllStravaSummaryGear() {
        kjsDAO.deleteAllStravaSummaryGear()
    }
}