package com.exner.tools.jkbikemechanicaldisasterprevention.database

import android.util.Log
import androidx.room.RoomDatabase
import androidx.sqlite.db.SupportSQLiteDatabase
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.loadTemplateActivitiesJsonAndImport
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.SupportedLanguages
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import javax.inject.Provider

private const val TAG = "KJsDBCallback"

class KJsDatabaseCallback(
    private val provider: Provider<KJsDAO>
) : RoomDatabase.Callback() {

    override fun onCreate(db: SupportSQLiteDatabase) {
        super.onCreate(db)
        CoroutineScope(Dispatchers.IO).launch {
            val localeToLoad = SupportedLanguages.determineLocaleToLoad()
            Log.d(TAG, "Going to load $localeToLoad...")
            loadTemplateActivitiesJsonAndImport(
                language = localeToLoad,
                importDataCallback = { newTAs ->
                    updateTemplateActivitiesInDatabase(newTAs)
                },
                errorCallback = {
                    CoroutineScope(Dispatchers.IO).launch {
                        val creator = PopulateDatabaseHelpersViaProvider(provider)
                        creator.createFallbackTemplateActivities()
                    }
                }
            )
        }
    }

    private fun updateTemplateActivitiesInDatabase(templateActivities: List<TemplateActivity>) {
        if (templateActivities.isNotEmpty()) {
            CoroutineScope(Dispatchers.IO).launch {
                // clear out old templates
                provider.get().deleteAllBuiltInTemplateActivities()
                // import new list
                templateActivities.forEach { templateActivity ->
                    provider.get().insertTemplateActivity(templateActivity)
                }
            }
        }
    }

}
