package com.exner.tools.jkbikemechanicaldisasterprevention.database

import androidx.room.Dao
import androidx.room.Delete
import androidx.room.Insert
import androidx.room.Query
import androidx.room.Update
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.BackgroundWorkLog
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.IntervalsSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.StravaSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ActivityWithBikeAndRideData
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.RetiredComponents
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.RideWithBikeAndActivityTotals
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ShelvedComponents
import kotlinx.coroutines.flow.Flow

@Dao
interface KJsDAO {
    //
    // OBSERVERS - return tables
    //
    @Query("SELECT * FROM bike WHERE is_active IS 1 ORDER BY last_used_date DESC")
    fun observeBikesOrderedByLastUsedDate(): Flow<List<Bike>>

    @Query("SELECT COUNT('uuid') FROM bike WHERE is_active IS 1")
    fun observeNumberOfBikes(): Flow<Int>

    @Query("SELECT * FROM bike ORDER BY last_used_date DESC")
    fun observeAllBikesOrderedByLastUsedDate(): Flow<List<Bike>>

    @Query("SELECT COUNT('uuid') FROM bike")
    fun observeNumberOfAllBikes(): Flow<Int>

    @Query("SELECT COUNT('uuid') FROM bike WHERE strava_gear_id NOT NULL")
    fun observeNumberOfStravaConnectedBikes(): Flow<Int>

    @Query("SELECT * FROM stravasummarygear ORDER BY distance DESC")
    fun observeStravaSummaryGear(): Flow<List<StravaSummaryGear>>

    @Query("SELECT COUNT('uuid') FROM bike WHERE intervals_gear_id NOT NULL")
    fun observeNumberOfIntervalsConnectedBikes(): Flow<Int>

    @Query("SELECT * FROM intervalssummarygear ORDER BY distance DESC")
    fun observeIntervalsSummaryGear(): Flow<List<IntervalsSummaryGear>>

    @Query("SELECT * FROM activity ORDER BY title")
    fun observeActivitiesOrderedByTitle(): Flow<List<Activity>>

    @Query("SELECT * FROM activitiesbutnotforcompletedrides")
    fun observeActivitiesButNotForCompletedRides(): Flow<List<Activity>>

    @Query("SELECT * FROM activitywithbikeandridedata ORDER BY activity_due_date DESC")
    fun observeActivitiesWithBikeDataOrderedByDueDate(): Flow<List<ActivityWithBikeAndRideData>>

    @Query("SELECT * FROM activitywithbikeandridedata WHERE activity_due_date NOT NULL ORDER BY activity_due_date DESC")
    fun observeActivitiesWithBikeDataAndDueDateOrderedByDueDate(): Flow<List<ActivityWithBikeAndRideData>>

    @Query("SELECT * FROM activitywithbikeandridedata WHERE ride_is_completed = 0 ORDER BY activity_due_date DESC")
    fun observeActivitiesWithBikeAndRideDataForOpenRidesOrderedByDueDate(): Flow<List<ActivityWithBikeAndRideData>>

    @Query("SELECT COUNT('uuid') FROM activity")
    fun observeNumberOfActivities(): Flow<Int>

    @Query("SELECT * FROM templateactivity ORDER BY ride_level, title")
    fun observeTemplateActivities(): Flow<List<TemplateActivity>>

    @Query("SELECT * FROM templateactivity WHERE type_of_template = 'BUILT-IN' ORDER BY ride_level, title")
    fun observeCustomTemplateActivities(): Flow<List<TemplateActivity>>

    @Query("SELECT COUNT('uid') FROM templateactivity WHERE type_of_template = 'BUILT-IN'")
    fun observeNumberOfCustomTemplateActivities(): Flow<Int>

    @Query("SELECT * FROM component ORDER BY name")
    fun observeComponentsOrderedByName(): Flow<List<Component>>

    @Query("SELECT COUNT('uuid') FROM component WHERE retirement_date IS NULL")
    fun observeNumberOfNonRetiredComponents(): Flow<Int>

    @Query("SELECT * FROM component WHERE retirement_date IS NULL ORDER BY name")
    fun observeNonRetiredComponentsOrderedByName(): Flow<List<Component>>

    @Query("SELECT * FROM retiredcomponents ORDER BY retirement_date ASC")
    fun observeRetiredComponents(): Flow<List<RetiredComponents>>

    @Query("SELECT COUNT('uuid') FROM retiredcomponents")
    fun observeNumberOfRetiredComponents(): Flow<Int>

    @Query("SELECT * FROM shelvedcomponents ORDER BY name")
    fun observeShelvedComponents(): Flow<List<ShelvedComponents>>

    @Query("SELECT * FROM backgroundworklog ORDER BY created_instant DESC")
    fun observeBackgroundWorkLog(): Flow<List<BackgroundWorkLog>>

    @Query("SELECT * FROM ride ORDER BY uid")
    fun observeRides(): Flow<List<Ride>>

    @Query("SELECT * FROM ride WHERE is_completed = 0 ORDER BY uid")
    fun observeRidesInPreparation(): Flow<List<Ride>>

    @Query("SELECT * FROM ridewithbikeandactivitytotals ORDER BY datetime(ride_created_instant) DESC")
    fun observeRidesWithBikeAndActivityTotals(): Flow<List<RideWithBikeAndActivityTotals>>

    //
    // GETTERS = return all lines
    //
    @Query("SELECT * FROM bike ORDER BY last_used_date DESC")
    suspend fun getAllBikesOrderedByLastUsedDate(): List<Bike>

    @Query("SELECT * FROM stravasummarygear ORDER BY distance DESC")
    suspend fun getAllStravaSummaryGearByDistance(): List<StravaSummaryGear>

    @Query("SELECT * FROM intervalssummarygear ORDER BY distance DESC")
    suspend fun getAllIntervalsSummaryGearByDistance(): List<IntervalsSummaryGear>

    @Query("SELECT * FROM activity ORDER BY title")
    suspend fun getAllActivitiesOrderedByTitle(): List<Activity>

    @Query("SELECT * FROM activitiesbutnotforcompletedrides")
    suspend fun getAllActivitiesButNotForClosedRides(): List<Activity>

    @Query("SELECT * FROM templateactivity ORDER BY ride_level, title")
    suspend fun getAllTemplateActivities(): List<TemplateActivity>

    @Query("SELECT * FROM component ORDER BY name")
    suspend fun getAllComponents(): List<Component>

    @Query("SELECT * FROM ride")
    suspend fun getAllRides(): List<Ride>

    @Query("SELECT * FROM activity WHERE ride_uid=:rideUid")
    suspend fun getActivitiesForRide(rideUid: Long): List<Activity>

    //
    // GETTERS - return individual lines
    //
    @Query("SELECT * FROM activity WHERE uid=:uid")
    suspend fun getActivityByUid(uid: Long): Activity?

    @Query("SELECT * FROM bike WHERE uid=:uid")
    suspend fun getBikeByUid(uid: Long): Bike?

    @Query("SELECT * FROM stravasummarygear WHERE id=:id")
    suspend fun getStravaSummaryGearById(id: String): StravaSummaryGear?

    @Query("SELECT * FROM intervalssummarygear WHERE id=:id")
    suspend fun getIntervalsSummaryGearById(id: String): IntervalsSummaryGear?

    @Query("SELECT * FROM templateactivity WHERE uid=:uid")
    suspend fun getTemplateActivityByUid(uid: Long): TemplateActivity?

    @Query("SELECT * FROM component WHERE uid=:uid")
    suspend fun getComponentByUid(uid: Long): Component?

    @Query("SELECT * FROM bike WHERE strava_gear_id=:gearId")
    suspend fun getBikeForStravaGearId(gearId: String): Bike?

    @Query("SELECT * FROM bike WHERE intervals_gear_id=:gearId")
    suspend fun getBikeForIntervalsGearId(gearId: String): Bike?

    @Query("SELECT * FROM ride WHERE uid=:rideUid")
    suspend fun getRideByUid(rideUid: Long): Ride?

    //
    // other helpers
    //
    @Query("SELECT COUNT(uid) FROM activity WHERE bike_uid=:bikeUid")
    suspend fun getActivityCountByBike(bikeUid: Long): Int

    @Query("SELECT * FROM templateactivity WHERE ride_level=:rideLevel OR ride_level IS NULL")
    suspend fun getTemplateActivityForRideLevelOrNoRideLevel(rideLevel: Int): List<TemplateActivity>

    @Query("SELECT * FROM component WHERE bike_uid=:bikeUid")
    suspend fun getAllComponentsForBike(bikeUid: Long): List<Component>

    @Query("SELECT COUNT('uuid') FROM activitywithbikeandridedata WHERE activity_ride_level=:rideLevel")
    suspend fun getNumberOfActivitiesForRideLevel(rideLevel: Int): Int

    @Query("SELECT COUNT('uid') FROM bike WHERE strava_gear_id IS NOT NULL OR intervals_gear_id IS NOT NULL")
    suspend fun getNumberOfBikesWithThirdPartyData(): Int

    @Query("SELECT COUNT('uid') FROM stravasummarygear")
    suspend fun getNumberOfImportedGearStrava(): Int

    @Query("SELECT COUNT('uid') FROM intervalssummarygear")
    suspend fun getNumberOfImportedGearIntervals(): Int

    @Query("SELECT COUNT('uid') FROM component WHERE bike_uid=:bikeUid")
    suspend fun getNumberOfComponentsForBike(bikeUid: Long): Int

    //
    // UPDATE/INSERT/DELETE
    //
    @Insert
    suspend fun insertBike(bike: Bike): Long

    @Update
    suspend fun updateBike(bike: Bike)

    @Delete
    suspend fun deleteBike(bike: Bike)

    @Query("DELETE FROM bike")
    suspend fun deleteAllBikes()

    @Insert
    suspend fun insertStravaSummaryGear(gear: StravaSummaryGear)

    @Update
    suspend fun updateStravaSummaryGear(gear: StravaSummaryGear)

    @Query("DELETE FROM stravasummarygear")
    suspend fun deleteAllStravaSummaryGear()

    @Insert
    suspend fun insertIntervalsSummaryGear(gear: IntervalsSummaryGear)

    @Update
    suspend fun updateIntervalsSummaryGear(gear: IntervalsSummaryGear)

    @Query("DELETE FROM intervalssummarygear")
    suspend fun deleteAllIntervalsSummaryGear()

    //

    @Insert
    suspend fun insertActivity(activity: Activity): Long

    @Update
    suspend fun updateActivity(activity: Activity)

    @Query("DELETE FROM Activity WHERE uid=:activityUid")
    suspend fun deleteActivityByUid(activityUid: Long)

    @Query("DELETE FROM Activity WHERE ride_uid=:rideUid")
    suspend fun deleteActivitiesForRide(rideUid: Long)

    @Query("DELETE FROM activity")
    suspend fun deleteAllActivities()

    //

    @Update
    suspend fun updateRide(ride: Ride)

    @Query("DELETE FROM ride")
    suspend fun deleteAllRides()

    //

    @Insert
    suspend fun insertTemplateActivity(templateActivity: TemplateActivity): Long

    @Update
    suspend fun updateTemplateActivity(templateActivity: TemplateActivity)

    @Query("DELETE FROM templateactivity WHERE uid=:uid")
    suspend fun deleteTemplateActivityByUid(uid: Long)

    @Query("DELETE FROM templateactivity WHERE type_of_template='BUILT_IN'")
    suspend fun deleteAllBuiltInTemplateActivities()

    //

    @Insert
    suspend fun insertRide(ride: Ride): Long

    //

    @Insert
    suspend fun insertComponent(component: Component): Long

    @Query("DELETE FROM component")
    suspend fun deleteAllComponents()

    @Update
    suspend fun updateComponent(component: Component)

    @Delete
    suspend fun deleteComponent(component: Component)

    //

    @Insert
    suspend fun insertAAGLogEntry(logEntry: BackgroundWorkLog)

}