package com.exner.tools.jkbikemechanicaldisasterprevention.database

import androidx.room.TypeConverter
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.RetirementReason
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.WearLevel
import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

@OptIn(ExperimentalTime::class)
object DataConverter {
    @TypeConverter
    fun fromTimestamp(value: String?): LocalDate? {
        return value?.let { LocalDate.parse(it) }
    }

    @TypeConverter
    fun dateToTimestamp(date: LocalDate?): String? {
        return date?.toString()
    }

    @OptIn(ExperimentalTime::class)
    @TypeConverter
    fun instantFromISO8601String(iso8601String: String?): Instant? {
        if (iso8601String?.length == 19) {
            // SQLite does this: "2025-09-01 19:23:00"
            // so we put a 'T' in the middle, then covert via LocalDateTime
            val localDateTime = LocalDateTime.parse(iso8601String.replace(' ', 'T'))
            return localDateTime.toInstant(TimeZone.currentSystemDefault())
        }
        return iso8601String?.let { Instant.parse(it) }
    }

    @OptIn(ExperimentalTime::class)
    @TypeConverter
    fun instantToISO8601String(instant: Instant?): String? {
        return instant?.toString()
    }

    @TypeConverter
    fun fromWearLevelString(value: String?): WearLevel {
        when (value) {
            "NEW" -> return WearLevel.NEW
            "USED" -> return WearLevel.USED
            "DUE_FOR_REPLACEMENT" -> return WearLevel.DUE_FOR_REPLACEMENT
        }
        // default is NEW
        return WearLevel.NEW
    }

    @TypeConverter
    fun wearLevelToString(wearLevel: WearLevel?): String? {
        return wearLevel?.toString() // good for the database, do not localise!
    }

    @TypeConverter
    fun fromRetirementReasonString(value: String?): RetirementReason? {
        when (value) {
            "WORN" -> return RetirementReason.WORN
            "IRREPARABLE" -> return RetirementReason.IRREPARABLE
            "UPGRADE" -> return RetirementReason.UPGRADE
            "NO_LONGER_NEEDED" -> return RetirementReason.NO_LONGER_NEEDED
        }
        return null
    }

    @TypeConverter
    fun retirementReasonToString(reason: RetirementReason?): String? {
        return reason?.toString() // good for the database, do not localise!
    }
}
