package com.exner.tools.jkbikemechanicaldisasterprevention

import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.Context
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.viewModels
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.material3.windowsizeclass.ExperimentalMaterial3WindowSizeClassApi
import androidx.compose.material3.windowsizeclass.calculateWindowSizeClass
import androidx.compose.runtime.getValue
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.WorkManager
import com.example.compose.KJsBikeMaintenanceCheckerTheme
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.scheduler.CheckComponentsIntervalsAndCreateActivitiesWorker
import com.exner.tools.jkbikemechanicaldisasterprevention.scheduler.UpdateBikeDistancesFromIntegrationWorker
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffold
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.MainViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.Theme
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithIntervalsViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithStravaViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.notifications.Notifications.Companion.CHANNEL_DESCRIPTION
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.notifications.Notifications.Companion.CHANNEL_ID
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.notifications.Notifications.Companion.CHANNEL_NAME
import dagger.hilt.android.AndroidEntryPoint
import io.github.vinceglb.filekit.FileKit
import io.github.vinceglb.filekit.manualFileKitCoreInitialization
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import javax.inject.Inject

private const val TAG = "MainActivity"

@AndroidEntryPoint
class MainActivity : ComponentActivity() {

    @Inject
    lateinit var userPreferencesManager: UserPreferencesManager

    private val viewModel: MainViewModel by viewModels()
    private val syncWithStravaViewModel: SyncWithStravaViewModel by viewModels()
    private val syncWithIntervalsViewModel: SyncWithIntervalsViewModel by viewModels()

    @OptIn(ExperimentalMaterial3WindowSizeClassApi::class)
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        createNotificationChannel()

        val context: Context = this

        // Initialize FileKit
        FileKit.manualFileKitCoreInitialization(context)

        // launch background workers
        val lifecycleOwner = this
        val schedulerWorkRequest =
            CheckComponentsIntervalsAndCreateActivitiesWorker.createWorkRequest()
        val workManager = WorkManager.getInstance(lifecycleOwner)
        workManager.enqueueUniquePeriodicWork(
            uniqueWorkName = "CheckComponentsIntervalsAndCreateActivities",
            request = schedulerWorkRequest,
            existingPeriodicWorkPolicy = ExistingPeriodicWorkPolicy.REPLACE
        )
        workManager.getWorkInfosForUniqueWorkLiveData("CheckComponentsIntervalsAndCreateActivities")
            .observe(lifecycleOwner) { workInfo ->
                Log.d(TAG, "WorkInfo: $workInfo")
            }

        val distanceUpdaterWorkRequest =
            UpdateBikeDistancesFromIntegrationWorker.createWorkRequest()
        workManager.enqueueUniquePeriodicWork(
            uniqueWorkName = "UpdateBikeDistancesFromIntegration",
            request = distanceUpdaterWorkRequest,
            existingPeriodicWorkPolicy = ExistingPeriodicWorkPolicy.REPLACE
        )
        workManager.getWorkInfosForUniqueWorkLiveData("UpdateBikeDistancesFromIntegration")
            .observe(lifecycleOwner) { workInfo ->
                Log.d(TAG, "WorkInfo: $workInfo")
            }

        // check where we are coming from
        // is there an Intent? Did Strava give us a code?
        val intent = intent
        Log.d(TAG, "Intent: $intent")
        Log.d(TAG, "Intent data: ${intent.dataString}")
        val hasCode = intent.data?.queryParameterNames?.contains("code") == true
        Log.d(TAG, "Intent data contains 'code' $hasCode")
        val hasScope = intent.data?.queryParameterNames?.contains("scope") == true
        Log.d(TAG, "Intent data contains 'scope' $hasScope")

        val successToast = Toast.makeText(context,
            getString(R.string.toast_backup_successful), Toast.LENGTH_SHORT)
        val failToast = Toast.makeText(context,
            getString(R.string.toast_backup_failed), Toast.LENGTH_LONG)

        // curiosity kills the cat
        CoroutineScope(Dispatchers.IO).launch {
            userPreferencesManager.dumpAllPreferences()

            // are we supposed to export for backup?
            if (userPreferencesManager.automaticallyExportData().firstOrNull() == true) {
                viewModel.createBackup(
                    context = context,
                    successCallback = {
                        successToast.show()
                    },
                    failureCallback = {
                        failToast.show()
                    }
                )
                if (userPreferencesManager.automaticallyPruneExportedData().firstOrNull() == true) {
                    viewModel.pruneBackups(
                        context = context
                    )
                }
            }
        }

        enableEdgeToEdge()
        setContent {
            // night mode has two possible triggers:
            // - device may be in night mode
            // - force night mode setting may be on
            val userTheme by viewModel.userSelectedTheme.collectAsStateWithLifecycle()

            // window size class
            val windowSizeClass = calculateWindowSizeClass(this)

            val stravaEnabled by viewModel.stravaEnabled.collectAsStateWithLifecycle()
            val intervalsEnabled by viewModel.intervalsEnabled.collectAsStateWithLifecycle()

            if (hasCode) {
                val code = intent.data?.getQueryParameter("code") ?: ""
                Log.d(TAG, "Intent data: Strava/Intervals.icu code $code")
                if (stravaEnabled) {
                    syncWithStravaViewModel.useCode(this, code)
                } else if (intervalsEnabled) {
                    syncWithIntervalsViewModel.useCode(this, code)
                }
            }
            if (hasScope || (hasCode && intervalsEnabled)) {
                val scope = intent.data?.getQueryParameter("scope") ?: ""
                Log.d(TAG, "Intent data: Strava/Intervals.icu allowed scope $scope")
                if (stravaEnabled) {
                    syncWithStravaViewModel.applyAllowedScopes(scope)
                } else if (intervalsEnabled) {
                    syncWithIntervalsViewModel.applyAllowedScopes(scope)
                }
            }

            KJsBikeMaintenanceCheckerTheme(
                darkTheme = userTheme == Theme.Dark || (userTheme == Theme.Auto && isSystemInDarkTheme())
            ) {
                KJsGlobalScaffold(activity = this, windowSizeClass = windowSizeClass)
            }
        }
    }

    private fun createNotificationChannel() {
        val name = CHANNEL_NAME // see ui/notifications/Notifications.kt
        val descriptionText = CHANNEL_DESCRIPTION
        val importance = NotificationManager.IMPORTANCE_DEFAULT
        val channel = NotificationChannel(CHANNEL_ID, name, importance).apply {
            description = descriptionText
        }
        // Register the channel with the system.
        val notificationManager: NotificationManager =
            getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        notificationManager.createNotificationChannel(channel)
    }

}
