package com.example.controldecolonias.ui.components

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.Card
import androidx.compose.material3.Checkbox
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.example.controldecolonias.Alimentacion
import com.example.controldecolonias.Colonia
import com.example.controldecolonias.Nido
import com.example.controldecolonias.Nota
import com.example.controldecolonias.Tarea
import com.example.controldecolonias.util.formatearTimestamp


// --- COMPONENTES Y FUNCIONES AUXILIARES ---
    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    fun ColoniaItem(colonia: Colonia, onClick: () -> Unit) {
        Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp), onClick = onClick) {
            Column(modifier = Modifier.padding(16.dp)) {
                Text(colonia.nombre, fontSize = 20.sp, fontWeight = FontWeight.Bold)
                Text(colonia.especie, fontSize = 16.sp)
            }
        }
    }

    @Composable
    fun NidoItem(nido: Nido, onClick: () -> Unit) {
        Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp), onClick = onClick) {
            Column(modifier = Modifier.padding(16.dp)) {
                Text(nido.nombre, fontSize = 20.sp, fontWeight = FontWeight.Bold)
                Text(nido.especie, fontSize = 16.sp)
            }
        }
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    fun NotaItem(nota: Nota, onClick: () -> Unit) {
        Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp), onClick = onClick) {
            Column(modifier = Modifier.padding(8.dp)) {
                Text(text = formatearTimestamp(nota.fecha, "dd/MM/yy HH:mm"), style = MaterialTheme.typography.labelSmall)
                Text(text = nota.texto, style = MaterialTheme.typography.bodyMedium)
            }
        }
    }


@Composable
    fun DetalleItem(etiqueta: String, valor: String) {
        Column(modifier = Modifier.padding(vertical = 4.dp)) {
            Text(text = etiqueta, style = MaterialTheme.typography.labelSmall, color = MaterialTheme.colorScheme.primary)
            Text(text = valor, style = MaterialTheme.typography.bodyLarge)
        }
    }

    @Composable
    fun CheckboxConEtiqueta(texto: String, checked: Boolean, onCheckedChange: (Boolean) -> Unit) {
        Column(horizontalAlignment = Alignment.CenterHorizontally) {
            Text(texto)
            Checkbox(checked = checked, onCheckedChange = onCheckedChange)
        }
    }

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AlimentacionItem(registro: Alimentacion, onClick: () -> Unit) { // <-- onClick ya estaba, nos aseguramos que se use
    Card(
        modifier = Modifier
            .fillMaxWidth()
            .padding(vertical = 4.dp),
        onClick = onClick // Se lo pasamos a la tarjeta
    ) {
        Column(modifier = Modifier.padding(16.dp)) {
            Text(
                text = registro.tipoComida,
                style = MaterialTheme.typography.titleMedium,
                fontWeight = FontWeight.Bold
            )
            if (registro.nota.isNotBlank()) {
                Text(
                    text = registro.nota,
                    style = MaterialTheme.typography.bodyMedium,
                    modifier = Modifier.padding(top = 4.dp)
                )
            }
            Text(
                text = formatearTimestamp(registro.fecha, "dd/MM/yy HH:mm"),
                style = MaterialTheme.typography.labelSmall,
                modifier = Modifier.align(Alignment.End).padding(top = 8.dp)
            )
        }
    }
}
@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun TareaItem(
    tarea: Tarea,
    onCheckedChange: (Boolean) -> Unit,
    onClick: () -> Unit
) {
    Card(
        modifier = Modifier
            .fillMaxWidth()
            .padding(vertical = 4.dp),
        onClick = onClick
    ) {
        Row(
            modifier = Modifier.padding(8.dp),
            verticalAlignment = Alignment.CenterVertically
        ) {
            Checkbox(
                checked = tarea.completada,
                onCheckedChange = onCheckedChange
            )
            Spacer(modifier = Modifier.width(8.dp))
            Column(modifier = Modifier.weight(1f)) {
                Text(
                    text = tarea.texto,
                    style = MaterialTheme.typography.bodyLarge,
                )
                tarea.fechaLimite?.let {
                    Text(
                        text = "Límite: ${formatearTimestamp(it, "dd/MM/yy HH:mm")}",
                        style = MaterialTheme.typography.labelSmall
                    )
                }
            }
        }
    }
}
