package com.example.controldecolonias.util

import android.Manifest
import android.content.ContentValues
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Typeface
import android.graphics.pdf.PdfDocument
import android.location.LocationManager // <-- Import nativo de Android
import android.net.Uri
import android.os.Build
import android.provider.MediaStore
import android.widget.Toast
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asAndroidBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.core.content.ContextCompat
import com.example.controldecolonias.Colonia
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.Nido
// --- Importaciones de Google (GMS) eliminadas ---
import com.google.zxing.BarcodeFormat
import com.google.zxing.qrcode.QRCodeWriter
import java.io.FileOutputStream
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.*

// --- FUNCIÓN DE UBICACIÓN MODIFICADA (SIN GOOGLE) ---
fun obtenerUbicacion(context: Context, onUbicacionObtenida: (android.location.Location?) -> Unit) {
    val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager

    if (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
        try {
            // Usamos el proveedor de GPS nativo del sistema
            // getLastKnownLocation es rápido y no gasta batería extra.
            val location = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER)
            if (location != null) {
                onUbicacionObtenida(location)
            } else {
                // Si el GPS falla, intentamos con la red (también nativo)
                val networkLocation = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER)
                onUbicacionObtenida(networkLocation)
            }
        } catch (e: SecurityException) {
            e.printStackTrace()
            onUbicacionObtenida(null)
        }
    } else {
        onUbicacionObtenida(null) // Permiso no concedido
    }
}

fun formatearTimestamp(timestamp: Long, pattern: String): String {
    val sdf = SimpleDateFormat(pattern, Locale.getDefault())
    sdf.timeZone = TimeZone.getDefault()
    return sdf.format(Date(timestamp))
}

fun construirColoniaDesdeFormulario(id: String, nombre: String, especie: String, fechaTexto: String, horaTexto: String, latitud: Double?, longitud: Double?, temperatura: String, clima: String, tieneHuevos: Boolean, tieneLarvas: Boolean, tienePupas: Boolean, tieneObreras: Boolean, estaEnferma: Boolean, fechaOriginal: Long?, tipoEtiqueta: String): Colonia? {
    val timestamp = if (fechaTexto.isNotBlank() && horaTexto.isNotBlank()) {
        parsearFechaHora(fechaTexto, horaTexto)
    } else {
        fechaOriginal ?: System.currentTimeMillis()
    }
    if (timestamp == null) return null
    return Colonia(
        id = id.trim(),
        nombre = nombre.trim(),
        especie = especie.trim(),
        fecha = timestamp,
        latitud = latitud, longitud = longitud, temperatura = temperatura.trim().toIntOrNull(),
        clima = clima.trim().takeIf { it.isNotBlank() },
        tieneHuevos = tieneHuevos, tieneLarvas = tieneLarvas, tienePupas = tienePupas, tieneObreras = tieneObreras, estaEnferma = estaEnferma,
        tipoEtiqueta = tipoEtiqueta
    )
}

fun construirNidoDesdeFormulario(
    id: String, nombre: String, especie: String, notas: String,
    latitud: Double?, longitud: Double?, fechaOriginal: Long?
): Nido? {
    if (latitud == null || longitud == null || nombre.isBlank()) return null
    return Nido(
        id = id.trim(),
        nombre = nombre.trim(),
        especie = especie.trim(),
        fecha = fechaOriginal ?: System.currentTimeMillis(),
        latitud = latitud,
        longitud = longitud,
        notas = notas.trim()
    )
}

fun parsearFechaHora(fecha: String, hora: String): Long? {
    return try {
        val formato = SimpleDateFormat("dd/MM/yyyy HH:mm", Locale.getDefault())
        formato.timeZone = TimeZone.getDefault()
        formato.parse("$fecha $hora")?.time
    } catch (e: ParseException) { null }
}

fun generarIdCorto(): String {
    val caracteresPermitidos = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
    return (1..8).map { caracteresPermitidos.random() }.joinToString("")
}

suspend fun generarIdCortoUnico(viewModel: ColoniaViewModel): String {
    var nuevoId: String
    do {
        nuevoId = generarIdCorto()
    } while (viewModel.idExiste(nuevoId))
    return nuevoId
}

fun generarQrBitmap(texto: String): ImageBitmap? {
    return try {
        val writer = QRCodeWriter()
        val bitMatrix = writer.encode(texto, BarcodeFormat.QR_CODE, 512, 512)
        val width = bitMatrix.width
        val height = bitMatrix.height
        val bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.RGB_565)
        for (x in 0 until width) {
            for (y in 0 until height) {
                bitmap.setPixel(x, y, if (bitMatrix[x, y]) Color.BLACK else Color.WHITE)
            }
        }
        bitmap.asImageBitmap()
    } catch (e: Exception) {
        e.printStackTrace()
        null
    }
}

fun guardarBitmapEnGaleria(context: Context, bitmap: ImageBitmap, nombreArchivo: String) {
    val contentResolver = context.contentResolver
    val coleccion = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
        MediaStore.Images.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY)
    } else {
        MediaStore.Images.Media.EXTERNAL_CONTENT_URI
    }

    val contentValues = ContentValues().apply {
        put(MediaStore.Images.Media.DISPLAY_NAME, "$nombreArchivo.jpg")
        put(MediaStore.Images.Media.MIME_TYPE, "image/jpeg")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            put(MediaStore.Images.Media.IS_PENDING, 1)
        }
    }

    val uri = contentResolver.insert(coleccion, contentValues)

    uri?.let {
        try {
            val outputStream = contentResolver.openOutputStream(it)
            outputStream?.use { stream ->
                bitmap.asAndroidBitmap().compress(Bitmap.CompressFormat.JPEG, 100, stream)
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                contentValues.clear()
                contentValues.put(MediaStore.Images.Media.IS_PENDING, 0)
                contentResolver.update(it, contentValues, null, null)
            }
            Toast.makeText(context, "QR guardado en la galería", Toast.LENGTH_SHORT).show()
        } catch (e: Exception) {
            e.printStackTrace()
            Toast.makeText(context, "Error al guardar la imagen", Toast.LENGTH_SHORT).show()
        }
    }
}


fun generarPdfDeQrs(context: Context, cantidad: Int, uri: Uri) {
    val qrPorPagina = 20 // 4 columnas x 5 filas
    val document = PdfDocument()

    val qrData = (1..cantidad).map {
        val id = generarIdCorto()
        val writer = QRCodeWriter()
        val bitMatrix = writer.encode(id, BarcodeFormat.QR_CODE, 150, 150)
        val bitmap = Bitmap.createBitmap(150, 150, Bitmap.Config.RGB_565)
        for (x in 0 until 150) {
            for (y in 0 until 150) {
                bitmap.setPixel(x, y, if (bitMatrix[x, y]) Color.BLACK else Color.WHITE)
            }
        }
        id to bitmap
    }

    val totalPaginas = (cantidad + qrPorPagina - 1) / qrPorPagina

    for (paginaIndex in 0 until totalPaginas) {
        val pageInfo = PdfDocument.PageInfo.Builder(595, 842, paginaIndex + 1).create()
        val page = document.startPage(pageInfo)
        val canvas = page.canvas

        val paint = Paint().apply {
            color = Color.BLACK
            textSize = 10f
            textAlign = Paint.Align.CENTER
            typeface = Typeface.create(Typeface.MONOSPACE, Typeface.NORMAL)
        }

        val qrSize = 100f
        val textHeight = 20f
        val itemHeight = qrSize + textHeight

        val numColumnas = 4
        val numFilas = 5

        val hMargin = (canvas.width - numColumnas * qrSize) / (numColumnas + 1)
        val vMargin = (canvas.height - numFilas * itemHeight) / (numFilas + 1)

        val startIndex = paginaIndex * qrPorPagina
        val endIndex = minOf((paginaIndex + 1) * qrPorPagina, cantidad)

        for (i in startIndex until endIndex) {
            val itemIndex = i % qrPorPagina
            val fila = itemIndex / numColumnas
            val col = itemIndex % numColumnas
            val (id, bitmap) = qrData[i]
            val x = hMargin + col * (qrSize + hMargin)
            val y = vMargin + fila * (itemHeight + vMargin)
            val scaledBitmap = Bitmap.createScaledBitmap(bitmap, qrSize.toInt(), qrSize.toInt(), false)

            canvas.drawBitmap(scaledBitmap, x, y, null)
            canvas.drawText(id, x + qrSize / 2, y + qrSize + 12, paint)
        }
        document.finishPage(page)
    }

    try {
        context.contentResolver.openFileDescriptor(uri, "w")?.use {
            FileOutputStream(it.fileDescriptor).use { outputStream ->
                document.writeTo(outputStream)
            }
        }
        document.close()
        Toast.makeText(context, "PDF guardado exitosamente", Toast.LENGTH_LONG).show()
    } catch (e: Exception) {
        e.printStackTrace()
        Toast.makeText(context, "Error al guardar el PDF", Toast.LENGTH_SHORT).show()
    }
}