package com.example.controldecolonias.util

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.provider.Settings
import android.widget.Toast
import androidx.core.app.NotificationManagerCompat // Import necesario
import com.example.controldecolonias.AlarmReceiver
import com.example.controldecolonias.Colonia
import com.example.controldecolonias.Tarea

object AlarmScheduler {

    // --- NUEVA FUNCIÓN PARA VERIFICAR AMBOS PERMISOS ---
    private fun checkAndRequestPermissions(context: Context): Boolean {
        val notificationManagerCompat = NotificationManagerCompat.from(context)

        // 1. Verificar si las notificaciones generales están habilitadas
        if (!notificationManagerCompat.areNotificationsEnabled()) {
            Toast.makeText(
                context,
                "Las notificaciones están bloqueadas. Por favor, actívalas en los ajustes de la app.",
                Toast.LENGTH_LONG
            ).show()
            // Abrir los ajustes de notificación de la app
            val intent = Intent().apply {
                action = Settings.ACTION_APP_NOTIFICATION_SETTINGS
                putExtra(Settings.EXTRA_APP_PACKAGE, context.packageName)
            }
            context.startActivity(intent)
            return false // Detener porque las notificaciones están bloqueadas
        }

        // 2. Verificar si se pueden programar alarmas exactas (Android 12+)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
            if (!alarmManager.canScheduleExactAlarms()) {
                Toast.makeText(
                    context,
                    "Se necesita permiso para alarmas exactas. Por favor, actívalo.",
                    Toast.LENGTH_LONG
                ).show()
                // Abrir los ajustes específicos para alarmas exactas
                Intent(Settings.ACTION_REQUEST_SCHEDULE_EXACT_ALARM).also {
                    context.startActivity(it)
                }
                return false // Detener porque falta el permiso de alarmas exactas
            }
        }

        return true // Ambos permisos están OK (o no son necesarios en versiones antiguas)
    }

    // --- Sobrecarga para Colonias ---
    fun schedule(context: Context, colonia: Colonia) {
        // --- VERIFICACIÓN DE PERMISOS ANTES DE CONTINUAR ---
        if (!checkAndRequestPermissions(context)) {
            return // Si falta algún permiso, no programamos nada
        }

        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
        val intent = Intent(context, AlarmReceiver::class.java).apply {
            putExtra("EXTRA_COLONIA_ID", colonia.id)
            putExtra("EXTRA_COLONIA_NOMBRE", colonia.nombre)
        }
        val requestCode = "colonia_${colonia.id}".hashCode()
        val pendingIntent = PendingIntent.getBroadcast(
            context,
            requestCode,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        colonia.recordatorioProximaFecha?.let {
            if (it > System.currentTimeMillis()) {
                try { // Añadimos try-catch por si acaso
                    alarmManager.setExactAndAllowWhileIdle(
                        AlarmManager.RTC_WAKEUP,
                        it,
                        pendingIntent
                    )
                } catch (se: SecurityException) {
                    // Aunque ya verificamos, capturamos por si acaso
                    Toast.makeText(context, "Error de seguridad al programar alarma.", Toast.LENGTH_SHORT).show()
                    se.printStackTrace()
                }
            }
        }
    }

    fun cancel(context: Context, colonia: Colonia) {
        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
        val intent = Intent(context, AlarmReceiver::class.java)
        val requestCode = "colonia_${colonia.id}".hashCode()

        val pendingIntent = PendingIntent.getBroadcast(
            context,
            requestCode,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        alarmManager.cancel(pendingIntent)
    }

    // --- Sobrecarga para Tareas ---
    fun schedule(context: Context, tarea: Tarea) {
        // --- VERIFICACIÓN DE PERMISOS ANTES DE CONTINUAR ---
        if (!checkAndRequestPermissions(context)) {
            return // Si falta algún permiso, no programamos nada
        }

        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
        val intent = Intent(context, AlarmReceiver::class.java).apply {
            putExtra("EXTRA_TAREA_ID", tarea.id)
            putExtra("EXTRA_TAREA_TEXTO", tarea.texto)
        }
        val requestCode = "tarea_${tarea.id}".hashCode()
        val pendingIntent = PendingIntent.getBroadcast(
            context,
            requestCode,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        tarea.recordatorioProximaFecha?.let {
            if (it > System.currentTimeMillis()) {
                try { // Añadimos try-catch por si acaso
                    alarmManager.setExactAndAllowWhileIdle(
                        AlarmManager.RTC_WAKEUP,
                        it,
                        pendingIntent
                    )
                } catch (se: SecurityException) {
                    Toast.makeText(context, "Error de seguridad al programar alarma.", Toast.LENGTH_SHORT).show()
                    se.printStackTrace()
                }
            }
        }
    }

    fun cancel(context: Context, tarea: Tarea) {
        val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
        val intent = Intent(context, AlarmReceiver::class.java)
        val requestCode = "tarea_${tarea.id}".hashCode()

        val pendingIntent = PendingIntent.getBroadcast(
            context,
            requestCode,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        alarmManager.cancel(pendingIntent)
    }
}