package com.example.controldecolonias.ui.screens.dashboard

import android.net.Uri
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.compose.foundation.layout.*
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.*
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import com.example.controldecolonias.Colonia
import com.example.controldecolonias.ColoniaViewModel
import com.example.controldecolonias.ResultadoScan
import com.example.controldecolonias.ScanMode
import com.example.controldecolonias.ui.navigation.Pantalla
import com.example.controldecolonias.ui.navigation.Rutas
import com.google.gson.Gson
import com.journeyapps.barcodescanner.ScanContract
import com.journeyapps.barcodescanner.ScanOptions

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaPrincipalDashboard(navController: NavController, viewModel: ColoniaViewModel) {
    val totalColonias by viewModel.totalColonias.collectAsState()
    val totalNidos by viewModel.totalNidos.collectAsState()
    val totalEnfermas by viewModel.totalEnfermas.collectAsState()
    val ninjaModeActivado by viewModel.ninjaModeActivado.collectAsState()

    val scanResult by viewModel.resultadoScan.collectAsState()
    val scanMode by viewModel.scanMode.collectAsState()
    val context = LocalContext.current
    var mostrarDialogoOpcionesAnadir by remember { mutableStateOf(false) }

    // Diálogo para error de ID simple duplicado (detiene el flujo)
    var mostrarDialogoIdSimpleConflicto by remember { mutableStateOf<Pair<String, Boolean>?>(null) }
    // Diálogo para ADVERTENCIA de ID duplicado en etiqueta con datos (permite continuar)
    var mostrarDialogoAdvertenciaIdCompleto by remember { mutableStateOf<Colonia?>(null) }


    val qrScannerLauncherBuscar = rememberLauncherForActivityResult(ScanContract()) { result ->
        result.contents?.let { idEscaneado -> viewModel.procesarQrLeido(idEscaneado) }
    }

    val qrScannerLauncherAnadir = rememberLauncherForActivityResult(ScanContract()) { result ->
        result.contents?.let { idEscaneado -> viewModel.procesarScanParaAnadir(idEscaneado, esNfc = false) }
    }

    LaunchedEffect(scanResult) {
        when (val result = scanResult) {
            is ResultadoScan.Exitoso -> {
                navController.navigate("detalleColonia/${result.idColonia}")
                viewModel.resetearScan()
            }
            is ResultadoScan.Error -> {
                Toast.makeText(context, result.mensaje, Toast.LENGTH_LONG).show()
                viewModel.resetearScan()
            }
            is ResultadoScan.IdValidoParaAnadir -> {
                navController.navigate("anadirEditarColonia?scannedId=${result.id}")
                viewModel.resetearScan()
            }
            // --- ESTA ES LA LÓGICA CORREGIDA ---
            is ResultadoScan.ColoniaValidaParaAnadir -> {
                if (result.idYaExiste) {
                    // Si el ID ya existe, mostramos la ADVERTENCIA en lugar de detener.
                    mostrarDialogoAdvertenciaIdCompleto = result.colonia
                } else {
                    // Si el ID es nuevo, navegamos directamente.
                    val json = Uri.encode(Gson().toJson(result.colonia))
                    navController.navigate("anadirEditarColonia?coloniaJson=$json&idYaExiste=false")
                }
                viewModel.resetearScan()
            }
            // Este caso es para cuando se escanea un ID simple (sin datos) que ya existe.
            // Aquí sí detenemos el flujo porque no hay datos que precargar.
            is ResultadoScan.IdYaExisteAlAnadir -> {
                mostrarDialogoIdSimpleConflicto = Pair(result.id, result.esNfc)
                viewModel.resetearScan()
            }
            else -> {}
        }
    }

    Scaffold(
        topBar = { TopAppBar(title = { Text("Dashboard") }) },
        floatingActionButton = {
            ExtendedFloatingActionButton(
                text = { Text("Escanear") },
                icon = { Icon(if (scanMode == ScanMode.NFC) Icons.Default.Nfc else Icons.Default.QrCodeScanner, contentDescription = "Escanear Etiqueta") },
                onClick = {
                    if (scanMode == ScanMode.NFC) {
                        viewModel.activarModoEscaneo()
                        Toast.makeText(context, "Modo NFC activado. Acerca una etiqueta para buscar...", Toast.LENGTH_SHORT).show()
                    } else {
                        val options = ScanOptions().setDesiredBarcodeFormats(ScanOptions.QR_CODE).setPrompt("Escanea un QR para buscar").setBeepEnabled(true)
                        qrScannerLauncherBuscar.launch(options)
                    }
                }
            )
        }
    ) { innerPadding ->
        Column(
            modifier = Modifier
                .padding(innerPadding)
                .padding(16.dp)
                .fillMaxSize(),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            TarjetaDeEstado(
                icono = Icons.Default.AddCircle,
                titulo = "Añadir Colonia",
                valor = "Escanear para registrar",
                colorFondo = MaterialTheme.colorScheme.primaryContainer,
                onClick = { mostrarDialogoOpcionesAnadir = true }
            )
            TarjetaDeEstado(
                icono = Icons.Default.Home,
                titulo = "Mis Colonias",
                valor = totalColonias.toString(),
                onClick = { navController.navigate(Pantalla.Colonias.ruta) }
            )
            if (!ninjaModeActivado) {
                TarjetaDeEstado(
                    icono = Icons.Default.LocationOn,
                    titulo = "Nidos en Seguimiento",
                    valor = totalNidos.toString(),
                    onClick = { navController.navigate(Pantalla.Nidos.ruta) }
                )
            }
            TarjetaDeEstado(
                icono = Icons.Default.Warning,
                titulo = "Colonias con Problemas",
                valor = totalEnfermas.toString(),
                colorFondo = if (totalEnfermas > 0) MaterialTheme.colorScheme.errorContainer else MaterialTheme.colorScheme.secondaryContainer,
                onClick = { /* ... */ }
            )
            TarjetaDeEstado(
                icono = Icons.Default.Settings,
                titulo = "Herramientas y Ajustes",
                valor = "Configuración",
                colorFondo = MaterialTheme.colorScheme.tertiaryContainer,
                onClick = { navController.navigate(Rutas.HERRAMIENTAS) }
            )
        }
    }

    if (mostrarDialogoOpcionesAnadir) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoOpcionesAnadir = false },
            title = { Text("Añadir Colonia") },
            text = { Text("Elige un método para leer los datos de la nueva colonia.") },
            confirmButton = {
                Button(onClick = {
                    mostrarDialogoOpcionesAnadir = false
                    val options = ScanOptions().setDesiredBarcodeFormats(ScanOptions.QR_CODE).setPrompt("Escanea el QR de la nueva colonia").setBeepEnabled(true)
                    qrScannerLauncherAnadir.launch(options)
                }) { Text("Escanear QR") }
            },
            dismissButton = {
                TextButton(onClick = {
                    mostrarDialogoOpcionesAnadir = false
                    viewModel.activarModoEscaneoParaAnadir()
                    Toast.makeText(context, "Acerca la etiqueta NFC de la nueva colonia", Toast.LENGTH_SHORT).show()
                }) { Text("Leer NFC") }
            }
        )
    }

    // --- NUEVO DIÁLOGO DE ADVERTENCIA ---
    mostrarDialogoAdvertenciaIdCompleto?.let { coloniaConIdDuplicado ->
        AlertDialog(
            onDismissRequest = { mostrarDialogoAdvertenciaIdCompleto = null },
            title = { Text("Advertencia: ID Duplicado") },
            text = { Text("El ID '${coloniaConIdDuplicado.id}' ya está en uso. Se cargarán los datos de la etiqueta, pero se asignará un nuevo ID al guardar.") },
            confirmButton = {
                Button(onClick = {
                    // Al confirmar, navegamos a la pantalla de añadir con los datos y el flag de ID duplicado.
                    val json = Uri.encode(Gson().toJson(coloniaConIdDuplicado))
                    navController.navigate("anadirEditarColonia?coloniaJson=$json&idYaExiste=true")
                    mostrarDialogoAdvertenciaIdCompleto = null
                }) {
                    Text("Entendido")
                }
            },
            dismissButton = {
                TextButton(onClick = { mostrarDialogoAdvertenciaIdCompleto = null }) { Text("Cancelar") }
            }
        )
    }


    // Diálogo de error para IDs SIMPLES que ya existen.
    mostrarDialogoIdSimpleConflicto?.let { (id, esNfc) ->
        val tipoEtiqueta = if (esNfc) "la etiqueta NFC" else "el código QR"
        AlertDialog(
            onDismissRequest = { mostrarDialogoIdSimpleConflicto = null },
            title = { Text("ID ya existente") },
            text = { Text("El ID '$id' leído desde $tipoEtiqueta ya está registrado. No puedes usarlo para una nueva colonia. Si quieres editar la existente, búscala en 'Mis Colonias'.") },
            confirmButton = {
                Button(onClick = { mostrarDialogoIdSimpleConflicto = null }) { Text("Entendido") }
            }
        )
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun TarjetaDeEstado(
    icono: ImageVector,
    titulo: String,
    valor: String,
    colorFondo: androidx.compose.ui.graphics.Color = MaterialTheme.colorScheme.secondaryContainer,
    onClick: () -> Unit
) {
    Card(
        modifier = Modifier.fillMaxWidth(),
        colors = CardDefaults.cardColors(containerColor = colorFondo),
        onClick = onClick
    ) {
        Row(
            modifier = Modifier.padding(16.dp),
            verticalAlignment = Alignment.CenterVertically
        ) {
            Icon(icono, contentDescription = null, modifier = Modifier.size(40.dp))
            Spacer(modifier = Modifier.width(16.dp))
            Column {
                Text(text = titulo, style = MaterialTheme.typography.titleMedium)
                Text(text = valor, style = MaterialTheme.typography.headlineMedium)
            }
        }
    }
}