package com.example.controldecolonias.ui.screens.colonias

import android.Manifest
import android.app.DatePickerDialog
import android.app.TimePickerDialog
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.PickVisualMediaRequest
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.grid.GridCells
import androidx.compose.foundation.lazy.grid.LazyVerticalGrid
import androidx.compose.foundation.lazy.grid.items
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.*
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.core.content.ContextCompat
import androidx.core.content.FileProvider
import androidx.navigation.NavController
import coil.compose.AsyncImage
import com.example.controldecolonias.*
import com.example.controldecolonias.ui.components.*
import com.example.controldecolonias.ui.navigation.Rutas
import com.example.controldecolonias.util.*
import com.google.gson.Gson
import com.journeyapps.barcodescanner.ScanContract
import com.journeyapps.barcodescanner.ScanOptions
import kotlinx.coroutines.launch
import java.io.File
import java.text.SimpleDateFormat
import java.util.*

@Composable
fun PantallaPrincipalColonias(navController: NavController, viewModel: ColoniaViewModel) {
    PantallaListaDeColonias(
        viewModel = viewModel,
        onNavigateToAnadir = {
            viewModel.cancelarModoNfc()
            navController.navigate(Rutas.ANADIR_EDITAR_COLONIA)
        },
        onColoniaClick = { id -> navController.navigate("detalleColonia/$id") },
        onNavigateToDetalle = { id -> navController.navigate("detalleColonia/$id") }
    )
}

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun PantallaListaDeColonias(
    viewModel: ColoniaViewModel, onNavigateToAnadir: () -> Unit,
    onColoniaClick: (String) -> Unit, onNavigateToDetalle: (String) -> Unit
) {
    val coloniasAgrupadas by viewModel.coloniasFiltradasYAgrupadas.collectAsState()
    val textoBusqueda by viewModel.textoBusqueda.collectAsState()
    val scanResult by viewModel.resultadoScan.collectAsState()
    var mostrarDialogoDesconocido by remember { mutableStateOf<Colonia?>(null) }
    var mostrarDialogoConflicto by remember { mutableStateOf<Colonia?>(null) }
    val context = LocalContext.current
    val scanMode by viewModel.scanMode.collectAsState()

    val qrScannerLauncher = rememberLauncherForActivityResult(ScanContract()) { result ->
        result.contents?.let { qrData -> viewModel.procesarQrLeido(qrData) }
    }

    LaunchedEffect(scanResult) {
        when (val result = scanResult) {
            is ResultadoScan.Exitoso -> { onNavigateToDetalle(result.idColonia); viewModel.resetearScan() }
            is ResultadoScan.Desconocido -> { mostrarDialogoDesconocido = result.coloniaEnTag; viewModel.resetearScan() }
            is ResultadoScan.ConflictoIdExistente -> { mostrarDialogoConflicto = result.coloniaEnTag; viewModel.resetearScan() }
            is ResultadoScan.Error -> { Toast.makeText(context, result.mensaje, Toast.LENGTH_LONG).show(); viewModel.resetearScan() }
            else -> {}
        }
    }
    val scrollBehavior = TopAppBarDefaults.enterAlwaysScrollBehavior(rememberTopAppBarState())
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            TopAppBar(
                title = { Text("Mis Colonias") },
                scrollBehavior = scrollBehavior,
                actions = {}
            )
        },
        floatingActionButton = {
            Column(horizontalAlignment = Alignment.End, verticalArrangement = Arrangement.spacedBy(8.dp)) {
                FloatingActionButton(onClick = onNavigateToAnadir) {
                    Icon(Icons.Default.Add, contentDescription = "Añadir Colonia Manualmente")
                }
                ExtendedFloatingActionButton(
                    text = { Text("Escanear") },
                    icon = { Icon(if (scanMode == ScanMode.NFC) Icons.Default.Nfc else Icons.Default.QrCodeScanner, contentDescription = "Escanear Etiqueta") },
                    onClick = {
                        if (scanMode == ScanMode.NFC) {
                            viewModel.activarModoEscaneo()
                            Toast.makeText(context, "Modo NFC activado. Acerca una etiqueta...", Toast.LENGTH_SHORT).show()
                        } else {
                            val options = ScanOptions().setDesiredBarcodeFormats(ScanOptions.QR_CODE).setPrompt("Escanea un QR").setBeepEnabled(true)
                            qrScannerLauncher.launch(options)
                        }
                    }
                )
            }
        }
    ) { innerPadding ->
        Column(modifier = Modifier.padding(innerPadding)) {
            OutlinedTextField(
                value = textoBusqueda,
                onValueChange = viewModel::onTextoBusquedaChange,
                modifier = Modifier.fillMaxWidth().padding(horizontal = 16.dp, vertical = 8.dp),
                label = { Text("Buscar por Apodo o Especie...") },
                singleLine = true,
                leadingIcon = { Icon(Icons.Default.Search, contentDescription = null) }
            )
            if (coloniasAgrupadas.isEmpty()) {
                Box(modifier = Modifier.fillMaxSize().padding(8.dp), contentAlignment = Alignment.Center) {
                    Text(if (textoBusqueda.isNotBlank()) "No se encontraron colonias." else "No hay colonias registradas.\nUsa el Dashboard para añadir una nueva.")
                }
            } else {
                LazyColumn(modifier = Modifier.padding(horizontal = 8.dp)) {
                    coloniasAgrupadas.forEach { (especie, coloniasDeLaEspecie) ->
                        stickyHeader {
                            Surface(modifier = Modifier.fillMaxWidth(), color = MaterialTheme.colorScheme.surfaceVariant) {
                                Text(
                                    text = "$especie (${coloniasDeLaEspecie.size})",
                                    style = MaterialTheme.typography.titleMedium,
                                    modifier = Modifier.padding(horizontal = 8.dp, vertical = 4.dp)
                                )
                            }
                        }
                        items(coloniasDeLaEspecie) { colonia ->
                            ColoniaItem(colonia = colonia, onClick = { onColoniaClick(colonia.id) })
                        }
                    }
                }
            }
        }
    }
    mostrarDialogoDesconocido?.let { colonia ->
        AlertDialog(
            onDismissRequest = { mostrarDialogoDesconocido = null },
            title = { Text("Código no Registrado") },
            text = {
                Column {
                    Text("El código con ID '${colonia.id}' no está en tu base de datos.")
                    Spacer(modifier = Modifier.height(8.dp))
                    Text("¿Quieres importar los datos de este código como una nueva colonia?", style = MaterialTheme.typography.bodyMedium)
                    Spacer(modifier = Modifier.height(16.dp))
                    Text("Datos en el código:", fontWeight = FontWeight.Bold)
                    Text("- Apodo: ${colonia.nombre}")
                    Text("- Especie: ${colonia.especie}")
                }
            },
            dismissButton = { TextButton(onClick = { mostrarDialogoDesconocido = null }) { Text("Cerrar") } },
            confirmButton = {
                Button(onClick = {
                    viewModel.insertar(colonia)
                    Toast.makeText(context, "'${colonia.nombre}' importada con éxito.", Toast.LENGTH_SHORT).show()
                    mostrarDialogoDesconocido = null
                }) { Text("Importar") }
            }
        )
    }

    mostrarDialogoConflicto?.let { colonia ->
        AlertDialog(
            onDismissRequest = { mostrarDialogoConflicto = null },
            title = { Text("Conflicto de ID") },
            text = {
                Text("Ya tienes una colonia con el ID '${colonia.id}'. ¿Qué deseas hacer con los datos escaneados?")
            },
            confirmButton = {
                Column(modifier = Modifier.fillMaxWidth()) {
                    Button(
                        onClick = {
                            viewModel.actualizar(colonia)
                            Toast.makeText(context, "Colonia '${colonia.nombre}' sobrescrita.", Toast.LENGTH_SHORT).show()
                            mostrarDialogoConflicto = null
                        },
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text("Sobrescribir")
                    }
                    Spacer(modifier = Modifier.height(8.dp))
                    OutlinedButton(
                        onClick = {
                            viewModel.importarComoNueva(colonia)
                            Toast.makeText(context, "Colonia importada como nueva.", Toast.LENGTH_SHORT).show()
                            mostrarDialogoConflicto = null
                        },
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text("Importar como Nueva")
                    }
                }
            },
            dismissButton = {
                TextButton(onClick = { mostrarDialogoConflicto = null }) {
                    Text("Cancelar")
                }
            }
        )
    }
}


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaAnadirEditarColonia(
    coloniaId: String?,
    scannedId: String?,
    coloniaJson: String?,
    idYaExiste: Boolean,
    viewModel: ColoniaViewModel,
    onNavigateBack: () -> Unit,
    onGuardarColonia: (Colonia) -> Unit
) {
    val ninjaModeActivado by viewModel.ninjaModeActivado.collectAsState()
    val coloniaExistente by if (coloniaId != null) {
        viewModel.obtenerColoniaFullDetail(coloniaId).collectAsState(initial = null)
    } else {
        remember { mutableStateOf(null) }
    }
    var nombre by remember { mutableStateOf("") }
    var especie by remember { mutableStateOf("") }
    val ahora = remember { System.currentTimeMillis() }
    var fechaTexto by remember { mutableStateOf(formatearTimestamp(ahora, "dd/MM/yyyy")) }
    var horaTexto by remember { mutableStateOf(formatearTimestamp(ahora, "HH:mm")) }
    var latitud by remember { mutableStateOf<Double?>(null) }
    var longitud by remember { mutableStateOf<Double?>(null) }
    var temperatura by remember { mutableStateOf("") }
    var clima by remember { mutableStateOf("") }
    var textoBotonGps by remember { mutableStateOf("Obtener Ubicación GPS") }
    var textoBotonClima by remember { mutableStateOf("Obtener Clima") }
    var tieneHuevos by remember { mutableStateOf(false) }
    var tieneLarvas by remember { mutableStateOf(false) }
    var tienePupas by remember { mutableStateOf(false) }
    var tieneObreras by remember { mutableStateOf(false) }
    var estaEnferma by remember { mutableStateOf(false) }

    // Este estado ahora solo se usa para la lógica de guardado.
    var idOriginalEscaneado by remember { mutableStateOf<String?>(null) }
    // Este estado es para lo que se muestra en el campo de texto de la UI.
    var idParaMostrar by remember { mutableStateOf<String?>(scannedId) }

    var mostrarDialogoGuardado by remember { mutableStateOf(false) }
    var qrParaMostrar by remember { mutableStateOf<String?>(null) }
    var mostrarDialogoError by remember { mutableStateOf(false) }
    var mensajeDeError by remember { mutableStateOf("") }
    var mostrarDialogoEtiquetaInvalida by remember { mutableStateOf(false) }
    val apiKey by viewModel.apiKey.collectAsState()
    val context = LocalContext.current
    val coroutineScope = rememberCoroutineScope()

    LaunchedEffect(coloniaJson, idYaExiste) {
        if (!coloniaJson.isNullOrBlank()) {
            try {
                val c = Gson().fromJson(Uri.decode(coloniaJson), Colonia::class.java)

                // **LA CORRECCIÓN CLAVE**
                // Guardamos el ID real para la lógica interna.
                idOriginalEscaneado = c.id

                // Si el ID ya existe, no lo mostramos en la UI.
                if (!idYaExiste) {
                    idParaMostrar = c.id
                }

                // El resto de los datos se rellenan siempre.
                nombre = c.nombre; especie = c.especie
                fechaTexto = formatearTimestamp(c.fecha, "dd/MM/yyyy")
                horaTexto = formatearTimestamp(c.fecha, "HH:mm")
                if (!ninjaModeActivado) { latitud = c.latitud; longitud = c.longitud }
                temperatura = c.temperatura?.toString() ?: ""
                clima = c.clima ?: ""
                tieneHuevos = c.tieneHuevos; tieneLarvas = c.tieneLarvas; tienePupas = c.tienePupas
                tieneObreras = c.tieneObreras; estaEnferma = c.estaEnferma
                if (c.latitud != null && !ninjaModeActivado) textoBotonGps = "Ubicación Obtenida"
            } catch (e: Exception) {
                e.printStackTrace()
                Toast.makeText(context, "Error al leer los datos de la etiqueta.", Toast.LENGTH_SHORT).show()
            }
        }
    }

    LaunchedEffect(coloniaExistente) {
        if (coloniaId != null) {
            coloniaExistente?.colonia?.let {
                idParaMostrar = it.id
                nombre = it.nombre; especie = it.especie
                fechaTexto = formatearTimestamp(it.fecha, "dd/MM/yyyy")
                horaTexto = formatearTimestamp(it.fecha, "HH:mm")
                if (!ninjaModeActivado) {
                    latitud = it.latitud; longitud = it.longitud
                }
                temperatura = it.temperatura?.toString() ?: ""
                clima = it.clima ?: ""
                tieneHuevos = it.tieneHuevos; tieneLarvas = it.tieneLarvas; tienePupas = it.tienePupas
                tieneObreras = it.tieneObreras; estaEnferma = it.estaEnferma
                if (it.latitud != null && !ninjaModeActivado) textoBotonGps = "Ubicación Obtenida"
            }
        }
    }

    val titulo = if (coloniaId == null) "Añadir Nueva Colonia" else "Editar Colonia"
    val locationPermissionLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.RequestMultiplePermissions(),
        onResult = { permissions ->
            if (permissions.getOrDefault(Manifest.permission.ACCESS_FINE_LOCATION, false)) {
                textoBotonGps = "Obteniendo..."; obtenerUbicacion(context) { location ->
                    latitud = location?.latitude; longitud = location?.longitude; textoBotonGps = "Ubicación Obtenida"
                }
            } else { textoBotonGps = "Permiso denegado" }
        }
    )

    Scaffold(topBar = { TopAppBar(title = { Text(titulo) }) }) { innerPadding ->
        Column(modifier = Modifier.padding(innerPadding).padding(16.dp).fillMaxSize().verticalScroll(rememberScrollState())) {
            if (coloniaId == null) {
                OutlinedTextField(
                    // Mostramos el estado 'idParaMostrar'
                    value = idParaMostrar ?: "ID se asignará al guardar",
                    onValueChange = {}, label = { Text("ID de la Colonia") },
                    modifier = Modifier.fillMaxWidth(), readOnly = true
                )
                Spacer(modifier = Modifier.height(8.dp))
            }
            OutlinedTextField(value = nombre, onValueChange = { nombre = it }, label = { Text("Apodo de la colonia") }, modifier = Modifier.fillMaxWidth(), singleLine = true)
            Spacer(modifier = Modifier.height(8.dp))
            OutlinedTextField(value = especie, onValueChange = { especie = it }, label = { Text("Especie") }, modifier = Modifier.fillMaxWidth(), singleLine = true)
            Spacer(modifier = Modifier.height(16.dp))
            Row {
                OutlinedTextField(value = fechaTexto, onValueChange = { fechaTexto = it }, label = { Text("Fecha (dd/MM/yyyy)") }, modifier = Modifier.weight(1f))
                Spacer(modifier = Modifier.width(8.dp))
                OutlinedTextField(value = horaTexto, onValueChange = { horaTexto = it }, label = { Text("Hora (HH:mm)") }, modifier = Modifier.weight(1f))
            }
            Spacer(modifier = Modifier.height(16.dp))
            Text("Estado de la cría:", style = MaterialTheme.typography.titleMedium)
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.SpaceAround) {
                CheckboxConEtiqueta(texto = "Huevos", checked = tieneHuevos, onCheckedChange = { tieneHuevos = it })
                CheckboxConEtiqueta(texto = "Larvas", checked = tieneLarvas, onCheckedChange = { tieneLarvas = it })
                CheckboxConEtiqueta(texto = "Pupas", checked = tienePupas, onCheckedChange = { tienePupas = it })
                CheckboxConEtiqueta(texto = "Obreras", checked = tieneObreras, onCheckedChange = { tieneObreras = it })
            }
            Spacer(modifier = Modifier.height(8.dp))
            Row(verticalAlignment = Alignment.CenterVertically) {
                Checkbox(checked = estaEnferma, onCheckedChange = { estaEnferma = it })
                Text("Marcar como enferma/en problemas")
            }

            if (!ninjaModeActivado) {
                Spacer(modifier = Modifier.height(16.dp))
                Button(
                    onClick = {
                        if (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
                            textoBotonGps = "Obteniendo..."; obtenerUbicacion(context) { location ->
                                latitud = location?.latitude; longitud = location?.longitude; textoBotonGps = "Ubicación Obtenida"
                            }
                        } else { locationPermissionLauncher.launch(arrayOf(Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION)) }
                    },
                    enabled = latitud == null
                ) { Text(textoBotonGps) }
                if (latitud != null) { Text("Lat: $latitud, Lon: $longitud") }
            }

            Spacer(modifier = Modifier.height(16.dp))
            Row(verticalAlignment = Alignment.CenterVertically) {
                OutlinedTextField(value = temperatura, onValueChange = { temperatura = it }, label = { Text("Temp (°C)") }, modifier = Modifier.weight(1f))
                Spacer(modifier = Modifier.width(8.dp))
                OutlinedTextField(value = clima, onValueChange = { clima = it }, label = { Text("Clima") }, modifier = Modifier.weight(2f))
            }
            Spacer(modifier = Modifier.height(8.dp))
            Button(
                onClick = {
                    val lat = latitud
                    val lon = longitud
                    if (lat != null && lon != null) {
                        textoBotonClima = "Obteniendo..."
                        coroutineScope.launch {
                            try {
                                val response = RetrofitClient.instance.getCurrentWeather(lat, lon, apiKey)
                                temperatura = response.main.temp.toInt().toString()
                                clima = response.weather.firstOrNull()?.description?.replaceFirstChar { it.uppercase() } ?: ""
                                textoBotonClima = "Clima Obtenido"
                            } catch (e: Exception) {
                                textoBotonClima = "Error (Ver Detalles)"
                                mensajeDeError = e.localizedMessage ?: "Ocurrió un error desconocido."
                                mostrarDialogoError = true
                                e.printStackTrace()
                            }
                        }
                    }
                },
                enabled = latitud != null && !ninjaModeActivado
            ) { Text(textoBotonClima) }
            Spacer(modifier = Modifier.weight(1f, fill = true))
            Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.End) {
                TextButton(onClick = onNavigateBack) { Text("Cancelar") }
                Spacer(modifier = Modifier.width(8.dp))
                Button(onClick = {
                    if (nombre.isNotBlank() && especie.isNotBlank()) {
                        mostrarDialogoGuardado = true
                    } else {
                        Toast.makeText(context, "El apodo y la especie son obligatorios", Toast.LENGTH_SHORT).show()
                    }
                }) { Text("Guardar...") }
            }
        }
    }

    if (mostrarDialogoError) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoError = false },
            title = { Text("Error al Obtener el Clima") },
            text = { Text(mensajeDeError) },
            confirmButton = { TextButton(onClick = { mostrarDialogoError = false }) { Text("Cerrar") } }
        )
    }

    if (mostrarDialogoGuardado) {
        OpcionesDeGuardadoDialog(
            viewModel = viewModel,
            coloniaId = coloniaId,
            // Pasamos el ID real a la lógica de guardado
            idEscaneado = idOriginalEscaneado ?: scannedId,
            idYaExiste = idYaExiste,
            onDismiss = { mostrarDialogoGuardado = false },
            onGuardarAccion = { colonia, accion, idCambiado ->
                onGuardarColonia(colonia)

                if (idCambiado) {
                    mostrarDialogoEtiquetaInvalida = true
                } else {
                    onNavigateBack()
                }

                when (accion) {
                    "nfc" -> {
                        viewModel.activarModoEscritura(crearPayloadNfc(colonia))
                        Toast.makeText(context, "Datos guardados. Acerca la etiqueta para escribir.", Toast.LENGTH_LONG).show()
                    }
                    "generar_qr" -> {
                        qrParaMostrar = colonia.id
                    }
                }
            },
            formState = FormState(nombre, especie, fechaTexto, horaTexto, latitud, longitud, temperatura, clima, tieneHuevos, tieneLarvas, tienePupas, tieneObreras, estaEnferma, coloniaExistente?.colonia?.fecha)
        )
    }

    if (mostrarDialogoEtiquetaInvalida) {
        AlertDialog(
            onDismissRequest = { onNavigateBack() },
            title = { Text("¡ID Cambiado!") },
            text = { Text("La colonia se guardó con un nuevo ID porque el que escaneaste ya estaba en uso. La etiqueta NFC/QR original ya no funcionará para esta colonia. Deberás vincular una nueva desde la pantalla de detalles.") },
            confirmButton = {
                Button(onClick = { onNavigateBack() }) { Text("Entendido") }
            }
        )
    }

    qrParaMostrar?.let { texto ->
        val bitmap = generarQrBitmap(texto)
        if (bitmap != null) {
            QrDisplayDialog(bitmap = bitmap, onDismiss = { qrParaMostrar = null; onNavigateBack() })
        }
    }
}

data class FormState(
    val nombre: String, val especie: String, val fechaTexto: String, val horaTexto: String,
    val latitud: Double?, val longitud: Double?, val temperatura: String, val clima: String,
    val tieneHuevos: Boolean, val tieneLarvas: Boolean, val tienePupas: Boolean, val tieneObreras: Boolean,
    val estaEnferma: Boolean, val fechaOriginal: Long?
)

@Composable
fun OpcionesDeGuardadoDialog(
    viewModel: ColoniaViewModel,
    coloniaId: String?,
    idEscaneado: String?,
    idYaExiste: Boolean,
    onDismiss: () -> Unit,
    onGuardarAccion: (Colonia, String, Boolean) -> Unit,
    formState: FormState
) {
    val coroutineScope = rememberCoroutineScope()
    val context = LocalContext.current

    val qrScannerLauncher = rememberLauncherForActivityResult(ScanContract()) { result ->
        result.contents?.let { id ->
            coroutineScope.launch {
                if (viewModel.idExiste(id)) {
                    Toast.makeText(context, "Error: El ID '$id' ya está en uso.", Toast.LENGTH_LONG).show()
                } else {
                    val colonia = construirColoniaDesdeFormulario(
                        id = id, nombre = formState.nombre, especie = formState.especie, fechaTexto = formState.fechaTexto, horaTexto = formState.horaTexto, latitud = formState.latitud,
                        longitud = formState.longitud, temperatura = formState.temperatura, clima = formState.clima, tieneHuevos = formState.tieneHuevos, tieneLarvas = formState.tieneLarvas,
                        tienePupas = formState.tienePupas, tieneObreras = formState.tieneObreras, estaEnferma = formState.estaEnferma, fechaOriginal = formState.fechaOriginal, tipoEtiqueta = "QR"
                    )
                    if (colonia != null) {
                        onGuardarAccion(colonia, "escanear_qr", false)
                        onDismiss()
                    }
                }
            }
        }
    }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Opciones de Guardado") },
        dismissButton = { TextButton(onClick = onDismiss) { Text("Cancelar") } },
        confirmButton = {},
        text = {
            Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
                Button(onClick = {
                    coroutineScope.launch {
                        viewModel.cancelarModoNfc()
                        val idCambiado = coloniaId == null && idEscaneado != null && idYaExiste
                        val idFinal = if (idCambiado) {
                            generarIdCortoUnico(viewModel)
                        } else {
                            coloniaId ?: idEscaneado ?: generarIdCortoUnico(viewModel)
                        }

                        val colonia = construirColoniaDesdeFormulario(id = idFinal, nombre = formState.nombre, especie = formState.especie, fechaTexto = formState.fechaTexto, horaTexto = formState.horaTexto,
                            latitud = formState.latitud, longitud = formState.longitud, temperatura = formState.temperatura, clima = formState.clima,
                            tieneHuevos = formState.tieneHuevos, tieneLarvas = formState.tieneLarvas, tienePupas = formState.tienePupas,
                            tieneObreras = formState.tieneObreras, estaEnferma = formState.estaEnferma, fechaOriginal = formState.fechaOriginal, tipoEtiqueta = "NFC")
                        if (colonia != null) { onGuardarAccion(colonia, "nfc", idCambiado) }
                        onDismiss()
                    }
                }, modifier = Modifier.fillMaxWidth()) { Text("Guardar y Escribir NFC") }

                if (coloniaId == null) {
                    Button(onClick = {
                        val options = ScanOptions().setPrompt("Escanea un QR pre-impreso").setBeepEnabled(true)
                        qrScannerLauncher.launch(options)
                    }, modifier = Modifier.fillMaxWidth()) { Text("Guardar con QR Escaneado") }
                    Button(onClick = {
                        coroutineScope.launch {
                            val idCambiado = idEscaneado != null && idYaExiste
                            val idFinal = if (idCambiado) {
                                generarIdCortoUnico(viewModel)
                            } else {
                                idEscaneado ?: generarIdCortoUnico(viewModel)
                            }
                            val colonia = construirColoniaDesdeFormulario(id = idFinal, nombre = formState.nombre, especie = formState.especie, fechaTexto = formState.fechaTexto, horaTexto = formState.horaTexto,
                                latitud = formState.latitud, longitud = formState.longitud, temperatura = formState.temperatura, clima = formState.clima,
                                tieneHuevos = formState.tieneHuevos, tieneLarvas = formState.tieneLarvas, tienePupas = formState.tienePupas,
                                tieneObreras = formState.tieneObreras, estaEnferma = formState.estaEnferma, fechaOriginal = formState.fechaOriginal, tipoEtiqueta = "QR")
                            if (colonia != null) { onGuardarAccion(colonia, "generar_qr", idCambiado) }
                            onDismiss()
                        }
                    }, modifier = Modifier.fillMaxWidth()) { Text("Guardar y Generar QR Nuevo") }
                }

                OutlinedButton(onClick = {
                    coroutineScope.launch {
                        val idCambiado = coloniaId == null && idEscaneado != null && idYaExiste
                        val idFinal = if (idCambiado) {
                            generarIdCortoUnico(viewModel)
                        } else {
                            coloniaId ?: idEscaneado ?: generarIdCortoUnico(viewModel)
                        }
                        val colonia = construirColoniaDesdeFormulario(id = idFinal, nombre = formState.nombre, especie = formState.especie, fechaTexto = formState.fechaTexto, horaTexto = formState.horaTexto,
                            latitud = formState.latitud, longitud = formState.longitud, temperatura = formState.temperatura, clima = formState.clima,
                            tieneHuevos = formState.tieneHuevos, tieneLarvas = formState.tieneLarvas, tienePupas = formState.tienePupas,
                            tieneObreras = formState.tieneObreras, estaEnferma = formState.estaEnferma, fechaOriginal = formState.fechaOriginal, tipoEtiqueta = "Ninguna")
                        if (colonia != null) { onGuardarAccion(colonia, "solo_guardar", idCambiado) }
                        onDismiss()
                    }
                }, modifier = Modifier.fillMaxWidth()) { Text(if (coloniaId != null) "Solo Guardar Cambios" else "Solo Guardar (Sin Etiqueta)") }
            }
        }
    )
}

@Composable
fun QrDisplayDialog(bitmap: androidx.compose.ui.graphics.ImageBitmap, onDismiss: () -> Unit) {
    val context = LocalContext.current
    val sdf = SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault())
    val timestamp = sdf.format(Date())
    val nombreArchivo = "QR_Colonia_$timestamp"

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Código QR Generado") },
        text = {
            Column(modifier = Modifier.fillMaxWidth(), horizontalAlignment = Alignment.CenterHorizontally) {
                Image(bitmap = bitmap, contentDescription = "Código QR", modifier = Modifier.size(256.dp))
                Text("¡Listo para escanear o hacer captura de pantalla!")
            }
        },
        confirmButton = {
            Row {
                Button(onClick = {
                    guardarBitmapEnGaleria(context, bitmap, nombreArchivo)
                }) {
                    Text("Guardar")
                }
                Spacer(modifier = Modifier.width(8.dp))
                TextButton(onClick = onDismiss) { Text("Cerrar") }
            }
        }
    )
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaDetalleColonia(
    coloniaId: String, viewModel: ColoniaViewModel, onNavigateBack: () -> Unit, onNavigateToEditar: (String) -> Unit
) {
    val coloniaDetail by viewModel.obtenerColoniaFullDetail(coloniaId).collectAsState(initial = null)
    val alimentacion by viewModel.obtenerAlimentacion(coloniaId).collectAsState(initial = emptyList())
    val ninjaModeActivado by viewModel.ninjaModeActivado.collectAsState()
    var mostrarDialogoBorrado by remember { mutableStateOf(false) }
    var mostrarDialogoNota by remember { mutableStateOf(false) }
    var textoNota by remember { mutableStateOf("") }
    var notaAEditar by remember { mutableStateOf<Nota?>(null) }
    var mostrarDialogoEtiqueta by remember { mutableStateOf(false) }
    var mostrarDialogoAlimentacion by remember { mutableStateOf(false) }
    var alimentacionAEditar by remember { mutableStateOf<Alimentacion?>(null) }
    var tipoComida by remember { mutableStateOf("") }
    var notaComida by remember { mutableStateOf("") }
    val context = LocalContext.current
    var fotoSeleccionadaUri by remember { mutableStateOf<String?>(null) }
    var qrParaMostrar by remember { mutableStateOf<String?>(null) }
    var mostrarDialogoRecordatorio by remember { mutableStateOf(false) }

    val pickMediaLauncher = rememberLauncherForActivityResult(ActivityResultContracts.PickVisualMedia()) { uri ->
        if (uri != null) {
            viewModel.anadirFoto(coloniaId, uri, context)
        }
    }

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text(coloniaDetail?.colonia?.nombre ?: "Cargando...") },
                actions = {
                    IconButton(onClick = { mostrarDialogoRecordatorio = true }) {
                        Icon(Icons.Default.Alarm, contentDescription = "Configurar Recordatorio")
                    }
                    IconButton(onClick = {
                        coloniaDetail?.colonia?.let {
                            val payload = crearPayloadNfc(it)
                            qrParaMostrar = payload
                        }
                    }) {
                        Icon(Icons.Default.QrCode, contentDescription = "Generar QR de Venta")
                    }
                    IconButton(onClick = {
                        viewModel.exportarColoniaEspecifica(coloniaId) { jsonString ->
                            try {
                                val file = File(context.cacheDir, "colonia_${coloniaId}.json")
                                file.writeText(jsonString)
                                val uri = FileProvider.getUriForFile(context, "${context.packageName}.provider", file)
                                val sendIntent = Intent(Intent.ACTION_SEND).apply {
                                    type = "application/json"
                                    putExtra(Intent.EXTRA_STREAM, uri)
                                    putExtra(Intent.EXTRA_SUBJECT, "Exportación de Colonia: ${coloniaDetail?.colonia?.nombre}")
                                    addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                                }
                                val shareIntent = Intent.createChooser(sendIntent, "Compartir registro como JSON...")
                                context.startActivity(shareIntent)
                            } catch (e: Exception) {
                                e.printStackTrace()
                                Toast.makeText(context, "Error al exportar el archivo.", Toast.LENGTH_SHORT).show()
                            }
                        }
                    }) {
                        Icon(Icons.Default.Share, contentDescription = "Compartir Colonia")
                    }
                }
            )
        }
    ) { innerPadding ->
        if (coloniaDetail == null) {
            Box(modifier = Modifier.fillMaxSize().padding(innerPadding), contentAlignment = Alignment.Center) { CircularProgressIndicator() }
        } else {
            val colonia = coloniaDetail!!.colonia
            LazyColumn(modifier = Modifier.padding(innerPadding).padding(horizontal = 16.dp)) {
                item { Spacer(modifier = Modifier.height(16.dp)) }

                // --- Sección de Detalles (sin cambios) ---
                item {
                    DetalleItem(etiqueta = "Apodo", valor = colonia.nombre)
                    DetalleItem(etiqueta = "Especie", valor = colonia.especie)
                    DetalleItem(etiqueta = "ID", valor = colonia.id)
                    DetalleItem(etiqueta = "Fecha de registro", valor = formatearTimestamp(colonia.fecha, "dd/MM/yyyy 'a las' HH:mm"))
                    if (!ninjaModeActivado) {
                        colonia.latitud?.let { DetalleItem(etiqueta = "Coordenadas", valor = "${colonia.latitud}, ${colonia.longitud}") }
                    }
                    colonia.temperatura?.let { DetalleItem(etiqueta = "Condiciones", valor = "${colonia.temperatura}°C, ${colonia.clima}") }
                    Spacer(modifier = Modifier.height(8.dp))
                    Text("Estado Actual:", style = MaterialTheme.typography.labelSmall, color = MaterialTheme.colorScheme.primary)
                    Row(modifier = Modifier.fillMaxWidth(), horizontalArrangement = Arrangement.spacedBy(8.dp)) {
                        if (colonia.tieneHuevos) Text("🥚 Huevos", style = MaterialTheme.typography.bodyLarge)
                        if (colonia.tieneLarvas) Text("🐛 Larvas", style = MaterialTheme.typography.bodyLarge)
                        if (colonia.tienePupas) Text("⚪ Pupas", style = MaterialTheme.typography.bodyLarge)
                        if (colonia.tieneObreras) Text("🐜 Obreras", style = MaterialTheme.typography.bodyLarge)
                    }
                    if (colonia.estaEnferma) {
                        Text("⚠️ ¡Enferma!", color = MaterialTheme.colorScheme.error, fontWeight = FontWeight.Bold, modifier = Modifier.padding(top = 4.dp))
                    }
                    colonia.recordatorioProximaFecha?.let {
                        Spacer(modifier = Modifier.height(8.dp))
                        Row(verticalAlignment = Alignment.CenterVertically, modifier = Modifier.fillMaxWidth()) {
                            Icon(Icons.Default.AlarmOn, contentDescription = null, tint = MaterialTheme.colorScheme.primary)
                            Spacer(Modifier.width(8.dp))
                            Column(modifier = Modifier.weight(1f)) {
                                Text(
                                    when (colonia.recordatorioTipo) {
                                        "DIARIO" -> "Recordatorio diario a las ${formatearTimestamp(it, "HH:mm")}"
                                        "CADA_X_DIAS" -> "Recordatorio cada ${colonia.recordatorioIntervalo} días a las ${formatearTimestamp(it, "HH:mm")}"
                                        else -> "Próximo recordatorio:"
                                    }, style = MaterialTheme.typography.labelSmall
                                )
                                Text(formatearTimestamp(it, "dd/MM/yyyy 'a las' HH:mm"), style = MaterialTheme.typography.bodyLarge)
                            }
                            IconButton(onClick = { viewModel.cancelarRecordatorio(colonia, context) }) {
                                Icon(Icons.Default.Close, contentDescription = "Cancelar Recordatorio")
                            }
                        }
                    }
                    Spacer(modifier = Modifier.height(16.dp))
                }

                // --- Sección Etiqueta (sin cambios) ---
                item {
                    Row(verticalAlignment = Alignment.CenterVertically) {
                        Icon(imageVector = if (colonia.tipoEtiqueta != "Ninguna") Icons.Default.Link else Icons.Default.LinkOff, contentDescription = null)
                        Spacer(modifier = Modifier.width(8.dp))
                        Text("Etiqueta: ${colonia.tipoEtiqueta}")
                        Spacer(modifier = Modifier.weight(1f))
                        OutlinedButton(onClick = {
                            if (colonia.tipoEtiqueta == "QR") { qrParaMostrar = colonia.id }
                            else if (colonia.tipoEtiqueta == "NFC") {
                                viewModel.activarModoEscritura(crearPayloadNfc(colonia))
                                Toast.makeText(context, "Acerca la etiqueta para re-escribir...", Toast.LENGTH_SHORT).show()
                            } else { mostrarDialogoEtiqueta = true }
                        }) {
                            Text( when (colonia.tipoEtiqueta) { "QR" -> "Ver QR"; "NFC" -> "Re-escribir NFC"; else -> "Vincular" } )
                        }
                    }
                    Spacer(modifier = Modifier.height(16.dp))
                }

                // --- Sección Registro de Alimentación (MODIFICADA) ---
                item {
                    Row(modifier = Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {
                        Text("Registro de Alimentación", style = MaterialTheme.typography.titleMedium, modifier = Modifier.weight(1f))
                        Button(onClick = {
                            alimentacionAEditar = null; tipoComida = ""; notaComida = ""; mostrarDialogoAlimentacion = true
                        }) { Icon(Icons.Default.Add, contentDescription = "Añadir Alimentación") }
                    }
                }
                // Contenedor scrollable para Alimentación
                item {
                    Column(
                        modifier = Modifier
                            .height(200.dp) // <-- Altura fija para el área de scroll
                            .fillMaxWidth()
                            .verticalScroll(rememberScrollState()) // <-- Habilita el scroll
                            .padding(vertical = 4.dp)
                    ) {
                        if (alimentacion.isEmpty()) {
                            Text("No hay registros de alimentación.", modifier = Modifier.padding(top = 8.dp))
                        } else {
                            alimentacion.forEach { registro ->
                                AlimentacionItem(registro = registro, onClick = {
                                    alimentacionAEditar = registro
                                    tipoComida = registro.tipoComida
                                    notaComida = registro.nota
                                    mostrarDialogoAlimentacion = true
                                })
                                Spacer(Modifier.height(4.dp)) // Espacio entre items
                            }
                        }
                    }
                }

                // --- Sección Historial de Notas (MODIFICADA) ---
                item {
                    Spacer(modifier = Modifier.height(16.dp)) // Espacio antes de la sección
                    Row(modifier = Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {
                        Text("Historial de Notas", style = MaterialTheme.typography.titleMedium, modifier = Modifier.weight(1f))
                        Button(onClick = { notaAEditar = null; textoNota = ""; mostrarDialogoNota = true }) {
                            Icon(Icons.Default.Add, contentDescription = "Añadir Nota")
                        }
                    }
                }
                // Contenedor scrollable para Notas
                item {
                    Column(
                        modifier = Modifier
                            .height(200.dp) // <-- Altura fija para el área de scroll
                            .fillMaxWidth()
                            .verticalScroll(rememberScrollState()) // <-- Habilita el scroll
                            .padding(vertical = 4.dp)
                    ) {
                        val sortedNotas = coloniaDetail!!.notas.sortedByDescending { it.fecha }
                        if (sortedNotas.isEmpty()) {
                            Text("No hay notas.", modifier = Modifier.padding(top = 8.dp))
                        } else {
                            sortedNotas.forEach { nota ->
                                NotaItem(nota = nota, onClick = {
                                    notaAEditar = nota; textoNota = nota.texto; mostrarDialogoNota = true
                                })
                                Spacer(Modifier.height(4.dp)) // Espacio entre items
                            }
                        }
                    }
                }

                // --- Sección Galería de Fotos (sin cambios) ---
                item {
                    Spacer(modifier = Modifier.height(16.dp))
                    Row(modifier = Modifier.fillMaxWidth(), verticalAlignment = Alignment.CenterVertically) {
                        Text("Galería de Fotos", style = MaterialTheme.typography.titleMedium, modifier = Modifier.weight(1f))
                        Button(onClick = { pickMediaLauncher.launch(PickVisualMediaRequest(ActivityResultContracts.PickVisualMedia.ImageOnly)) }) {
                            Icon(Icons.Default.AddAPhoto, contentDescription = "Añadir Foto")
                        }
                    }
                }
                item {
                    if (coloniaDetail!!.fotos.isEmpty()) {
                        Text("No hay fotos en la galería.", modifier = Modifier.padding(vertical = 8.dp))
                    } else {
                        LazyVerticalGrid(
                            columns = GridCells.Adaptive(minSize = 128.dp),
                            modifier = Modifier.height(280.dp).padding(vertical = 8.dp), // Altura para la cuadrícula
                            verticalArrangement = Arrangement.spacedBy(4.dp),
                            horizontalArrangement = Arrangement.spacedBy(4.dp)
                        ) {
                            items(coloniaDetail!!.fotos.sortedByDescending { it.fecha }) { foto ->
                                AsyncImage(
                                    model = Uri.parse(foto.uri), contentDescription = foto.descripcion,
                                    modifier = Modifier.aspectRatio(1f).clickable { fotoSeleccionadaUri = foto.uri },
                                    contentScale = ContentScale.Crop
                                )
                            }
                        }
                    }
                }

                // --- Sección Botones (sin cambios) ---
                item {
                    Spacer(modifier = Modifier.height(16.dp))
                    Row(modifier = Modifier.fillMaxWidth().padding(bottom = 16.dp), horizontalArrangement = Arrangement.SpaceAround) {
                        Button(onClick = onNavigateBack, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.secondary)) { Text("Volver") }
                        Button(onClick = { onNavigateToEditar(colonia.id) }) { Text("Editar") }
                        Button(onClick = { mostrarDialogoBorrado = true }, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)) { Text("Eliminar") }
                    }
                }
            }
        }
    }

    fotoSeleccionadaUri?.let { uri ->
        DialogoFotoEnGrande(
            uri = uri,
            onDismiss = { fotoSeleccionadaUri = null }
        )
    }

    qrParaMostrar?.let { texto ->
        val bitmap = generarQrBitmap(texto)
        if (bitmap != null) {
            QrDisplayDialog(bitmap = bitmap, onDismiss = { qrParaMostrar = null })
        }
    }

    if (mostrarDialogoBorrado) {
        val colonia = coloniaDetail?.colonia
        AlertDialog(
            onDismissRequest = { mostrarDialogoBorrado = false },
            title = { Text("Confirmar Borrado") },
            text = { Text("¿Estás seguro de que quieres eliminar la colonia '${colonia?.nombre}'? Esta acción no se puede deshacer.") },
            confirmButton = { Button(onClick = {
                colonia?.let { viewModel.borrar(it) }; mostrarDialogoBorrado = false; onNavigateBack()
            }, colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)) { Text("Eliminar") } },
            dismissButton = { TextButton(onClick = { mostrarDialogoBorrado = false }) { Text("Cancelar") } }
        )
    }
    if (mostrarDialogoNota) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoNota = false; notaAEditar = null; textoNota = "" },
            title = { Text(if (notaAEditar == null) "Añadir Nota" else "Editar Nota") },
            text = { OutlinedTextField(value = textoNota, onValueChange = { textoNota = it }, label = { Text("Escribe tu observación...") }) },
            confirmButton = {
                Button(onClick = {
                    if (textoNota.isNotBlank()) {
                        notaAEditar?.let { viewModel.actualizarNota(it.copy(texto = textoNota)) }
                            ?: viewModel.anadirNota(coloniaId, textoNota)
                        textoNota = ""; mostrarDialogoNota = false; notaAEditar = null
                    }
                }) { Text("Guardar") }
            },
            dismissButton = {
                Row {
                    notaAEditar?.let {
                        TextButton(onClick = {
                            viewModel.borrarNota(it)
                            textoNota = ""; mostrarDialogoNota = false; notaAEditar = null
                        }) { Text("Borrar", color = MaterialTheme.colorScheme.error) }
                        Spacer(modifier = Modifier.weight(1f))
                    }
                    TextButton(onClick = { textoNota = ""; mostrarDialogoNota = false; notaAEditar = null }) { Text("Cancelar") }
                }
            }
        )
    }
    if (mostrarDialogoEtiqueta) {
        coloniaDetail?.colonia?.let { colonia ->
            OpcionesDeEtiquetaDialog(
                viewModel = viewModel,
                colonia = colonia,
                onDismiss = { mostrarDialogoEtiqueta = false }
            )
        }
    }
    if (mostrarDialogoAlimentacion) {
        AlertDialog(
            onDismissRequest = { mostrarDialogoAlimentacion = false },
            title = { Text(if (alimentacionAEditar == null) "Añadir Comida" else "Editar Comida") },
            text = {
                Column {
                    OutlinedTextField(value = tipoComida, onValueChange = { tipoComida = it }, label = { Text("Tipo de Comida (Ej: Insectos)") })
                    Spacer(modifier = Modifier.height(8.dp))
                    OutlinedTextField(value = notaComida, onValueChange = { notaComida = it }, label = { Text("Nota (Opcional)") })
                }
            },
            confirmButton = {
                Button(onClick = {
                    if (tipoComida.isNotBlank()) {
                        alimentacionAEditar?.let {
                            viewModel.actualizarAlimentacion(it.copy(tipoComida = tipoComida, nota = notaComida))
                        } ?: viewModel.anadirAlimentacion(coloniaId, tipoComida, notaComida)
                        mostrarDialogoAlimentacion = false
                    }
                }) { Text("Guardar") }
            },
            dismissButton = {
                Row {
                    alimentacionAEditar?.let {
                        TextButton(onClick = {
                            viewModel.borrarAlimentacion(it)
                            mostrarDialogoAlimentacion = false
                        }) { Text("Borrar", color = MaterialTheme.colorScheme.error) }
                        Spacer(modifier = Modifier.weight(1f))
                    }
                    TextButton(onClick = { mostrarDialogoAlimentacion = false }) { Text("Cancelar") }
                }
            }
        )
    }

    if (mostrarDialogoRecordatorio) {
        coloniaDetail?.colonia?.let { colonia ->
            DialogoConfigurarRecordatorio(
                tipoInicial = colonia.recordatorioTipo,
                intervaloInicial = colonia.recordatorioIntervalo,
                horaInicial = colonia.recordatorioHora,
                onDismiss = { mostrarDialogoRecordatorio = false },
                onConfirm = { tipo, intervalo, hora, proximaFecha ->
                    val config = ColoniaViewModel.RecordatorioConfig(
                        tipo = tipo,
                        intervalo = intervalo,
                        hora = hora,
                        proximaFecha = proximaFecha
                    )
                    viewModel.guardarRecordatorio(colonia, config, context)
                    mostrarDialogoRecordatorio = false
                },
                onCancel = {
                    viewModel.cancelarRecordatorio(colonia, context)
                    mostrarDialogoRecordatorio = false
                }
            )
        }
    }
}

@Composable
fun DialogoFotoEnGrande(uri: String, onDismiss: () -> Unit) {
    Dialog(onDismissRequest = onDismiss) {
        Box(
            modifier = Modifier
                .fillMaxWidth()
                .fillMaxHeight(0.8f)
                .clickable { onDismiss() },
            contentAlignment = Alignment.Center
        ) {
            AsyncImage(
                model = Uri.parse(uri),
                contentDescription = "Foto en grande",
                modifier = Modifier.fillMaxSize(),
                contentScale = ContentScale.Fit
            )
        }
    }
}

@Composable
fun OpcionesDeEtiquetaDialog(
    viewModel: ColoniaViewModel,
    colonia: Colonia,
    onDismiss: () -> Unit
) {
    val context = LocalContext.current
    var qrParaMostrar by remember { mutableStateOf<String?>(null) }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Vincular Etiqueta para '${colonia.nombre}'") },
        text = {
            Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
                Button(
                    onClick = {
                        viewModel.activarModoEscritura(crearPayloadNfc(colonia))
                        viewModel.actualizar(colonia.copy(tipoEtiqueta = "NFC"))
                        Toast.makeText(context, "Acerca la etiqueta NFC para escribir...", Toast.LENGTH_LONG).show()
                        onDismiss()
                    },
                    modifier = Modifier.fillMaxWidth()
                ) { Text("Vincular con Etiqueta NFC") }

                Button(
                    onClick = {
                        qrParaMostrar = colonia.id
                        viewModel.actualizar(colonia.copy(tipoEtiqueta = "QR"))
                    },
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Text("Vincular con Código QR")
                }
            }
        },
        confirmButton = {
            TextButton(onClick = onDismiss) {
                Text("Cerrar")
            }
        }
    )

    qrParaMostrar?.let { texto ->
        val bitmap = generarQrBitmap(texto)
        if (bitmap != null) {
            QrDisplayDialog(bitmap = bitmap, onDismiss = { qrParaMostrar = null; onDismiss() })
        }
    }
}


@Composable
fun DialogoConfigurarRecordatorio(
    tipoInicial: String?,
    intervaloInicial: Int?,
    horaInicial: Int?,
    onDismiss: () -> Unit,
    onConfirm: (tipo: String, intervalo: Int?, hora: Int, proximaFecha: Long) -> Unit,
    onCancel: () -> Unit
) {
    val context = LocalContext.current
    val opciones = listOf("Una vez", "Diario", "Cada X días")
    var tipoSeleccionado by remember { mutableStateOf(
        when(tipoInicial) {
            "DIARIO" -> "Diario"
            "CADA_X_DIAS" -> "Cada X días"
            else -> "Una vez"
        }
    ) }
    var intervalo by remember { mutableStateOf(intervaloInicial?.toString() ?: "2") }

    val calendar = Calendar.getInstance()
    if (horaInicial != null) {
        calendar.set(Calendar.HOUR_OF_DAY, horaInicial / 60)
        calendar.set(Calendar.MINUTE, horaInicial % 60)
    }
    var fecha by remember { mutableStateOf(calendar.timeInMillis) }
    var hora by remember { mutableStateOf(Pair(calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE))) }

    val datePickerDialog = DatePickerDialog(
        context,
        { _, year, month, dayOfMonth ->
            val newCal = Calendar.getInstance().apply { timeInMillis = fecha }
            newCal.set(year, month, dayOfMonth)
            fecha = newCal.timeInMillis
        },
        calendar.get(Calendar.YEAR),
        calendar.get(Calendar.MONTH),
        calendar.get(Calendar.DAY_OF_MONTH)
    )

    val timePickerDialog = TimePickerDialog(
        context,
        { _, hourOfDay, minute -> hora = Pair(hourOfDay, minute) },
        hora.first, hora.second, true
    )

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text("Configurar Recordatorio") },
        text = {
            Column {
                Column(Modifier.selectableGroup()) {
                    opciones.forEach { text ->
                        Row(
                            Modifier
                                .fillMaxWidth()
                                .height(48.dp)
                                .selectable(
                                    selected = (text == tipoSeleccionado),
                                    onClick = { tipoSeleccionado = text },
                                    role = Role.RadioButton
                                )
                                .padding(horizontal = 16.dp),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            RadioButton(
                                selected = (text == tipoSeleccionado),
                                onClick = null
                            )
                            Text(text = text, modifier = Modifier.padding(start = 16.dp))
                        }
                    }
                }

                if (tipoSeleccionado == "Cada X días") {
                    OutlinedTextField(
                        value = intervalo,
                        onValueChange = { intervalo = it.filter { char -> char.isDigit() } },
                        label = { Text("Intervalo en días") },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                        modifier = Modifier.padding(top = 8.dp, start = 16.dp, end = 16.dp)
                    )
                }

                Spacer(Modifier.height(16.dp))
                Row(verticalAlignment = Alignment.CenterVertically) {
                    if (tipoSeleccionado == "Una vez") {
                        OutlinedButton(onClick = { datePickerDialog.show() }, modifier = Modifier.weight(1f)) {
                            Text(formatearTimestamp(fecha, "dd/MM/yyyy"))
                        }
                        Spacer(Modifier.width(8.dp))
                    }
                    OutlinedButton(onClick = { timePickerDialog.show() }, modifier = Modifier.weight(1f)) {
                        Text(String.format("%02d:%02d", hora.first, hora.second))
                    }
                }
            }
        },
        confirmButton = {
            Button(onClick = {
                val cal = Calendar.getInstance()
                if (tipoSeleccionado == "Una vez") {
                    cal.timeInMillis = fecha
                }
                cal.set(Calendar.HOUR_OF_DAY, hora.first)
                cal.set(Calendar.MINUTE, hora.second)
                cal.set(Calendar.SECOND, 0)

                if (cal.timeInMillis < System.currentTimeMillis() && tipoSeleccionado != "Una vez") {
                    cal.add(Calendar.DAY_OF_YEAR, 1)
                }

                if (cal.timeInMillis < System.currentTimeMillis() && tipoSeleccionado == "Una vez") {
                    Toast.makeText(context, "Elige una fecha y hora futuras", Toast.LENGTH_SHORT).show()
                    return@Button
                }

                val tipoFinal = when (tipoSeleccionado) {
                    "Diario" -> "DIARIO"
                    "Cada X días" -> "CADA_X_DIAS"
                    else -> "UNICA_VEZ"
                }
                val intervaloFinal = if (tipoFinal == "CADA_X_DIAS") intervalo.toIntOrNull() else null
                val horaFinal = hora.first * 60 + hora.second

                onConfirm(tipoFinal, intervaloFinal, horaFinal, cal.timeInMillis)
            }) {
                Text("Guardar")
            }
        },
        dismissButton = {
            Row {
                if(tipoInicial != null){
                    TextButton(onClick = onCancel) {
                        Text("Eliminar", color = MaterialTheme.colorScheme.error)
                    }
                }
                Spacer(modifier = Modifier.weight(1f))
                TextButton(onClick = onDismiss) { Text("Cancelar") }
            }
        }
    )
}