package com.example.controldecolonias

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.media.RingtoneManager // Import necesario para el sonido
import android.os.Build
import androidx.core.app.NotificationCompat
import com.example.controldecolonias.util.AlarmScheduler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.Calendar

class AlarmReceiver : BroadcastReceiver() {
    override fun onReceive(context: Context, intent: Intent) {
        val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        val channelId = "colonia_reminders"

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            // Aseguramos que la importancia sea ALTA para que suene y aparezca
            val channel = NotificationChannel(channelId, "Recordatorios", NotificationManager.IMPORTANCE_HIGH).apply {
                description = "Canal para recordatorios de colonias y tareas"
                // Opcional: Puedes habilitar luces y vibración por defecto en el canal
                enableLights(true)
                enableVibration(true)
            }
            notificationManager.createNotificationChannel(channel)
        }

        val coloniaId = intent.getStringExtra("EXTRA_COLONIA_ID")
        val tareaId = intent.getIntExtra("EXTRA_TAREA_ID", -1)

        val pendingResult = goAsync()

        if (coloniaId != null) {
            handleColoniaNotification(context, intent, notificationManager, channelId, pendingResult)
        } else if (tareaId != -1) {
            handleTareaNotification(context, intent, notificationManager, channelId, pendingResult)
        } else {
            // Si no hay ID de colonia ni tarea, finalizamos para evitar errores
            pendingResult.finish()
        }
    }

    private fun handleColoniaNotification(context: Context, intent: Intent, notificationManager: NotificationManager, channelId: String, pendingResult: PendingResult) {
        val coloniaId = intent.getStringExtra("EXTRA_COLONIA_ID") ?: return
        val coloniaNombre = intent.getStringExtra("EXTRA_COLONIA_NOMBRE") ?: "una colonia"

        val openAppIntent = Intent(context, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
            putExtra("NAVIGATE_TO_COLONIA_ID", coloniaId)
        }
        val pendingIntent = PendingIntent.getActivity(context, "colonia_$coloniaId".hashCode(), openAppIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE)

        // --- SONIDO Y VIBRACIÓN AÑADIDOS ---
        val defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION) // Obtiene el sonido de notificación predeterminado
        val vibrationPattern = longArrayOf(0, 500, 250, 500) // Patrón de vibración: espera, vibra, espera, vibra

        val notification = NotificationCompat.Builder(context, channelId)
            .setSmallIcon(R.drawable.ic_launcher_foreground) // Asegúrate de tener este ícono o cámbialo
            .setContentTitle("Recordatorio de Colonia")
            .setContentText("¡Es hora de revisar a '$coloniaNombre'!")
            .setPriority(NotificationCompat.PRIORITY_HIGH) // Prioridad alta para que aparezca
            .setAutoCancel(true)
            .setContentIntent(pendingIntent)
            .setSound(defaultSoundUri) // <-- AÑADIDO: Establece el sonido
            .setVibrate(vibrationPattern) // <-- AÑADIDO: Establece la vibración
            .build()

        notificationManager.notify("colonia_$coloniaId".hashCode(), notification)

        CoroutineScope(Dispatchers.IO).launch {
            try {
                val dao = (context.applicationContext as ColoniasApplication).database.coloniaDao()
                val colonia = dao.obtenerColoniaPorIdSync(coloniaId)
                colonia?.let {
                    var proximaFecha: Long? = null
                    when (it.recordatorioTipo) {
                        "DIARIO" -> {
                            val cal = Calendar.getInstance().apply { timeInMillis = it.recordatorioProximaFecha ?: System.currentTimeMillis(); add(Calendar.DAY_OF_YEAR, 1) }
                            proximaFecha = cal.timeInMillis
                        }
                        "CADA_X_DIAS" -> {
                            val interval = it.recordatorioIntervalo ?: 1 // Usar 1 si el intervalo es nulo
                            val cal = Calendar.getInstance().apply { timeInMillis = it.recordatorioProximaFecha ?: System.currentTimeMillis(); add(Calendar.DAY_OF_YEAR, interval) }
                            proximaFecha = cal.timeInMillis
                        }
                        // "UNICA_VEZ" no necesita reprogramación, proximaFecha queda null
                    }

                    // Solo actualizamos si realmente hubo un cambio o si era recurrente
                    if (it.recordatorioTipo != "UNICA_VEZ" || proximaFecha != it.recordatorioProximaFecha) {
                        val coloniaActualizada = it.copy(recordatorioProximaFecha = proximaFecha)
                        dao.actualizar(coloniaActualizada)

                        // Volvemos a programar solo si hay una próxima fecha
                        if (proximaFecha != null) {
                            AlarmScheduler.schedule(context, coloniaActualizada)
                        }
                    } else if (it.recordatorioTipo == "UNICA_VEZ") {
                        // Si era de una sola vez, limpiamos los datos del recordatorio
                        val coloniaActualizada = it.copy(
                            recordatorioProximaFecha = null,
                            recordatorioTipo = null,
                            recordatorioIntervalo = null,
                            recordatorioHora = null
                        )
                        dao.actualizar(coloniaActualizada)
                    }
                }
            } finally {
                pendingResult.finish()
            }
        }
    }

    private fun handleTareaNotification(context: Context, intent: Intent, notificationManager: NotificationManager, channelId: String, pendingResult: PendingResult) {
        val tareaId = intent.getIntExtra("EXTRA_TAREA_ID", -1)
        if (tareaId == -1) return
        val tareaTexto = intent.getStringExtra("EXTRA_TAREA_TEXTO") ?: "una tarea"

        val openAppIntent = Intent(context, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
            putExtra("NAVIGATE_TO_TAREAS", true)
        }
        val pendingIntent = PendingIntent.getActivity(context, "tarea_$tareaId".hashCode(), openAppIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE)

        // --- SONIDO Y VIBRACIÓN AÑADIDOS ---
        val defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION)
        val vibrationPattern = longArrayOf(0, 500, 250, 500)

        val notification = NotificationCompat.Builder(context, channelId)
            .setSmallIcon(R.drawable.ic_launcher_foreground) // Asegúrate de tener este ícono o cámbialo
            .setContentTitle("Recordatorio de Tarea")
            .setContentText(tareaTexto)
            .setPriority(NotificationCompat.PRIORITY_HIGH)
            .setAutoCancel(true)
            .setContentIntent(pendingIntent)
            .setSound(defaultSoundUri) // <-- AÑADIDO: Establece el sonido
            .setVibrate(vibrationPattern) // <-- AÑADIDO: Establece la vibración
            .build()

        notificationManager.notify("tarea_$tareaId".hashCode(), notification)

        // --- Lógica para reprogramar TAREAS ---
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val dao = (context.applicationContext as ColoniasApplication).database.coloniaDao()
                // Necesitamos una función en el DAO para obtener una tarea por ID
                // val tarea = dao.obtenerTareaPorIdSync(tareaId) // Asumiendo que existe esta función
                // tarea?.let {
                //    // Lógica similar a la de colonias para reprogramar
                //    var proximaFecha: Long? = null
                //    when (it.recordatorioTipo) {
                //        "DIARIO" -> { ... }
                //        "CADA_X_DIAS" -> { ... }
                //    }
                //    val tareaActualizada = it.copy(recordatorioProximaFecha = proximaFecha)
                //    dao.actualizarTarea(tareaActualizada)
                //    if (proximaFecha != null) {
                //        AlarmScheduler.schedule(context, tareaActualizada)
                //    } else if (it.recordatorioTipo == "UNICA_VEZ") {
                //         // Limpiar datos si era única vez
                //    }
                // }
            } finally {
                pendingResult.finish()
            }
        }
        // Borra esta línea si implementas la reprogramación de tareas:
        // pendingResult.finish() // Asegúrate de llamar a finish() dentro del bloque finally de la corutina
    }
}