package com.etesync.syncadapter.ui

import android.accounts.Account
import android.content.Context
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.AdapterView
import android.widget.ArrayAdapter
import android.widget.TextView
import androidx.appcompat.app.AlertDialog
import androidx.fragment.app.ListFragment
import com.etesync.syncadapter.*
import com.etesync.journalmanager.JournalManager
import com.etesync.syncadapter.model.CollectionInfo
import com.etesync.syncadapter.model.JournalEntity
import com.etesync.syncadapter.model.JournalModel
import com.etesync.syncadapter.model.MyEntityDataStore
import okhttp3.HttpUrl.Companion.toHttpUrlOrNull
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class CollectionMembersListFragment : ListFragment(), AdapterView.OnItemClickListener, Refreshable {
    private lateinit var data: MyEntityDataStore
    private lateinit var account: Account
    private lateinit var info: CollectionInfo
    private lateinit var journalEntity: JournalEntity
    private var members: List<JournalManager.Member>? = null
    private var asyncTask: Job? = null

    private var emptyTextView: TextView? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        data = (requireContext().applicationContext as App).data
        account = requireArguments().getParcelable(Constants.KEY_ACCOUNT)!!
        info = requireArguments().getSerializable(Constants.KEY_COLLECTION_INFO) as CollectionInfo
        journalEntity = JournalModel.Journal.fetch(data, info.getServiceEntity(data), info.uid)
    }

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
        val view = inflater.inflate(R.layout.collection_members_list, container, false)

        //This is instead of setEmptyText() function because of Google bug
        //See: https://code.google.com/p/android/issues/detail?id=21742
        emptyTextView = view.findViewById<View>(android.R.id.empty) as TextView
        return view
    }

    override fun refresh() {
        asyncTask = lifecycleScope.launch {
            try {
                val membersList = withContext(Dispatchers.IO) {
                    val settings = AccountSettings(requireContext(), account)
                    val httpClient = HttpClient.Builder(context, settings).build().okHttpClient
                    val journalsManager = JournalManager(httpClient, settings.uri?.toHttpUrlOrNull()!!)

                    val journal = JournalManager.Journal.fakeWithUid(journalEntity.uid)
                    journalsManager.listMembers(journal)
                }
                members = membersList
                setListAdapterMembers(membersList)
            } catch (e: Exception) {
                emptyTextView!!.text = e.localizedMessage
            }
        }
    }

    private fun setListAdapterMembers(members: List<JournalManager.Member>) {
        val context = context
        if (context != null) {
            val listAdapter = MembersListAdapter(context)
            setListAdapter(listAdapter)

            listAdapter.addAll(members)

            emptyTextView!!.setText(R.string.collection_members_list_empty)
        }
    }

    override fun onResume() {
        super.onResume()
        if (members != null) {
            setListAdapterMembers(members!!)
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        refresh()

        listView.onItemClickListener = this
    }

    override fun onDestroyView() {
        super.onDestroyView()
        if (asyncTask != null)
            asyncTask!!.cancel()
    }

    override fun onItemClick(parent: AdapterView<*>, view: View, position: Int, id: Long) {
        val member = listAdapter?.getItem(position) as JournalManager.Member

        AlertDialog.Builder(requireActivity())
                .setIcon(R.drawable.ic_info_dark)
                .setTitle(R.string.collection_members_remove_title)
                .setMessage(getString(R.string.collection_members_remove, member.user))
                .setPositiveButton(android.R.string.yes) { dialog, which ->
                    val frag = RemoveMemberFragment.newInstance(account, info, member.user!!)
                    frag.show(requireFragmentManager(), null)
                }
                .setNegativeButton(android.R.string.no) { dialog, which -> }.show()
    }

    internal inner class MembersListAdapter(context: Context) : ArrayAdapter<JournalManager.Member>(context, R.layout.collection_members_list_item) {

        override fun getView(position: Int, _v: View?, parent: ViewGroup): View {
            var v = _v
            if (v == null)
                v = LayoutInflater.from(context).inflate(R.layout.collection_members_list_item, parent, false)

            val member = getItem(position)

            val tv = v!!.findViewById<View>(R.id.title) as TextView
            tv.text = member!!.user

            val readOnly = v.findViewById<View>(R.id.read_only)
            readOnly.visibility = if (member.readOnly) View.VISIBLE else View.GONE

            return v
        }
    }

    companion object {

        fun newInstance(account: Account, info: CollectionInfo): CollectionMembersListFragment {
            val frag = CollectionMembersListFragment()
            val args = Bundle(1)
            args.putParcelable(Constants.KEY_ACCOUNT, account)
            args.putSerializable(Constants.KEY_COLLECTION_INFO, info)
            frag.arguments = args
            return frag
        }
    }
}
