/*
 *
 *   Copyright 2023 Einstein Blanco
 *
 *   Licensed under the GNU General Public License v3.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       https://www.gnu.org/licenses/gpl-3.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
package com.eblan.launcher.feature.home

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.eblan.launcher.domain.framework.AppWidgetHostWrapper
import com.eblan.launcher.domain.framework.FileManager
import com.eblan.launcher.domain.framework.PackageManagerWrapper
import com.eblan.launcher.domain.model.FolderDataById
import com.eblan.launcher.domain.model.GridItem
import com.eblan.launcher.domain.model.GridItemCache
import com.eblan.launcher.domain.model.GridItemData
import com.eblan.launcher.domain.model.GridItemData.ShortcutInfo
import com.eblan.launcher.domain.model.MoveGridItemResult
import com.eblan.launcher.domain.model.PageItem
import com.eblan.launcher.domain.model.PinItemRequestType
import com.eblan.launcher.domain.repository.EblanAppWidgetProviderInfoRepository
import com.eblan.launcher.domain.repository.FolderGridCacheRepository
import com.eblan.launcher.domain.repository.GridCacheRepository
import com.eblan.launcher.domain.usecase.GetHomeDataUseCase
import com.eblan.launcher.domain.usecase.applicationcomponent.GetEblanAppWidgetProviderInfosByLabelUseCase
import com.eblan.launcher.domain.usecase.applicationcomponent.GetEblanApplicationComponentUseCase
import com.eblan.launcher.domain.usecase.applicationcomponent.GetEblanApplicationInfosByLabelUseCase
import com.eblan.launcher.domain.usecase.applicationcomponent.GetEblanShortcutConfigByLabelUseCase
import com.eblan.launcher.domain.usecase.applicationcomponent.GetEblanShortcutInfosUseCase
import com.eblan.launcher.domain.usecase.grid.DeleteGridItemUseCase
import com.eblan.launcher.domain.usecase.grid.GetFolderDataByIdUseCase
import com.eblan.launcher.domain.usecase.grid.GetGridItemsCacheUseCase
import com.eblan.launcher.domain.usecase.grid.MoveFolderGridItemUseCase
import com.eblan.launcher.domain.usecase.grid.MoveGridItemOutsideFolderUseCase
import com.eblan.launcher.domain.usecase.grid.MoveGridItemUseCase
import com.eblan.launcher.domain.usecase.grid.ResizeGridItemUseCase
import com.eblan.launcher.domain.usecase.grid.UpdateGridItemsAfterMoveUseCase
import com.eblan.launcher.domain.usecase.grid.UpdateGridItemsAfterResizeUseCase
import com.eblan.launcher.domain.usecase.grid.UpdateGridItemsUseCase
import com.eblan.launcher.domain.usecase.iconpack.GetIconPackFilePathsUseCase
import com.eblan.launcher.domain.usecase.page.CachePageItemsUseCase
import com.eblan.launcher.domain.usecase.page.UpdatePageItemsUseCase
import com.eblan.launcher.domain.usecase.pin.GetPinGridItemUseCase
import com.eblan.launcher.feature.home.model.EblanApplicationComponentUiState
import com.eblan.launcher.feature.home.model.HomeUiState
import com.eblan.launcher.feature.home.model.Screen
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import javax.inject.Inject

@HiltViewModel
internal class HomeViewModel @Inject constructor(
    getHomeDataUseCase: GetHomeDataUseCase,
    private val gridCacheRepository: GridCacheRepository,
    private val folderGridCacheRepository: FolderGridCacheRepository,
    private val moveGridItemUseCase: MoveGridItemUseCase,
    private val resizeGridItemUseCase: ResizeGridItemUseCase,
    getEblanApplicationComponentUseCase: GetEblanApplicationComponentUseCase,
    private val cachePageItemsUseCase: CachePageItemsUseCase,
    private val updatePageItemsUseCase: UpdatePageItemsUseCase,
    private val appWidgetHostWrapper: AppWidgetHostWrapper,
    private val updateGridItemsAfterResizeUseCase: UpdateGridItemsAfterResizeUseCase,
    private val updateGridItemsAfterMoveUseCase: UpdateGridItemsAfterMoveUseCase,
    private val updateGridItemsUseCase: UpdateGridItemsUseCase,
    private val moveFolderGridItemUseCase: MoveFolderGridItemUseCase,
    private val getFolderDataByIdUseCase: GetFolderDataByIdUseCase,
    getEblanApplicationInfosByLabelUseCase: GetEblanApplicationInfosByLabelUseCase,
    getEblanAppWidgetProviderInfosByLabelUseCase: GetEblanAppWidgetProviderInfosByLabelUseCase,
    getGridItemsCacheUseCase: GetGridItemsCacheUseCase,
    private val deleteGridItemUseCase: DeleteGridItemUseCase,
    private val getPinGridItemUseCase: GetPinGridItemUseCase,
    private val fileManager: FileManager,
    private val packageManagerWrapper: PackageManagerWrapper,
    getEblanShortcutConfigByLabelUseCase: GetEblanShortcutConfigByLabelUseCase,
    getEblanShortcutInfosUseCase: GetEblanShortcutInfosUseCase,
    eblanAppWidgetProviderInfoRepository: EblanAppWidgetProviderInfoRepository,
    getIconPackFilePathsUseCase: GetIconPackFilePathsUseCase,
    private val moveGridItemOutsideFolderUseCase: MoveGridItemOutsideFolderUseCase,
) : ViewModel() {
    val homeUiState = getHomeDataUseCase().map(HomeUiState::Success).stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5_000),
        initialValue = HomeUiState.Loading,
    )

    val eblanApplicationComponentUiState =
        getEblanApplicationComponentUseCase().map(EblanApplicationComponentUiState::Success)
            .stateIn(
                scope = viewModelScope,
                started = SharingStarted.WhileSubscribed(5_000),
                initialValue = EblanApplicationComponentUiState.Loading,
            )

    private val _screen = MutableStateFlow(Screen.Pager)

    val screen = _screen.asStateFlow()

    private val _moveGridItemResult = MutableStateFlow<MoveGridItemResult?>(null)

    val movedGridItemResult = _moveGridItemResult.asStateFlow()

    private val defaultDelay = 500L

    private val _pageItems = MutableStateFlow(emptyList<PageItem>())

    val pageItems = _pageItems.asStateFlow()

    private var moveGridItemJob: Job? = null

    private val _foldersDataById = MutableStateFlow(ArrayDeque<FolderDataById>())

    val foldersDataById = _foldersDataById.asStateFlow()

    private val _eblanApplicationLabel = MutableStateFlow<String?>(null)

    @OptIn(ExperimentalCoroutinesApi::class, FlowPreview::class)
    val eblanApplicationInfosByLabel =
        _eblanApplicationLabel.filterNotNull().debounce(defaultDelay).flatMapLatest { label ->
            getEblanApplicationInfosByLabelUseCase(label = label)
        }.stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5_000),
            initialValue = emptyList(),
        )

    private val _eblanAppWidgetProviderInfoLabel = MutableStateFlow<String?>(null)

    @OptIn(ExperimentalCoroutinesApi::class, FlowPreview::class)
    val eblanAppWidgetProviderInfosByLabel =
        _eblanAppWidgetProviderInfoLabel.filterNotNull().debounce(defaultDelay)
            .flatMapLatest { label ->
                getEblanAppWidgetProviderInfosByLabelUseCase(label = label)
            }.stateIn(
                scope = viewModelScope,
                started = SharingStarted.WhileSubscribed(5_000),
                initialValue = emptyMap(),
            )

    private val _eblanShortcutConfigsLabel = MutableStateFlow<String?>(null)

    @OptIn(ExperimentalCoroutinesApi::class, FlowPreview::class)
    val eblanShortcutConfigsByLabel =
        _eblanShortcutConfigsLabel.filterNotNull().debounce(defaultDelay)
            .flatMapLatest { label ->
                getEblanShortcutConfigByLabelUseCase(label = label)
            }.stateIn(
                scope = viewModelScope,
                started = SharingStarted.WhileSubscribed(5_000),
                initialValue = emptyMap(),
            )

    val gridItemsCache = getGridItemsCacheUseCase().stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5_000),
        initialValue = GridItemCache(
            gridItemsCacheByPage = emptyMap(),
            dockGridItemsCache = emptyList(),
            folderGridItemsCacheByPage = emptyMap(),
        ),
    )

    private val _pinGridItem = MutableStateFlow<GridItem?>(null)

    val pinGridItem = _pinGridItem.asStateFlow()

    val eblanShortcutInfos = getEblanShortcutInfosUseCase().stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(5_000),
        initialValue = emptyMap(),
    )

    val eblanAppWidgetProviderInfos =
        eblanAppWidgetProviderInfoRepository.eblanAppWidgetProviderInfos.map { eblanAppWidgetProviderInfos ->
            eblanAppWidgetProviderInfos.groupBy { eblanAppWidgetProviderInfo ->
                eblanAppWidgetProviderInfo.packageName
            }
        }.stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5_000),
            initialValue = emptyMap(),
        )

    val iconPackFilePaths = getIconPackFilePathsUseCase()
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5_000),
            initialValue = emptyMap(),
        )

    fun moveGridItem(
        movingGridItem: GridItem,
        x: Int,
        y: Int,
        columns: Int,
        rows: Int,
        gridWidth: Int,
        gridHeight: Int,
        lockMovement: Boolean,
    ) {
        moveGridItemJob?.cancel()

        moveGridItemJob = viewModelScope.launch {
            _moveGridItemResult.update {
                moveGridItemUseCase(
                    movingGridItem = movingGridItem,
                    x = x,
                    y = y,
                    columns = columns,
                    rows = rows,
                    gridWidth = gridWidth,
                    gridHeight = gridHeight,
                    lockMovement = lockMovement,
                )
            }
        }
    }

    fun resizeGridItem(
        resizingGridItem: GridItem,
        columns: Int,
        rows: Int,
        lockMovement: Boolean,
    ) {
        moveGridItemJob?.cancel()

        moveGridItemJob = viewModelScope.launch {
            _moveGridItemResult.update {
                resizeGridItemUseCase(
                    resizingGridItem = resizingGridItem,
                    columns = columns,
                    rows = rows,
                    lockMovement = lockMovement,
                )
            }
        }
    }

    fun moveFolderGridItem(
        movingGridItem: GridItem,
        x: Int,
        y: Int,
        columns: Int,
        rows: Int,
        gridWidth: Int,
        gridHeight: Int,
        lockMovement: Boolean,
    ) {
        moveGridItemJob?.cancel()

        moveGridItemJob = viewModelScope.launch {
            _moveGridItemResult.update {
                moveFolderGridItemUseCase(
                    movingGridItem = movingGridItem,
                    x = x,
                    y = y,
                    columns = columns,
                    rows = rows,
                    gridWidth = gridWidth,
                    gridHeight = gridHeight,
                    lockMovement = lockMovement,
                )
            }
        }
    }

    fun showGridCache(
        gridItems: List<GridItem>,
        screen: Screen,
    ) {
        viewModelScope.launch {
            gridCacheRepository.insertGridItems(gridItems = gridItems)

            delay(defaultDelay)

            _screen.update {
                screen
            }
        }
    }

    fun showFolderGridCache(
        gridItems: List<GridItem>,
        screen: Screen,
    ) {
        viewModelScope.launch {
            folderGridCacheRepository.insertGridItems(gridItems = gridItems)

            delay(defaultDelay)

            _screen.update {
                screen
            }
        }
    }

    fun showPageCache(gridItems: List<GridItem>) {
        viewModelScope.launch {
            _screen.update {
                Screen.Loading
            }

            _pageItems.update {
                cachePageItemsUseCase(gridItems = gridItems)
            }

            delay(defaultDelay)

            _screen.update {
                Screen.EditPage
            }
        }
    }

    fun saveEditPage(
        id: Int,
        pageItems: List<PageItem>,
        pageItemsToDelete: List<PageItem>,
    ) {
        viewModelScope.launch {
            _screen.update {
                Screen.Loading
            }

            updatePageItemsUseCase(
                id = id,
                pageItems = pageItems,
                pageItemsToDelete = pageItemsToDelete,
            )

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }
        }
    }

    fun updateScreen(screen: Screen) {
        _screen.update {
            screen
        }
    }

    fun resetGridCacheAfterResize(resizingGridItem: GridItem) {
        viewModelScope.launch {
            updateGridItemsAfterResizeUseCase(resizingGridItem = resizingGridItem)

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }

            _moveGridItemResult.update {
                null
            }
        }
    }

    fun resetGridCacheAfterMove(moveGridItemResult: MoveGridItemResult) {
        viewModelScope.launch {
            moveGridItemJob?.cancelAndJoin()

            updateGridItemsAfterMoveUseCase(moveGridItemResult = moveGridItemResult)

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }

            _moveGridItemResult.update {
                null
            }
        }
    }

    fun resetGridCacheAfterMoveFolder() {
        viewModelScope.launch {
            val lastFolderId = _foldersDataById.value.last().folderId

            updateGridItemsUseCase(gridItems = folderGridCacheRepository.gridItemsCache.first())

            getFolderDataByIdUseCase(folderId = lastFolderId)?.let { folder ->
                _foldersDataById.update { currentFolders ->
                    ArrayDeque(currentFolders).apply {
                        val index = indexOfFirst { it.folderId == lastFolderId }

                        set(index, folder)
                    }
                }

                delay(defaultDelay)

                _screen.update {
                    Screen.Folder
                }
            }

            _moveGridItemResult.update {
                null
            }
        }
    }

    fun cancelGridCache() {
        viewModelScope.launch {
            moveGridItemJob?.cancelAndJoin()

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }

            _moveGridItemResult.update {
                null
            }
        }
    }

    fun cancelFolderDragGridCache() {
        viewModelScope.launch {
            moveGridItemJob?.cancelAndJoin()

            val lastFolderId = _foldersDataById.value.last().folderId

            getFolderDataByIdUseCase(folderId = lastFolderId)?.let { folder ->
                _foldersDataById.update { currentFolders ->
                    ArrayDeque(currentFolders).apply {
                        val index = indexOfFirst { it.folderId == lastFolderId }

                        set(index, folder)
                    }
                }

                delay(defaultDelay)

                _screen.update {
                    Screen.Folder
                }
            }

            _moveGridItemResult.update {
                null
            }
        }
    }

    fun updateGridItemDataCache(gridItem: GridItem) {
        viewModelScope.launch {
            gridCacheRepository.updateGridItemData(
                id = gridItem.id,
                data = gridItem.data,
            )
        }
    }

    fun updateShortcutConfigGridItemDataCache(
        byteArray: ByteArray?,
        moveGridItemResult: MoveGridItemResult,
        gridItem: GridItem,
        data: GridItemData.ShortcutConfig,
    ) {
        viewModelScope.launch {
            val shortcutIntentIcon = byteArray?.let { currentByteArray ->
                fileManager.updateAndGetFilePath(
                    fileManager.getFilesDirectory(FileManager.SHORTCUT_INTENT_ICONS_DIR),
                    gridItem.id,
                    currentByteArray,
                )
            }

            gridCacheRepository.updateGridItemData(
                id = gridItem.id,
                data = data.copy(shortcutIntentIcon = shortcutIntentIcon),
            )

            resetGridCacheAfterMove(moveGridItemResult = moveGridItemResult)
        }
    }

    fun deleteGridItemCache(gridItem: GridItem) {
        viewModelScope.launch {
            gridCacheRepository.deleteGridItem(gridItem = gridItem)

            updateGridItemsUseCase(gridItems = gridCacheRepository.gridItemsCache.first())

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }
        }
    }

    fun deleteWidgetGridItemCache(
        gridItem: GridItem,
        appWidgetId: Int,
    ) {
        viewModelScope.launch {
            appWidgetHostWrapper.deleteAppWidgetId(appWidgetId = appWidgetId)

            gridCacheRepository.deleteGridItem(gridItem = gridItem)

            updateGridItemsUseCase(gridItems = gridCacheRepository.gridItemsCache.first())

            delay(defaultDelay)

            _screen.update {
                Screen.Pager
            }
        }
    }

    fun showFolder(folderId: String) {
        viewModelScope.launch {
            getFolderDataByIdUseCase(folderId = folderId)?.let { folder ->
                _foldersDataById.update { currentFolders ->
                    ArrayDeque(currentFolders).apply {
                        clear()

                        add(folder)
                    }
                }

                _screen.update {
                    Screen.Folder
                }
            }
        }
    }

    fun addFolder(folderId: String) {
        viewModelScope.launch {
            getFolderDataByIdUseCase(folderId = folderId)?.let { folder ->
                _foldersDataById.update { currentFolders ->
                    ArrayDeque(currentFolders).apply {
                        add(folder)
                    }
                }
            }
        }
    }

    fun removeLastFolder() {
        _foldersDataById.update { currentFolders ->
            ArrayDeque(currentFolders).apply {
                removeLast()
            }
        }
    }

    fun getEblanApplicationInfosByLabel(label: String) {
        _eblanApplicationLabel.update {
            label
        }
    }

    fun getEblanAppWidgetProviderInfosByLabel(label: String) {
        _eblanAppWidgetProviderInfoLabel.update {
            label
        }
    }

    fun getEblanShortcutConfigsByLabel(label: String) {
        _eblanShortcutConfigsLabel.update {
            label
        }
    }

    fun deleteGridItem(gridItem: GridItem) {
        viewModelScope.launch {
            deleteGridItemUseCase(gridItem = gridItem)
        }
    }

    fun getPinGridItem(pinItemRequestType: PinItemRequestType) {
        viewModelScope.launch {
            _pinGridItem.update {
                getPinGridItemUseCase(pinItemRequestType = pinItemRequestType)
            }
        }
    }

    fun resetPinGridItem() {
        _pinGridItem.update {
            null
        }
    }

    fun updateShortcutConfigIntoShortcutInfoGridItem(
        moveGridItemResult: MoveGridItemResult,
        pinItemRequestType: PinItemRequestType.ShortcutInfo,
    ) {
        viewModelScope.launch {
            gridCacheRepository.deleteGridItem(gridItem = moveGridItemResult.movingGridItem)

            val icon = pinItemRequestType.icon?.let { byteArray ->
                fileManager.updateAndGetFilePath(
                    directory = fileManager.getFilesDirectory(FileManager.SHORTCUTS_DIR),
                    name = pinItemRequestType.shortcutId,
                    byteArray = byteArray,
                )
            }

            val eblanApplicationInfoIcon =
                packageManagerWrapper.getApplicationIcon(packageName = pinItemRequestType.packageName)
                    ?.let { byteArray ->
                        fileManager.updateAndGetFilePath(
                            directory = fileManager.getFilesDirectory(FileManager.ICONS_DIR),
                            name = pinItemRequestType.packageName,
                            byteArray = byteArray,
                        )
                    }

            val data = ShortcutInfo(
                shortcutId = pinItemRequestType.shortcutId,
                packageName = pinItemRequestType.packageName,
                serialNumber = pinItemRequestType.serialNumber,
                shortLabel = pinItemRequestType.shortLabel,
                longLabel = pinItemRequestType.longLabel,
                icon = icon,
                isEnabled = pinItemRequestType.isEnabled,
                eblanApplicationInfoIcon = eblanApplicationInfoIcon,
                customIcon = null,
                customShortLabel = null,
            )

            gridCacheRepository.insertGridItem(
                gridItem = moveGridItemResult.movingGridItem.copy(
                    data = data,
                ),
            )

            resetGridCacheAfterMove(moveGridItemResult = moveGridItemResult)
        }
    }

    fun moveGridItemOutsideFolder(
        folderId: String,
        movingGridItem: GridItem,
        gridItems: List<GridItem>,
        screen: Screen,
    ) {
        viewModelScope.launch {
            moveGridItemOutsideFolderUseCase(
                folderId = folderId,
                movingGridItem = movingGridItem,
                gridItems = gridItems,
            )

            delay(defaultDelay)

            _screen.update {
                screen
            }
        }
    }

    fun showFolderWhenDragging(folderId: String) {
        viewModelScope.launch {
            getFolderDataByIdUseCase(folderId = folderId)?.let { folder ->
                _foldersDataById.update { currentFolders ->
                    ArrayDeque(currentFolders).apply {
                        clear()

                        add(folder)
                    }
                }

                showFolderGridCache(
                    gridItems = folder.gridItems,
                    screen = Screen.FolderDrag,
                )
            }
        }
    }
}
