/*
 *
 *   Copyright 2023 Einstein Blanco
 *
 *   Licensed under the GNU General Public License v3.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       https://www.gnu.org/licenses/gpl-3.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 *
 */
package com.eblan.launcher.data.datastore.mapper

import com.eblan.launcher.data.datastore.proto.appdrawer.AppDrawerSettingsProto
import com.eblan.launcher.data.datastore.proto.experimental.ExperimentalSettingsProto
import com.eblan.launcher.data.datastore.proto.general.GeneralSettingsProto
import com.eblan.launcher.data.datastore.proto.general.ThemeProto
import com.eblan.launcher.data.datastore.proto.gesture.EblanActionProto
import com.eblan.launcher.data.datastore.proto.gesture.GestureSettingsProto
import com.eblan.launcher.data.datastore.proto.gesture.LockScreenProto
import com.eblan.launcher.data.datastore.proto.gesture.NoneProto
import com.eblan.launcher.data.datastore.proto.gesture.OpenAppDrawerProto
import com.eblan.launcher.data.datastore.proto.gesture.OpenAppProto
import com.eblan.launcher.data.datastore.proto.gesture.OpenNotificationPanelProto
import com.eblan.launcher.data.datastore.proto.gesture.OpenQuickSettingsProto
import com.eblan.launcher.data.datastore.proto.gesture.OpenRecentsProto
import com.eblan.launcher.data.datastore.proto.home.GridItemSettingsProto
import com.eblan.launcher.data.datastore.proto.home.HomeSettingsProto
import com.eblan.launcher.data.datastore.proto.home.HorizontalAlignmentProto
import com.eblan.launcher.data.datastore.proto.home.TextColorProto
import com.eblan.launcher.data.datastore.proto.home.VerticalArrangementProto
import com.eblan.launcher.domain.model.AppDrawerSettings
import com.eblan.launcher.domain.model.EblanAction
import com.eblan.launcher.domain.model.EblanAction.LockScreen
import com.eblan.launcher.domain.model.EblanAction.None
import com.eblan.launcher.domain.model.EblanAction.OpenApp
import com.eblan.launcher.domain.model.EblanAction.OpenAppDrawer
import com.eblan.launcher.domain.model.EblanAction.OpenNotificationPanel
import com.eblan.launcher.domain.model.EblanAction.OpenQuickSettings
import com.eblan.launcher.domain.model.EblanAction.OpenRecents
import com.eblan.launcher.domain.model.ExperimentalSettings
import com.eblan.launcher.domain.model.GeneralSettings
import com.eblan.launcher.domain.model.GestureSettings
import com.eblan.launcher.domain.model.GridItemSettings
import com.eblan.launcher.domain.model.HomeSettings
import com.eblan.launcher.domain.model.HorizontalAlignment
import com.eblan.launcher.domain.model.TextColor
import com.eblan.launcher.domain.model.Theme
import com.eblan.launcher.domain.model.VerticalArrangement

internal fun HomeSettingsProto.toHomeSettings(): HomeSettings {
    return HomeSettings(
        columns = columns,
        rows = rows,
        pageCount = pageCount,
        infiniteScroll = infiniteScroll,
        dockColumns = dockColumns,
        dockRows = dockRows,
        dockHeight = dockHeight,
        initialPage = initialPage,
        wallpaperScroll = wallpaperScroll,
        folderColumns = folderColumns,
        folderRows = folderRows,
        gridItemSettings = gridItemSettingsProto.toGridItemSettings(),
        lockScreenOrientation = lockScreenOrientation,
    )
}

internal fun AppDrawerSettingsProto.toAppDrawerSettings(): AppDrawerSettings {
    return AppDrawerSettings(
        appDrawerColumns = appDrawerColumns,
        appDrawerRowsHeight = appDrawerRowsHeight,
        gridItemSettings = gridItemSettingsProto.toGridItemSettings(),
    )
}

internal fun GridItemSettingsProto.toGridItemSettings(): GridItemSettings {
    return GridItemSettings(
        iconSize = iconSize,
        textColor = textColorProto.toTextColor(),
        textSize = textSize,
        showLabel = showLabel,
        singleLineLabel = singleLineLabel,
        horizontalAlignment = horizontalAlignmentProto.toHorizontalAlignment(),
        verticalArrangement = verticalArrangementProto.toVerticalArrangement(),
    )
}

internal fun GeneralSettingsProto.toGeneralSettings(): GeneralSettings {
    return GeneralSettings(
        theme = themeProto.toDarkThemeConfig(),
        dynamicTheme = dynamicTheme,
        iconPackInfoPackageName = iconPackInfoPackageName,
    )
}

internal fun TextColor.toTextColorProto(): TextColorProto {
    return when (this) {
        TextColor.System -> TextColorProto.TextColorSystem
        TextColor.Light -> TextColorProto.TextColorLight
        TextColor.Dark -> TextColorProto.TextColorDark
    }
}

internal fun TextColorProto.toTextColor(): TextColor {
    return when (this) {
        TextColorProto.TextColorSystem, TextColorProto.UNRECOGNIZED -> TextColor.System
        TextColorProto.TextColorLight -> TextColor.Light
        TextColorProto.TextColorDark -> TextColor.Dark
    }
}

internal fun HorizontalAlignment.toHorizontalAlignmentProto(): HorizontalAlignmentProto {
    return when (this) {
        HorizontalAlignment.Start -> HorizontalAlignmentProto.Start
        HorizontalAlignment.CenterHorizontally -> HorizontalAlignmentProto.CenterHorizontally
        HorizontalAlignment.End -> HorizontalAlignmentProto.End
    }
}

internal fun HorizontalAlignmentProto.toHorizontalAlignment(): HorizontalAlignment {
    return when (this) {
        HorizontalAlignmentProto.Start -> HorizontalAlignment.Start
        HorizontalAlignmentProto.CenterHorizontally, HorizontalAlignmentProto.UNRECOGNIZED -> HorizontalAlignment.CenterHorizontally
        HorizontalAlignmentProto.End -> HorizontalAlignment.End
    }
}

internal fun VerticalArrangement.toVerticalArrangementProto(): VerticalArrangementProto {
    return when (this) {
        VerticalArrangement.Top -> VerticalArrangementProto.Top
        VerticalArrangement.Center -> VerticalArrangementProto.Center
        VerticalArrangement.Bottom -> VerticalArrangementProto.Bottom
    }
}

internal fun VerticalArrangementProto.toVerticalArrangement(): VerticalArrangement {
    return when (this) {
        VerticalArrangementProto.Top -> VerticalArrangement.Top
        VerticalArrangementProto.Center, VerticalArrangementProto.UNRECOGNIZED -> VerticalArrangement.Center
        VerticalArrangementProto.Bottom -> VerticalArrangement.Bottom
    }
}

internal fun EblanAction.toEblanActionProto(): EblanActionProto {
    return when (this) {
        None -> EblanActionProto.newBuilder()
            .setNoneProto(NoneProto.getDefaultInstance())
            .build()

        is OpenAppDrawer -> EblanActionProto.newBuilder()
            .setOpenAppDrawerProto(OpenAppDrawerProto.getDefaultInstance())
            .build()

        is OpenNotificationPanel -> EblanActionProto.newBuilder()
            .setOpenNotificationPanelProto(OpenNotificationPanelProto.getDefaultInstance())
            .build()

        is OpenApp -> EblanActionProto.newBuilder()
            .setOpenAppProto(OpenAppProto.newBuilder().setComponentName(componentName))
            .build()

        LockScreen -> EblanActionProto.newBuilder()
            .setLockScreenProto(LockScreenProto.getDefaultInstance())
            .build()

        OpenQuickSettings -> EblanActionProto.newBuilder()
            .setOpenQuickSettingsProto(OpenQuickSettingsProto.getDefaultInstance())
            .build()

        OpenRecents -> EblanActionProto.newBuilder()
            .setOpenRecentsProto(OpenRecentsProto.getDefaultInstance())
            .build()
    }
}

internal fun GestureSettingsProto.toGestureSettings(): GestureSettings {
    return GestureSettings(
        doubleTap = doubleTapProto.toEblanAction(),
        swipeUp = swipeUpProto.toEblanAction(),
        swipeDown = swipeDownProto.toEblanAction(),
    )
}

internal fun EblanActionProto.toEblanAction(): EblanAction {
    return when (typeCase) {
        EblanActionProto.TypeCase.NONEPROTO -> None

        EblanActionProto.TypeCase.OPENAPPDRAWERPROTO -> OpenAppDrawer

        EblanActionProto.TypeCase.OPENNOTIFICATIONPANELPROTO -> OpenNotificationPanel

        EblanActionProto.TypeCase.OPENAPPPROTO ->
            OpenApp(openAppProto.componentName)

        EblanActionProto.TypeCase.LOCKSCREENPROTO -> LockScreen
        EblanActionProto.TypeCase.OPENQUICKSETTINGSPROTO -> OpenQuickSettings
        EblanActionProto.TypeCase.OPENRECENTSPROTO -> OpenRecents

        EblanActionProto.TypeCase.TYPE_NOT_SET, null ->
            error("EblanActionProto type not set")
    }
}

internal fun Theme.toThemeProto(): ThemeProto {
    return when (this) {
        Theme.System -> ThemeProto.DarkThemeConfigSystem
        Theme.Light -> ThemeProto.DarkThemeConfigLight
        Theme.Dark -> ThemeProto.DarkThemeConfigDark
    }
}

internal fun ThemeProto.toDarkThemeConfig(): Theme {
    return when (this) {
        ThemeProto.DarkThemeConfigSystem, ThemeProto.UNRECOGNIZED -> Theme.System
        ThemeProto.DarkThemeConfigLight -> Theme.Light
        ThemeProto.DarkThemeConfigDark -> Theme.Dark
    }
}

internal fun GridItemSettings.toGridItemSettingsProto(): GridItemSettingsProto {
    return GridItemSettingsProto.newBuilder()
        .setIconSize(iconSize)
        .setTextColorProto(textColor.toTextColorProto())
        .setTextSize(textSize)
        .setShowLabel(showLabel)
        .setSingleLineLabel(singleLineLabel)
        .setHorizontalAlignmentProto(horizontalAlignment.toHorizontalAlignmentProto())
        .setVerticalArrangementProto(verticalArrangement.toVerticalArrangementProto())
        .build()
}

internal fun HomeSettings.toHomeSettingsProto(): HomeSettingsProto {
    return HomeSettingsProto.newBuilder()
        .setColumns(columns)
        .setRows(rows)
        .setPageCount(pageCount)
        .setInfiniteScroll(infiniteScroll)
        .setDockColumns(dockColumns)
        .setDockRows(dockRows)
        .setDockHeight(dockHeight)
        .setInitialPage(initialPage)
        .setWallpaperScroll(wallpaperScroll)
        .setFolderColumns(folderColumns)
        .setFolderRows(folderRows)
        .setGridItemSettingsProto(gridItemSettings.toGridItemSettingsProto())
        .setLockScreenOrientation(lockScreenOrientation)
        .build()
}

internal fun AppDrawerSettings.toAppDrawerSettingsProto(): AppDrawerSettingsProto {
    val gridItemSettingsProto = GridItemSettingsProto.newBuilder()
        .setIconSize(gridItemSettings.iconSize)
        .setTextColorProto(gridItemSettings.textColor.toTextColorProto())
        .setTextSize(gridItemSettings.textSize)
        .setShowLabel(gridItemSettings.showLabel)
        .setSingleLineLabel(gridItemSettings.singleLineLabel)
        .setHorizontalAlignmentProto(gridItemSettings.horizontalAlignment.toHorizontalAlignmentProto())
        .setVerticalArrangementProto(gridItemSettings.verticalArrangement.toVerticalArrangementProto())
        .build()

    return AppDrawerSettingsProto.newBuilder()
        .setAppDrawerColumns(appDrawerColumns)
        .setAppDrawerRowsHeight(appDrawerRowsHeight)
        .setGridItemSettingsProto(gridItemSettingsProto)
        .build()
}

internal fun GeneralSettings.toGeneralSettingsProto(): GeneralSettingsProto {
    return GeneralSettingsProto.newBuilder()
        .setThemeProto(theme.toThemeProto())
        .setDynamicTheme(dynamicTheme)
        .setIconPackInfoPackageName(iconPackInfoPackageName)
        .build()
}

internal fun GestureSettings.toGestureSettingsProto(): GestureSettingsProto {
    return GestureSettingsProto.newBuilder()
        .setDoubleTapProto(doubleTap.toEblanActionProto())
        .setSwipeUpProto(swipeUp.toEblanActionProto())
        .setSwipeDownProto(swipeDown.toEblanActionProto())
        .build()
}

internal fun ExperimentalSettings.toExperimentalSettingsProto(): ExperimentalSettingsProto {
    return ExperimentalSettingsProto.newBuilder()
        .setSyncData(syncData)
        .setFirstLaunch(firstLaunch)
        .setLockMovement(lockMovement)
        .build()
}

internal fun ExperimentalSettingsProto.toExperimentalSettings(): ExperimentalSettings {
    return ExperimentalSettings(
        syncData = syncData,
        firstLaunch = firstLaunch,
        lockMovement = lockMovement,
    )
}
