/*
 * Copyright (c) 2022 DuckDuckGo
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.duckduckgo.voice.impl

import android.Manifest
import android.content.Context
import android.content.Intent
import androidx.activity.result.ActivityResultCaller
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts.RequestPermission
import androidx.activity.result.contract.ActivityResultContracts.StartActivityForResult
import com.duckduckgo.di.scopes.ActivityScope
import com.duckduckgo.voice.api.VoiceSearchLauncher.VoiceSearchMode
import com.duckduckgo.voice.impl.ActivityResultLauncherWrapper.Action
import com.duckduckgo.voice.impl.ActivityResultLauncherWrapper.Request
import com.duckduckgo.voice.impl.ActivityResultLauncherWrapper.Request.Permission
import com.duckduckgo.voice.impl.ActivityResultLauncherWrapper.Request.ResultFromVoiceSearch
import com.duckduckgo.voice.impl.listeningmode.VoiceSearchActivity
import com.squareup.anvil.annotations.ContributesBinding
import javax.inject.Inject

interface ActivityResultLauncherWrapper {
    fun register(
        caller: ActivityResultCaller,
        request: Request,
    )

    fun launch(action: Action)

    sealed class Request {
        data class Permission(val onResult: (Boolean) -> Unit) : Request()
        data class ResultFromVoiceSearch(
            val onResult: (Int, String, VoiceSearchMode) -> Unit,
        ) : Request()
    }

    sealed class Action {
        data object LaunchPermissionRequest : Action()
        data class LaunchVoiceSearch(val initialMode: VoiceSearchMode) : Action()
    }
}

@ContributesBinding(ActivityScope::class)
class RealActivityResultLauncherWrapper @Inject constructor(
    private val context: Context,
) : ActivityResultLauncherWrapper {

    private var permissionLauncher: ActivityResultLauncher<String>? = null
    private var voiceSearchActivityLaucher: ActivityResultLauncher<Intent>? = null

    override fun register(
        caller: ActivityResultCaller,
        request: Request,
    ) {
        when (request) {
            is Permission -> registerPermissionRequest(caller, request.onResult)
            is ResultFromVoiceSearch -> registerResultFromVoiceSearch(caller, request.onResult)
        }
    }

    private fun registerResultFromVoiceSearch(
        caller: ActivityResultCaller,
        onResult: (Int, String, VoiceSearchMode) -> Unit,
    ) {
        voiceSearchActivityLaucher = caller.registerForActivityResult(StartActivityForResult()) {
            val resultCode = it.resultCode
            val result = it.data?.getStringExtra(VoiceSearchActivity.EXTRA_VOICE_RESULT) ?: ""
            val mode = it.data?.getIntExtra(VoiceSearchActivity.EXTRA_SELECTED_MODE, 0) ?: 0
            onResult(resultCode, result, VoiceSearchMode.fromValue(mode))
        }
    }

    private fun registerPermissionRequest(
        caller: ActivityResultCaller,
        onResult: (Boolean) -> Unit,
    ) {
        permissionLauncher = caller.registerForActivityResult(RequestPermission()) {
            onResult(it)
        }
    }

    override fun launch(action: Action) {
        when (action) {
            is Action.LaunchPermissionRequest -> launchPermissionRequest()
            is Action.LaunchVoiceSearch -> launchVoiceSearch(action.initialMode)
        }
    }

    private fun launchVoiceSearch(initialMode: VoiceSearchMode) {
        Intent(context, VoiceSearchActivity::class.java).apply {
            putExtra(VoiceSearchActivity.EXTRA_INITIAL_MODE, initialMode.value)
        }.let {
            voiceSearchActivityLaucher?.launch(it)
        }
    }

    private fun launchPermissionRequest() {
        permissionLauncher?.launch(Manifest.permission.RECORD_AUDIO)
    }
}
