/*
 * Copyright (c) 2021 DuckDuckGo
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.duckduckgo.privacy.config.store

import androidx.room.Database
import androidx.room.RoomDatabase
import androidx.room.TypeConverters
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import com.duckduckgo.privacy.config.store.features.amplinks.AmpLinksDao
import com.duckduckgo.privacy.config.store.features.contentblocking.ContentBlockingDao
import com.duckduckgo.privacy.config.store.features.drm.DrmDao
import com.duckduckgo.privacy.config.store.features.gpc.GpcContentScopeConfigDao
import com.duckduckgo.privacy.config.store.features.gpc.GpcExceptionsDao
import com.duckduckgo.privacy.config.store.features.gpc.GpcHeadersDao
import com.duckduckgo.privacy.config.store.features.https.HttpsDao
import com.duckduckgo.privacy.config.store.features.trackerallowlist.TrackerAllowlistDao
import com.duckduckgo.privacy.config.store.features.trackingparameters.TrackingParametersDao
import com.duckduckgo.privacy.config.store.features.unprotectedtemporary.UnprotectedTemporaryDao

@TypeConverters(
    RuleTypeConverter::class,
)
@Database(
    exportSchema = true,
    version = 18,
    entities = [
        TrackerAllowlistEntity::class,
        UnprotectedTemporaryEntity::class,
        HttpsExceptionEntity::class,
        GpcExceptionEntity::class,
        GpcHeaderEnabledSiteEntity::class,
        GpcContentScopeConfigEntity::class,
        ContentBlockingExceptionEntity::class,
        DrmExceptionEntity::class,
        PrivacyConfig::class,
        AmpLinkFormatEntity::class,
        AmpKeywordEntity::class,
        AmpLinkExceptionEntity::class,
        TrackingParameterEntity::class,
        TrackingParameterExceptionEntity::class,
    ],
)
abstract class PrivacyConfigDatabase : RoomDatabase() {
    abstract fun trackerAllowlistDao(): TrackerAllowlistDao
    abstract fun unprotectedTemporaryDao(): UnprotectedTemporaryDao
    abstract fun httpsDao(): HttpsDao
    abstract fun gpcExceptionsDao(): GpcExceptionsDao
    abstract fun gpcHeadersDao(): GpcHeadersDao
    abstract fun gpcContentScopeConfigDao(): GpcContentScopeConfigDao
    abstract fun contentBlockingDao(): ContentBlockingDao
    abstract fun privacyConfigDao(): PrivacyConfigDao
    abstract fun drmDao(): DrmDao
    abstract fun ampLinksDao(): AmpLinksDao
    abstract fun trackingParametersDao(): TrackingParametersDao
}

val MIGRATION_2_3 = object : Migration(2, 3) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DROP TABLE toggles")
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_3_4 = object : Migration(3, 4) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_10_11 = object : Migration(10, 11) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DROP TABLE IF EXISTS `autofill_exceptions`")
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_11_12 = object : Migration(11, 12) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_13_14 = object : Migration(13, 14) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_15_16 = object : Migration(15, 16) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_16_17 = object : Migration(16, 17) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("ALTER TABLE 'privacy_config' ADD COLUMN 'timestamp' TEXT")
        database.execSQL("DELETE FROM privacy_config")
    }
}

val MIGRATION_17_18 = object : Migration(17, 18) {
    override fun migrate(database: SupportSQLiteDatabase) {
        database.execSQL("DROP TABLE IF EXISTS `user_agent_exceptions`")
        database.execSQL("DROP TABLE IF EXISTS `user_agent_sites`")
        database.execSQL("DROP TABLE IF EXISTS `user_agent_states`")
        database.execSQL("DROP TABLE IF EXISTS `user_agent_versions`")
        database.execSQL("DELETE FROM privacy_config")
    }
}

val ALL_MIGRATIONS = arrayOf(
    MIGRATION_2_3,
    MIGRATION_3_4,
    MIGRATION_10_11,
    MIGRATION_11_12,
    MIGRATION_13_14,
    MIGRATION_15_16,
    MIGRATION_16_17,
    MIGRATION_17_18,
)
