/*
 * Copyright (c) 2025 DuckDuckGo
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.duckduckgo.pir.impl.optout

import android.app.Notification
import android.app.Service
import android.content.Intent
import android.os.IBinder
import android.os.Process
import com.duckduckgo.anvil.annotations.InjectWith
import com.duckduckgo.di.scopes.ServiceScope
import com.duckduckgo.pir.impl.R
import com.duckduckgo.pir.impl.checker.PirWorkHandler
import com.duckduckgo.pir.impl.notifications.PirNotificationManager
import dagger.android.AndroidInjection
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import logcat.logcat
import javax.inject.Inject

@InjectWith(scope = ServiceScope::class)
class PirForegroundOptOutService : Service(), CoroutineScope by MainScope() {
    @Inject
    lateinit var pirOptOut: PirOptOut

    @Inject
    lateinit var pirNotificationManager: PirNotificationManager

    @Inject
    lateinit var pirWorkHandler: PirWorkHandler

    override fun onCreate() {
        super.onCreate()
        AndroidInjection.inject(this)
    }

    override fun onBind(intent: Intent?): IBinder? {
        return null
    }

    override fun onStartCommand(
        intent: Intent?,
        flags: Int,
        startId: Int,
    ): Int {
        logcat { "PIR-OPT-OUT: PIR service started on ${Process.myPid()} thread: ${Thread.currentThread().name}" }
        val notification: Notification = pirNotificationManager.createScanStatusNotification(
            title = getString(R.string.pirOptOutNotificationTitle),
            message = getString(
                R.string.pirOptOutNotificationMessageInProgress,
            ),
        )
        startForeground(1, notification)

        launch {
            if (pirWorkHandler.canRunPir().firstOrNull() == false) {
                logcat { "PIR-OPT-OUT: PIR opt-out not allowed to run!" }
                pirWorkHandler.cancelWork()
                stopSelf()
                return@launch
            }

            val brokers = intent?.getStringExtra(EXTRA_BROKER_TO_OPT_OUT)

            val result = if (!brokers.isNullOrEmpty()) {
                pirOptOut.execute(listOf(brokers), this@PirForegroundOptOutService)
            } else {
                pirOptOut.executeForBrokersWithRecords(this@PirForegroundOptOutService)
            }

            if (result.isSuccess) {
                pirNotificationManager.showScanStatusNotification(
                    title = getString(R.string.pirOptOutNotificationTitle),
                    message = getString(R.string.pirOptOutNotificationMessageComplete),
                )
            }
            stopSelf()
        }

        logcat { "PIR-OPT-OUT: START_NOT_STICKY" }
        return START_NOT_STICKY
    }

    override fun onDestroy() {
        logcat { "PIR-OPT-OUT: PIR service destroyed" }
        pirOptOut.stop()
    }

    companion object {
        const val EXTRA_BROKER_TO_OPT_OUT = "EXTRA_BROKER_TO_OPT_OUT"
    }
}
