/*
 * Copyright (c) 2025 DuckDuckGo
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.duckduckgo.pir.impl.common.actions

import com.duckduckgo.common.utils.CurrentTimeProvider
import com.duckduckgo.di.scopes.AppScope
import com.duckduckgo.pir.impl.common.BrokerStepsParser.BrokerStep
import com.duckduckgo.pir.impl.common.BrokerStepsParser.BrokerStep.OptOutStep
import com.duckduckgo.pir.impl.common.PirRunStateHandler
import com.duckduckgo.pir.impl.common.PirRunStateHandler.PirRunState.BrokerOptOutConditionFound
import com.duckduckgo.pir.impl.common.actions.EventHandler.Next
import com.duckduckgo.pir.impl.common.actions.PirActionsRunnerStateEngine.Event
import com.duckduckgo.pir.impl.common.actions.PirActionsRunnerStateEngine.Event.ConditionExpectationSucceeded
import com.duckduckgo.pir.impl.common.actions.PirActionsRunnerStateEngine.Event.ExecuteBrokerStepAction
import com.duckduckgo.pir.impl.common.actions.PirActionsRunnerStateEngine.State
import com.duckduckgo.pir.impl.scripts.models.PirScriptRequestData.UserProfile
import com.squareup.anvil.annotations.ContributesMultibinding
import javax.inject.Inject
import kotlin.reflect.KClass

@ContributesMultibinding(
    scope = AppScope::class,
    boundType = EventHandler::class,
)
class ConditionExpectationSucceededEventHandler @Inject constructor(
    private val pirRunStateHandler: PirRunStateHandler,
    private val currentTimeProvider: CurrentTimeProvider,
) : EventHandler {
    override val event: KClass<out Event> = ConditionExpectationSucceeded::class

    override suspend fun invoke(
        state: State,
        event: Event,
    ): Next {
        val actionsToAppend = (event as ConditionExpectationSucceeded).conditionActions
        val currentBrokerStep = state.brokerStepsToExecute[state.currentBrokerStepIndex]
        attemptFireOptOutStagePixel(currentBrokerStep, state)

        val updatedBrokerSteps = state.brokerStepsToExecute.toMutableList()
        val updatedBrokerActions = currentBrokerStep.step.actions.toMutableList().apply {
            this.addAll(
                state.currentActionIndex + 1,
                actionsToAppend,
            )
        }

        val updatedBrokerStep = when (currentBrokerStep) {
            is BrokerStep.ScanStep -> currentBrokerStep.copy(
                step = currentBrokerStep.step.copy(
                    actions = updatedBrokerActions,
                ),
            )

            is BrokerStep.OptOutStep -> currentBrokerStep.copy(
                step = currentBrokerStep.step.copy(
                    actions = updatedBrokerActions,
                ),
            )

            is BrokerStep.EmailConfirmationStep -> currentBrokerStep.copy(
                step = currentBrokerStep.step.copy(
                    actions = updatedBrokerActions,
                ),
            )
        }

        updatedBrokerSteps[state.currentBrokerStepIndex] = updatedBrokerStep

        return Next(
            nextState = state.copy(
                currentActionIndex = state.currentActionIndex + 1,
                brokerStepsToExecute = updatedBrokerSteps,
            ),
            nextEvent = ExecuteBrokerStepAction(
                UserProfile(
                    userProfile = state.profileQuery,
                ),
            ),
        )
    }
    private suspend fun attemptFireOptOutStagePixel(
        currentBrokerStep: BrokerStep,
        state: State,
    ) {
        if (currentBrokerStep is OptOutStep) {
            pirRunStateHandler.handleState(
                BrokerOptOutConditionFound(
                    broker = currentBrokerStep.broker,
                    actionID = currentBrokerStep.step.actions[state.currentActionIndex].id,
                    attemptId = state.attemptId,
                    durationMs = currentTimeProvider.currentTimeMillis() - state.stageStatus.stageStartMs,
                    currentActionAttemptCount = state.actionRetryCount + 1, // retry count starts at 0.
                ),
            )
        }
    }
}
