/*
 * Copyright (c) 2022 DuckDuckGo
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.duckduckgo.cookies.impl.features.firstparty

import androidx.lifecycle.Lifecycle.State.INITIALIZED
import androidx.lifecycle.testing.TestLifecycleOwner
import androidx.test.ext.junit.runners.AndroidJUnit4
import androidx.work.ExistingPeriodicWorkPolicy.KEEP
import androidx.work.ExistingPeriodicWorkPolicy.REPLACE
import androidx.work.WorkManager
import com.duckduckgo.common.test.CoroutineTestRule
import com.duckduckgo.cookies.api.CookiesFeatureName
import com.duckduckgo.feature.toggles.api.FeatureToggle
import kotlinx.coroutines.test.runTest
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.mockito.kotlin.any
import org.mockito.kotlin.eq
import org.mockito.kotlin.mock
import org.mockito.kotlin.verify
import org.mockito.kotlin.whenever

@RunWith(AndroidJUnit4::class)
class FirstPartyCookiesModifierWorkerSchedulerTest {

    private val mockToggle: FeatureToggle = mock()
    private val mockWorkManager: WorkManager = mock()
    private val lifecycleOwner = TestLifecycleOwner(initialState = INITIALIZED)

    lateinit var firstPartyCookiesModifierWorkerScheduler: FirstPartyCookiesModifierWorkerScheduler

    @get:Rule
    val coroutineTestRule: CoroutineTestRule = CoroutineTestRule()

    @Before
    fun before() {
        firstPartyCookiesModifierWorkerScheduler = FirstPartyCookiesModifierWorkerScheduler(
            workManager = mockWorkManager,
            toggle = mockToggle,
            dispatchers = coroutineTestRule.testDispatcherProvider,
        )
    }

    @Test
    fun whenOnStopIfFeatureEnabledThenEnqueueWorkWithReplacePolicy() = runTest {
        whenever(mockToggle.isFeatureEnabled(CookiesFeatureName.Cookie.value)).thenReturn(true)

        firstPartyCookiesModifierWorkerScheduler.onStop(lifecycleOwner)

        verify(mockWorkManager).enqueueUniquePeriodicWork(any(), eq(REPLACE), any())
    }

    @Test
    fun whenOnStopIfFeatureNotEnabledThenDeleteTag() = runTest {
        whenever(mockToggle.isFeatureEnabled(CookiesFeatureName.Cookie.value)).thenReturn(false)

        firstPartyCookiesModifierWorkerScheduler.onStop(lifecycleOwner)

        verify(mockWorkManager).cancelAllWorkByTag(any())
    }

    @Test
    fun whenOnStartIfFeatureEnabledThenEnqueueWorkWithKeepPolicy() = runTest {
        whenever(mockToggle.isFeatureEnabled(CookiesFeatureName.Cookie.value)).thenReturn(true)

        firstPartyCookiesModifierWorkerScheduler.onStart(lifecycleOwner)

        verify(mockWorkManager).enqueueUniquePeriodicWork(any(), eq(KEEP), any())
    }

    @Test
    fun whenOnStartIfFeatureNotEnabledThenDeleteTag() = runTest {
        whenever(mockToggle.isFeatureEnabled(CookiesFeatureName.Cookie.value)).thenReturn(false)

        firstPartyCookiesModifierWorkerScheduler.onStart(lifecycleOwner)

        verify(mockWorkManager).cancelAllWorkByTag(any())
    }
}
