/*
Copyright (c) 2015, Apps4Av Inc. (apps4av.com)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    *     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    *
    *     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.ds.avare.place;

import com.ds.avare.shapes.ChartShape;
import com.ds.avare.shapes.Shape;
import com.ds.avare.utils.BitmapHolder;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Created by zkhan on 8/18/15.
 * Chart boundaries class. Need to be created once like a database.
 */
public class Boundaries {

    private HashMap<String, Shape> mPolygons;
    private static Boundaries mInstance = null;

    private Boundaries() {
        makePolygons();
    }


    /**
     * Loop through to get name of chart we are on.
     *
     * @param lon
     * @param lat
     * @return
     */
    public String findChartOn(String chartIndex, double lon, double lat) {
        Iterator it = mPolygons.entrySet().iterator();
        while (it.hasNext()) {
            HashMap.Entry<String, ChartShape> pair = (HashMap.Entry<String, ChartShape>)it.next();
            ChartShape s = pair.getValue();
            String name = pair.getKey();
            String type = s.getName();
            // Search only specific type
            if(type.equals(chartIndex)) {
                // find if this point is in this chart
                type = s.getTextIfTouched(lon, lat);
                if(type != null) {
                    return name;
                }
            }
        }
        return "";
    }

    /*
     * Create once
     */
    public static Boundaries getInstance() {
        if(mInstance == null) {
            mInstance = new Boundaries();
        }
        return mInstance;
    }


    // Make chart boundary shapes
    private void makePolygons() {
        mPolygons = new HashMap<String, Shape>();

        // loop and add shapes
        for (int i = 0; i < mData.length; i += 4) {
            String type = mData[i + 0];
            String name = mData[i + 1];
            double lon = Double.parseDouble(mData[i + 2]);
            double lat = Double.parseDouble(mData[i + 3]);

            // add to hash, but check if it exists first
            Shape s = mPolygons.get(name);
            if (s == null) {
                // hashmap will save name, shape will save type
                s = new ChartShape(type);
                mPolygons.put(name, s);
            }
            // add point to shape
            s.add(lon, lat, false);
        }

        // Make all shapes
        Iterator it = mPolygons.entrySet().iterator();
        while (it.hasNext()) {
            HashMap.Entry<String, ChartShape> pair = (HashMap.Entry<String, ChartShape>)it.next();
            ChartShape s = pair.getValue();
            s.makePolygon();
        }
    }


    public static int getZoom(int index) {
        switch (BitmapHolder.WIDTH) {
            case 256:
                return mZooms[index] + 1;
            case 512:
                return mZooms[index];

        }
        return 0;
    }


    public static String getChartType(int index) {
        return mChartType[index];
    }

    public static ArrayList<String> getChartTypes() {
        return new ArrayList<String>(Arrays.asList(mChartType));
    }

    private static String[] mChartType = {
            "Sectional",
            "TAC",
            "WAC",
            "IFR Low",
            "IFR High",
            "IFR Area",
            "Terrain",
            "Shaded Relief",
            "Topographic",
            "Heli/Other",
            "ONC World",
            "TPC World",
            "Misc.",
            "Flyway"
    };


    // Zooms are for 512x512 tiles on charts
    private static int[] mZooms = {
            10,
            11,
            9,
            10,
            9,
            11,
            10,
            9,
            10,
            12,
            9,
            10,
            12,
	    11
    };


    // name,lon,lat
    private static String[] mData = {
        "0","Albuquerque","-109","36.0001",
        "0","Albuquerque","-109","32",
        "0","Albuquerque","-102","32",
        "0","Albuquerque","-102","36.0001",
        "0","Atlanta","-88.0004","36.0003",
        "0","Atlanta","-88.0004","32.0003",
        "0","Atlanta","-81.0002","32.0003",
        "0","Atlanta","-81.0002","36.0003",
        "0","Billings","-109.001","49.0001",
        "0","Billings","-109.001","44.5002",
        "0","Billings","-101","44.5002",
        "0","Billings","-101","49.0001",
        "0","Brownsville","-103","28.0001",
        "0","Brownsville","-103","24",
        "0","Brownsville","-97.0001","24",
        "0","Brownsville","-97.0001","28.0001",
        "0","Caribbean1","-85.5000","24.0000",
        "0","Caribbean1","-85.5000","16.0000",
        "0","Caribbean1","-72.3000","16.0000",
        "0","Caribbean1","-72.3000","24.0000",
        "0","Charlotte","-81.0004","36.0001",
        "0","Charlotte","-81.0004","32.0002",
        "0","Charlotte","-75.0005","32.0002",
        "0","Charlotte","-75.0005","36.0001",
        "0","Cheyenne","-109","44.5002",
        "0","Cheyenne","-109","40.0002",
        "0","Cheyenne","-101","40.0002",
        "0","Cheyenne","-101","44.5002",
        "0","Chicago","-93.0001","44.2002",
        "0","Chicago","-93.0001","40",
        "0","Chicago","-85.0002","40",
        "0","Chicago","-85.0002","44.2002",
        "0","Cincinnati","-85.0004","40.0003",
        "0","Cincinnati","-85.0004","36.0004",
        "0","Cincinnati","-78.0006","36.0004",
        "0","Cincinnati","-78.0006","40.0003",
        "0","Dallas-FtWorth","-102","36.0003",
        "0","Dallas-FtWorth","-102","32.0003",
        "0","Dallas-FtWorth","-95.0002","32.0003",
        "0","Dallas-FtWorth","-95.0002","36.0003",
        "0","Denver","-111","40.0003",
        "0","Denver","-111","35.7005",
        "0","Denver","-104","35.7005",
        "0","Denver","-104","40.0003",
        "0","Detroit","-85.0001","44.0001",
        "0","Detroit","-85.0001","40",
        "0","Detroit","-76.9999","40",
        "0","Detroit","-76.9999","44.0001",
        "0","ElPaso","-109","32.0003",
        "0","ElPaso","-109","28.0003",
        "0","ElPaso","-103","28.0003",
        "0","ElPaso","-103","32.0003",
        "0","GreatFalls","-117","49.0003",
        "0","GreatFalls","-117","44.5002",
        "0","GreatFalls","-109","44.5002",
        "0","GreatFalls","-109","49.0003",
        "0","GreenBay","-93.0005","48.3004",
        "0","GreenBay","-93.0005","44.2005",
        "0","GreenBay","-85.0005","44.2005",
        "0","GreenBay","-85.0005","48.3004",
        "0","Halifax","-69.0002","48.0002",
        "0","Halifax","-69.0002","44.0004",
        "0","Halifax","-61.0002","44.0004",
        "0","Halifax","-61.0002","48.0002",
        "0","HawaiianIslands","-160.5","23.0002",
        "0","HawaiianIslands","-160.5","18.5002",
        "0","HawaiianIslands","-154.5","18.5002",
        "0","HawaiianIslands","-154.5","23.0002",
        "0","Houston","-97.0001","32",
        "0","Houston","-97.0001","27.9999",
        "0","Houston","-91","27.9999",
        "0","Houston","-91","32",
        "0","Jacksonville","-85.0003","32.0002",
        "0","Jacksonville","-85.0003","28.0002",
        "0","Jacksonville","-79.0004","28.0002",
        "0","Jacksonville","-79.0004","32.0002",
        "0","KansasCity","-97.0001","40.0002",
        "0","KansasCity","-97.0001","36.0003",
        "0","KansasCity","-90.0001","36.0003",
        "0","KansasCity","-90.0001","40.0002",
        "0","KlamathFalls","-125","44.5001",
        "0","KlamathFalls","-125","40.0002",
        "0","KlamathFalls","-117","40.0002",
        "0","KlamathFalls","-117","44.5001",
        "0","LakeHuron","-85.0001","48.0001",
        "0","LakeHuron","-85.0001","44.0002",
        "0","LakeHuron","-77.0003","44.0002",
        "0","LakeHuron","-77.0003","48.0001",
        "0","LasVegas","-118","40.0003",
        "0","LasVegas","-118","35.7003",
        "0","LasVegas","-111","35.7003",
        "0","LasVegas","-111","40.0003",
        "0","LosAngeles","-122","36.0004",
        "0","LosAngeles","-122","32.0002",
        "0","LosAngeles","-116","32.0002",
        "0","LosAngeles","-116","36.0004",
        "0","Memphis","-95.0004","36.0001",
        "0","Memphis","-95.0004","32.0002",
        "0","Memphis","-88.0003","32.0002",
        "0","Memphis","-88.0003","36.0001",
        "0","Miami","-83.0003","28",
        "0","Miami","-83.0003","24",
        "0","Miami","-77.0003","24",
        "0","Miami","-77.0003","28",
        "0","Montreal","-77.0004","48.0003",
        "0","Montreal","-77.0004","44.0002",
        "0","Montreal","-69.0003","44.0002",
        "0","Montreal","-69.0003","48.0003",
        "0","NewOrleans","-91.0003","32.0003",
        "0","NewOrleans","-91.0003","28.0005",
        "0","NewOrleans","-85.0002","28.0005",
        "0","NewOrleans","-85.0002","32.0003",
        "0","NewYork","-77.0002","44.0001",
        "0","NewYork","-77.0002","40.0002",
        "0","NewYork","-69.0003","40.0002",
        "0","NewYork","-69.0003","44.0001",
        "0","Omaha","-101","44.5001",
        "0","Omaha","-101","40",
        "0","Omaha","-93.0002","40",
        "0","Omaha","-93.0002","44.5001",
        "0","Phoenix","-116","35.7001",
        "0","Phoenix","-116","31.3003",
        "0","Phoenix","-109","31.3003",
        "0","Phoenix","-109","35.7001",
        "0","SaltLakeCity","-117","44.5001",
        "0","SaltLakeCity","-117","40.0003",
        "0","SaltLakeCity","-109","40.0003",
        "0","SaltLakeCity","-109","44.5001",
        "0","SanAntonio","-103","32.0002",
        "0","SanAntonio","-103","28.0004",
        "0","SanAntonio","-97.0002","28.0004",
        "0","SanAntonio","-97.0002","32.0002",
        "0","SanFrancisco","-125","40.0002",
        "0","SanFrancisco","-125","36.0001",
        "0","SanFrancisco","-118","36.0001",
        "0","SanFrancisco","-118","40.0002",
        "0","Seattle","-125","49",
        "0","Seattle","-125","44.5002",
        "0","Seattle","-117","44.5002",
        "0","Seattle","-117","49",
        "0","StLouis","-90.0002","40.0001",
        "0","StLouis","-90.0002","36.0001",
        "0","StLouis","-85.0004","36.0001",
        "0","StLouis","-85.0004","40.0001",
        "0","TwinCities","-101","49.0003",
        "0","TwinCities","-101","44.5003",
        "0","TwinCities","-93","44.5003",
        "0","TwinCities","-93","49.0003",
        "0","Washington","-78.0002","40.0001",
        "0","Washington","-78.0002","36.0002",
        "0","Washington","-72.0002","36.0002",
        "0","Washington","-72.0002","40.0001",
        "0","Wichita","-104","40.0001",
        "0","Wichita","-104","36",
        "0","Wichita","-97.0001","36",
        "0","Wichita","-97.0001","40.0001",
        "0","Anchorage","-151.001","64.0001",
        "0","Anchorage","-151.001","60.0003",
        "0","Anchorage","-140.501","60.0003",
        "0","Anchorage","-140.501","64.0001",
        "0","Bethel","-173.001","64.0003",
        "0","Bethel","-173.001","59.7005",
        "0","Bethel","-162.001","59.7005",
        "0","Bethel","-162.001","64.0003",
        "0","CapeLisburne","-171.501","72.0001",
        "0","CapeLisburne","-171.501","68.0003",
        "0","CapeLisburne","-157.001","68.0003",
        "0","CapeLisburne","-157.001","72.0001",
        "0","ColdBay","-164.001","56.0003",
        "0","ColdBay","-164.001","54.0001",
        "0","ColdBay","-155.001","54.0001",
        "0","ColdBay","-155.001","56.0003",
        "0","Dawson","-145.001","68.0003",
        "0","Dawson","-145.001","64.0005",
        "0","Dawson","-132.001","64.0005",
        "0","Dawson","-132.001","68.0003",
        "0","DutchHarbor","-173.001","56.0002",
        "0","DutchHarbor","-173.001","52.0003",
        "0","DutchHarbor","-164.001","52.0003",
        "0","DutchHarbor","-164.001","56.0002",
        "0","Fairbanks","-158.001","68.0003",
        "0","Fairbanks","-158.001","64.0003",
        "0","Fairbanks","-145","64.0003",
        "0","Fairbanks","-145","68.0003",
        "0","Juneau","-141","60.0001",
        "0","Juneau","-141","56.0003",
        "0","Juneau","-130.5","56.0003",
        "0","Juneau","-130.5","60.0001",
        "0","Ketchikan","-139","56.0003",
        "0","Ketchikan","-139","52.0005",
        "0","Ketchikan","-130","52.0005",
        "0","Ketchikan","-130","56.0003",
        "0","Kodiak","-162.001","60.0001",
        "0","Kodiak","-162.001","56.0002",
        "0","Kodiak","-151.5","56.0002",
        "0","Kodiak","-151.5","60.0001",
        "0","McGrath","-162.001","64.0001",
        "0","McGrath","-162.001","60.0002",
        "0","McGrath","-151.001","60.0002",
        "0","McGrath","-151.001","64.0001",
        "0","Nome","-171.5","68.0003",
        "0","Nome","-171.5","64.0003",
        "0","Nome","-158.001","64.0003",
        "0","Nome","-158.001","68.0003",
        "0","PointBarrow","-157.001","72.0003",
        "0","PointBarrow","-157.001","68.0004",
        "0","PointBarrow","-141.001","68.0004",
        "0","PointBarrow","-141.001","72.0003",
        "0","Seward","-151.5","60.0001",
        "0","Seward","-151.5","59.3001",
        "0","Seward","-141","59.3001",
        "0","Seward","-141","60.0001",
        "0","Whitehorse","-140.501","64.0002",
        "0","Whitehorse","-140.501","60.0003",
        "0","Whitehorse","-130","60.0003",
        "0","Whitehorse","-130","64.0002",
        "1","AnchorageTAC","-152","61.65",
        "1","AnchorageTAC","-152","60.5666",
        "1","AnchorageTAC","-148","60.5666",
        "1","AnchorageTAC","-148","61.65",
        "1","AtlantaTAC","-85.5002","34.4001",
        "1","AtlantaTAC","-85.5002","33.0002",
        "1","AtlantaTAC","-83.6001","33.0002",
        "1","AtlantaTAC","-83.6001","34.4001",
        "1","Baltimore-WashingtonTAC","-78.6002","39.8002",
        "1","Baltimore-WashingtonTAC","-78.6002","38.1668",
        "1","Baltimore-WashingtonTAC","-75.7701","38.1668",
        "1","Baltimore-WashingtonTAC","-75.7701","39.8002",
        "1","BostonTAC","-72.3001","42.9",
        "1","BostonTAC","-72.3001","41.2501",
        "1","BostonTAC","-69.5","41.2501",
        "1","BostonTAC","-69.5","42.9",
        "1","CharlotteTAC","-82.1002","35.9001",
        "1","CharlotteTAC","-82.1002","34.5802",
        "1","CharlotteTAC","-80.0802","34.5802",
        "1","CharlotteTAC","-80.0802","35.9001",
        "1","ChicagoTAC","-89.1001","42.5001",
        "1","ChicagoTAC","-89.1001","41.4302",
        "1","ChicagoTAC","-86.9001","41.4302",
        "1","ChicagoTAC","-86.9001","42.5001",
        "1","CincinnatiTAC","-85.9001","40.1001",
        "1","CincinnatiTAC","-85.9001","38.4601",
        "1","CincinnatiTAC","-83.7002","38.4601",
        "1","CincinnatiTAC","-83.7002","40.1001",
        "1","ClevelandTAC","-82.9002","41.9502",
        "1","ClevelandTAC","-82.9002","40.8402",
        "1","ClevelandTAC","-80.6002","40.8402",
        "1","ClevelandTAC","-80.6002","41.9502",
        "1","ColoradoSpringsTAC","-106.2","39.25",
        "1","ColoradoSpringsTAC","-106.2","37.6499",
        "1","ColoradoSpringsTAC","-103.45","37.6499",
        "1","ColoradoSpringsTAC","-103.45","39.25",
        "1","Dallas-FtWorthTAC","-98.5","33.6701",
        "1","Dallas-FtWorthTAC","-98.5","32.0501",
        "1","Dallas-FtWorthTAC","-95.8501","32.0501",
        "1","Dallas-FtWorthTAC","-95.8501","33.6701",
        "1","DenverTAC","-105.9","40.6001",
        "1","DenverTAC","-105.9","39.2501",
        "1","DenverTAC","-103.73","39.2501",
        "1","DenverTAC","-103.73","40.6001",
        "1","DetroitTAC","-84.8001","42.7501",
        "1","DetroitTAC","-84.8001","41.4202",
        "1","DetroitTAC","-82.6602","41.4202",
        "1","DetroitTAC","-82.6602","42.7501",
        "1","FairbanksTAC","-150.6","65.2501",
        "1","FairbanksTAC","-150.6","64.1502",
        "1","FairbanksTAC","-145.8","64.1502",
        "1","FairbanksTAC","-145.8","65.2501",
        "1","HonoluluInset","-158.5054","21.6198",
        "1","HonoluluInset","-158.5054","20.7348",
        "1","HonoluluInset","-157.3469","20.7348",
        "1","HonoluluInset","-157.3469","21.6198",
        "1","HoustonTAC","-96.4","30.5502",
        "1","HoustonTAC","-96.4","29.1002",
        "1","HoustonTAC","-94.4999","29.1002",
        "1","HoustonTAC","-94.4999","30.5502",
        "1","KansasCityTAC","-96.0001","40",
        "1","KansasCityTAC","-96.0001","38.6601",
        "1","KansasCityTAC","-94.0002","38.6601",
        "1","KansasCityTAC","-94.0002","40",
        "1","LasVegasTAC","-116.3","36.8002",
        "1","LasVegasTAC","-116.3","35.6671",
        "1","LasVegasTAC","-113.85","35.6671",
        "1","LasVegasTAC","-113.85","36.8002",
        "1","LosAngelesTAC","-120.15","34.5",
        "1","LosAngelesTAC","-120.15","33.37",
        "1","LosAngelesTAC","-116.45","33.37",
        "1","LosAngelesTAC","-116.45","34.5",
        "1","MemphisTAC","-91.1501","35.8001",
        "1","MemphisTAC","-91.1501","34.4002",
        "1","MemphisTAC","-88.7002","34.4002",
        "1","MemphisTAC","-88.7002","35.8001",
        "1","MiamiTAC","-81.45","26.7",
        "1","MiamiTAC","-81.45","25.1",
        "1","MiamiTAC","-79.33","25.1",
        "1","MiamiTAC","-79.33","26.7",
        "1","Minneapolis-StPaulTAC","-94.3001","45.5",
        "1","Minneapolis-StPaulTAC","-94.3001","44.3",
        "1","Minneapolis-StPaulTAC","-92.0001","44.3",
        "1","Minneapolis-StPaulTAC","-92.0001","45.5",
        "1","NewOrleansTAC","-91.5","30.6001",
        "1","NewOrleansTAC","-91.5","29.5502",
        "1","NewOrleansTAC","-89.2999","29.5502",
        "1","NewOrleansTAC","-89.2999","30.6001",
        "1","NewYorkTAC","-75.6002","41.3001",
        "1","NewYorkTAC","-75.6002","40.22",
        "1","NewYorkTAC","-72.6603","40.22",
        "1","NewYorkTAC","-72.6603","41.3001",
        "1","OrlandoTAC","-82.5001","29.2",
        "1","OrlandoTAC","-82.5001","27.8001",
        "1","OrlandoTAC","-80.2001","27.8001",
        "1","OrlandoTAC","-80.2001","29.2",
        "1","PhiladelphiaTAC","-75.9169","40.5",
        "1","PhiladelphiaTAC","-75.9169","39.3833",
        "1","PhiladelphiaTAC","-74.5168","39.3833",
        "1","PhiladelphiaTAC","-74.5168","40.5",
        "1","PhoenixTAC","-113.5","34.15",
        "1","PhoenixTAC","-113.5","32.7501",
        "1","PhoenixTAC","-111.16","32.7501",
        "1","PhoenixTAC","-111.16","34.15",
        "1","PittsburghTAC","-81.0002","41.05",
        "1","PittsburghTAC","-81.0002","39.95",
        "1","PittsburghTAC","-79.0002","39.95",
        "1","PittsburghTAC","-79.0002","41.05",
        "1","PortlandTAC","-123.1922","46.0181",
        "1","PortlandTAC","-123.1922","45.182",
        "1","PortlandTAC","-122.0661","45.182",
        "1","PortlandTAC","-122.0661","46.0181",
        "1","PuertoRico-VITAC","-67.7502","18.7601",
        "1","PuertoRico-VITAC","-67.7502","17.6301",
        "1","PuertoRico-VITAC","-64.2203","17.6301",
        "1","PuertoRico-VITAC","-64.2203","18.7601",
        "1","SaltLakeCityTAC","-113.2","41.5",
        "1","SaltLakeCityTAC","-113.2","40.1",
        "1","SaltLakeCityTAC","-110.65","40.1",
        "1","SaltLakeCityTAC","-110.65","41.5",
        "1","SanDiegoTAC","-117.9","33.6",
        "1","SanDiegoTAC","-117.9","32.5001",
        "1","SanDiegoTAC","-116.3","32.5001",
        "1","SanDiegoTAC","-116.3","33.6",
        "1","SanFranciscoTAC","-123.8","38.2001",
        "1","SanFranciscoTAC","-123.8","36.9",
        "1","SanFranciscoTAC","-121.36","36.9",
        "1","SanFranciscoTAC","-121.36","38.2001",
        "1","SeattleTAC","-123.6","48.1501",
        "1","SeattleTAC","-123.6","46.7201",
        "1","SeattleTAC","-121.1","46.7201",
        "1","SeattleTAC","-121.1","48.1501",
        "1","StLouisTAC","-91.4001","39.3002",
        "1","StLouisTAC","-91.4001","38.1503",
        "1","StLouisTAC","-89.2502","38.1503",
        "1","StLouisTAC","-89.2502","39.3002",
        "1","TampaTAC","-83.3002","28.5701",
        "1","TampaTAC","-83.3002","27.2502",
        "1","TampaTAC","-81.8501","27.2502",
        "1","TampaTAC","-81.8501","28.5701",
        "2","CC-8","-179.901","72.0001",
        "2","CC-8","-179.901","64",
        "2","CC-8","-158.001","64",
        "2","CC-8","-158.001","72.0001",
        "2","CC-9","-158.001","72.0002",
        "2","CC-9","-158.001","64.0001",
        "2","CC-9","-132.001","64.0001",
        "2","CC-9","-132.001","72.0002",
        "2","CD-10","-179.9","64.0006",
        "2","CD-10","-179.9","56.0011",
        "2","CD-10","-162.001","56.0011",
        "2","CD-10","-162.001","64.0006",
        "2","CD-11","-162.001","64.0005",
        "2","CD-11","-162.001","56.0005",
        "2","CD-11","-141.002","56.0005",
        "2","CD-11","-141.002","64.0005",
        "2","CD-12","-141.001","64.0002",
        "2","CD-12","-141.001","56.0005",
        "2","CD-12","-120.001","56.0005",
        "2","CD-12","-120.001","64.0002",
        "2","CE-12","-179.901","54.0003",
        "2","CE-12","-179.901","50.0002",
        "2","CE-12","-171.002","50.0002",
        "2","CE-12","-171.002","54.0003",
        "2","CE-13","-171.001","56.0004",
        "2","CE-13","-171.001","52.0002",
        "2","CE-13","-153.001","52.0002",
        "2","CE-13","-153.001","56.0004",
        "2","CE-15","-141.001","56.0006",
        "2","CE-15","-141.001","48.001",
        "2","CE-15","-123.001","48.001",
        "2","CE-15","-123.001","56.0006",
        "2","CF-16","-125.001","49.0001",
        "2","CF-16","-125.001","40.0003",
        "2","CF-16","-109.001","40.0003",
        "2","CF-16","-109.001","49.0001",
        "2","CF-17","-109","49.0003",
        "2","CF-17","-109","40.0004",
        "2","CF-17","-92.9999","40.0004",
        "2","CF-17","-92.9999","49.0003",
        "2","CF-18","-93.0006","48.0006",
        "2","CF-18","-93.0006","40.0009",
        "2","CF-18","-77.0004","40.0009",
        "2","CF-18","-77.0004","48.0006",
        "2","CF-19","-77.0003","48.0001",
        "2","CF-19","-77.0003","40.0001",
        "2","CF-19","-61.0003","40.0001",
        "2","CF-19","-61.0003","48.0001",
        "2","CG-18","-125","40.0002",
        "2","CG-18","-125","32.0005",
        "2","CG-18","-111","32.0005",
        "2","CG-18","-111","40.0002",
        "2","CG-19","-111.001","40.0001",
        "2","CG-19","-111.001","32.0005",
        "2","CG-19","-100.001","32.0005",
        "2","CG-19","-100.001","40.0001",
        "2","CG-20","-100","40.0006",
        "2","CG-20","-100","32.0008",
        "2","CG-20","-86.0003","32.0008",
        "2","CG-20","-86.0003","40.0006",
        "2","CG-21","-86.0001","40.0003",
        "2","CG-21","-86.0001","32.0008",
        "2","CG-21","-72.9998","32.0008",
        "2","CG-21","-72.9998","40.0003",
        "2","CH-22","-121","32.0001",
        "2","CH-22","-121","23.9005",
        "2","CH-22","-109","23.9005",
        "2","CH-22","-109","32.0001",
        "2","CH-23","-109","32.0003",
        "2","CH-23","-109","24.0007",
        "2","CH-23","-97.0006","24.0007",
        "2","CH-23","-97.0006","32.0003",
        "2","CH-24","-97.0004","32.0007",
        "2","CH-24","-97.0004","24.0006",
        "2","CH-24","-85.0006","24.0006",
        "2","CH-24","-85.0006","32.0007",
        "2","CH-25","-85.0003","32.0007",
        "2","CH-25","-85.0003","24.0011",
        "2","CH-25","-73.0006","24.0011",
        "2","CH-25","-73.0006","32.0007",
        "2","CJ-26","-85.0002","24.0002",
        "2","CJ-26","-85.0002","16.0003",
        "2","CJ-26","-73.0002","16.0003",
        "2","CJ-26","-73.0002","24.0002",
        "2","CJ-27","-73.0008","24.0007",
        "2","CJ-27","-73.0008","16.0004",
        "2","CJ-27","-61.0004","16.0004",
        "2","CJ-27","-61.0004","24.0007",
        "3","ELUS_AK","-180.00","75.00",
        "3","ELUS_AK","-180.00","50.00",
        "3","ELUS_AK","-125.00","50.00",
        "3","ELUS_AK","-125.00","75.00",
        "3","ELUS_HI","-165.00","25.00",
        "3","ELUS_HI","-165.00","15.00",
        "3","ELUS_HI","-150.00","15.00",
        "3","ELUS_HI","-150.00","25.00",
        "3","ELUS_NE","-85.00","50.15",
        "3","ELUS_NE","-85.00","38.00",
        "3","ELUS_NE","-40.00","38.00",
        "3","ELUS_NE","-40.00","50.15",
        "3","ELUS_NC","-110.00","50.15",
        "3","ELUS_NC","-110.00","38.00",
        "3","ELUS_NC","-85.00","38.00",
        "3","ELUS_NC","-85.00","50.15",
        "3","ELUS_NW","-131.21","50.15",
        "3","ELUS_NW","-131.21","38.00",
        "3","ELUS_NW","-110.00","38.00",
        "3","ELUS_NW","-110.00","50.15",
        "3","ELUS_SE","-85.00","38.00",
        "3","ELUS_SE","-85.00","23.13",
        "3","ELUS_SE","-40.00","23.13",
        "3","ELUS_SE","-40.00","38.00",
        "3","ELUS_SC","-110.00","38.00",
        "3","ELUS_SC","-110.00","23.13",
        "3","ELUS_SC","-85.00","23.13",
        "3","ELUS_SC","-85.00","38.00",
        "3","ELUS_SW","-131.21","38.00",
        "3","ELUS_SW","-131.21","23.13",
        "3","ELUS_SW","-110.00","23.13",
        "3","ELUS_SW","-110.00","38.00",
        "4","EHUS_AK","-180.00","75.00",
        "4","EHUS_AK","-180.00","50.00",
        "4","EHUS_AK","-125.00","50.00",
        "4","EHUS_AK","-125.00","75.00",
        "4","EHUS_NE","-85.00","50.15",
        "4","EHUS_NE","-85.00","38.00",
        "4","EHUS_NE","-40.00","38.00",
        "4","EHUS_NE","-40.00","50.15",
        "4","EHUS_NC","-110.00","50.15",
        "4","EHUS_NC","-110.00","38.00",
        "4","EHUS_NC","-85.00","38.00",
        "4","EHUS_NC","-85.00","50.15",
        "4","EHUS_NW","-131.21","50.15",
        "4","EHUS_NW","-131.21","38.00",
        "4","EHUS_NW","-110.00","38.00",
        "4","EHUS_NW","-110.00","50.15",
        "4","EHUS_SE","-85.00","38.00",
        "4","EHUS_SE","-85.00","23.13",
        "4","EHUS_SE","-40.00","23.13",
        "4","EHUS_SE","-40.00","38.00",
        "4","EHUS_SC","-110.00","38.00",
        "4","EHUS_SC","-110.00","23.13",
        "4","EHUS_SC","-85.00","23.13",
        "4","EHUS_SC","-85.00","38.00",
        "4","EHUS_SW","-131.21","38.00",
        "4","EHUS_SW","-131.21","23.13",
        "4","EHUS_SW","-110.00","23.13",
        "4","EHUS_SW","-110.00","38.00",
        "5","ENRA_ATL","-85.0025","34.1349",
        "5","ENRA_ATL","-85.0025","33.1301",
        "5","ENRA_ATL","-83.7954","33.1301",
        "5","ENRA_ATL","-83.7954","34.1349",
        "5","ENRA_DCA","-78.4084","40.5082",
        "5","ENRA_DCA","-78.4084","38.1132",
        "5","ENRA_DCA","-74.5667","38.1132",
        "5","ENRA_DCA","-74.5667","40.5082",
        "5","ENRA_DET","-83.8536","42.7682",
        "5","ENRA_DET","-83.8536","41.9303",
        "5","ENRA_DET","-82.7183","41.9303",
        "5","ENRA_DET","-82.7183","42.7682",
        "5","ENRA_JAX","-82.1856","30.9551",
        "5","ENRA_JAX","-82.1856","29.9617",
        "5","ENRA_JAX","-81.0273","29.9617",
        "5","ENRA_JAX","-81.0273","30.9551",
        "5","ENRA_MIA","-80.8697","26.3454",
        "5","ENRA_MIA","-80.8697","25.3606",
        "5","ENRA_MIA","-79.7762","25.3606",
        "5","ENRA_MIA","-79.7762","26.3454",
        "5","ENRA_MSP","-94.1216","45.5044",
        "5","ENRA_MSP","-94.1216","44.1732",
        "5","ENRA_MSP","-92.2273","44.1732",
        "5","ENRA_MSP","-92.2273","45.5044",
        "5","ENRA_STL","-91.0337","39.2391",
        "5","ENRA_STL","-91.0337","38.2353",
        "5","ENRA_STL","-89.7394","38.2353",
        "5","ENRA_STL","-89.7394","39.2391",
        "5","ENRA_DEN","-105.43","40.436",
        "5","ENRA_DEN","-105.43","39.2729",
        "5","ENRA_DEN","-103.898","39.2729",
        "5","ENRA_DEN","-103.898","40.436",
        "5","ENRA_DFW","-98.1547","33.7802",
        "5","ENRA_DFW","-98.1547","31.4364",
        "5","ENRA_DFW","-96.0523","31.4364",
        "5","ENRA_DFW","-96.0523","33.7802",
        "5","ENRA_LAX","-119.158","34.6019",
        "5","ENRA_LAX","-119.158","33.5209",
        "5","ENRA_LAX","-117.02","33.5209",
        "5","ENRA_LAX","-117.02","34.6019",
        "5","ENRA_MKC","-95.1022","39.58",
        "5","ENRA_MKC","-95.1022","38.7357",
        "5","ENRA_MKC","-94.0181","38.7357",
        "5","ENRA_MKC","-94.0181","39.58",
        "5","ENRA_ORD","-88.6454","43.3283",
        "5","ENRA_ORD","-88.6454","41.3186",
        "5","ENRA_ORD","-87.2715","41.3186",
        "5","ENRA_ORD","-87.2715","43.3283",
        "5","ENRA_PHX","-112.901","34.1691",
        "5","ENRA_PHX","-112.901","32.7583",
        "5","ENRA_PHX","-111.19","32.7583",
        "5","ENRA_PHX","-111.19","34.1691",
        "5","ENRA_SFO","-122.935","38.3236",
        "5","ENRA_SFO","-122.935","37.3249",
        "5","ENRA_SFO","-121.657","37.3249",
        "5","ENRA_SFO","-121.657","38.3236",
        "5","ENRA_JNU","-138.968","60.1019",
        "5","ENRA_JNU","-138.968","56.682",
        "5","ENRA_JNU","-132.489","56.682",
        "5","ENRA_JNU","-132.489","60.1019",
        "5","ENRA_VR","-124.568","49.7313",
        "5","ENRA_VR","-124.568","47.1488",
        "5","ENRA_VR","-121.402","47.1488",
        "5","ENRA_VR","-121.402","49.7313",
        "5","ENRA_FAI","-150.258","65.3324",
        "5","ENRA_FAI","-150.258","63.7115",
        "5","ENRA_FAI","-145.022","63.7115",
        "5","ENRA_FAI","-145.022","65.3324",
        "5","ENRA_OME","-167.078","65.4198",
        "5","ENRA_OME","-167.078","63.823",
        "5","ENRA_OME","-163.218","63.823",
        "5","ENRA_OME","-163.218","65.4198",
        "5","ENRA_ANC","-153.303","62.8585",
        "5","ENRA_ANC","-153.303","58.2111",
        "5","ENRA_ANC","-143.853","58.2111",
        "5","ENRA_ANC","-143.853","62.8585",
        "5","ENRA_GUA","139.725","18.4257",
        "5","ENRA_GUA","139.725","8.55736",
        "5","ENRA_GUA","150.246","8.55736",
        "5","ENRA_GUA","150.246","18.4257",
        "6","ELEV_AK","-180.00","75.00",
        "6","ELEV_AK","-180.00","50.00",
        "6","ELEV_AK","-125.00","50.00",
        "6","ELEV_AK","-125.00","75.00",
        "6","ELEV_NE","-85.00","50.15",
        "6","ELEV_NE","-85.00","38.00",
        "6","ELEV_NE","-40.00","38.00",
        "6","ELEV_NE","-40.00","50.15",
        "6","ELEV_NC","-110.00","50.15",
        "6","ELEV_NC","-110.00","38.00",
        "6","ELEV_NC","-85.00","38.00",
        "6","ELEV_NC","-85.00","50.15",
        "6","ELEV_NW","-131.21","50.15",
        "6","ELEV_NW","-131.21","38.00",
        "6","ELEV_NW","-110.00","38.00",
        "6","ELEV_NW","-110.00","50.15",
        "6","ELEV_SE","-85.00","38.00",
        "6","ELEV_SE","-85.00","23.13",
        "6","ELEV_SE","-40.00","23.13",
        "6","ELEV_SE","-40.00","38.00",
        "6","ELEV_SC","-110.00","38.00",
        "6","ELEV_SC","-110.00","23.13",
        "6","ELEV_SC","-85.00","23.13",
        "6","ELEV_SC","-85.00","38.00",
        "6","ELEV_SW","-131.21","38.00",
        "6","ELEV_SW","-131.21","23.13",
        "6","ELEV_SW","-110.00","23.13",
        "6","ELEV_SW","-110.00","38.00",
        "7","REL_AK","-180.00","75.00",
        "7","REL_AK","-180.00","50.00",
        "7","REL_AK","-125.00","50.00",
        "7","REL_AK","-125.00","75.00",
        "7","REL_NE","-85.00","50.15",
        "7","REL_NE","-85.00","38.00",
        "7","REL_NE","-40.00","38.00",
        "7","REL_NE","-40.00","50.15",
        "7","REL_NC","-110.00","50.15",
        "7","REL_NC","-110.00","38.00",
        "7","REL_NC","-85.00","38.00",
        "7","REL_NC","-85.00","50.15",
        "7","REL_NW","-131.21","50.15",
        "7","REL_NW","-131.21","38.00",
        "7","REL_NW","-110.00","38.00",
        "7","REL_NW","-110.00","50.15",
        "7","REL_SE","-85.00","38.00",
        "7","REL_SE","-85.00","23.13",
        "7","REL_SE","-40.00","23.13",
        "7","REL_SE","-40.00","38.00",
        "7","REL_SC","-110.00","38.00",
        "7","REL_SC","-110.00","23.13",
        "7","REL_SC","-85.00","23.13",
        "7","REL_SC","-85.00","38.00",
        "7","REL_SW","-131.21","38.00",
        "7","REL_SW","-131.21","23.13",
        "7","REL_SW","-110.00","23.13",
        "7","REL_SW","-110.00","38.00",
        "8","CAN_001","-56.1186","48.0991",
        "8","CAN_001","-56.1186","45.7429",
        "8","CAN_001","-51.6041","45.7429",
        "8","CAN_001","-51.6041","48.0991",
        "8","CAN_002","-56.1319","52.1051",
        "8","CAN_002","-56.1319","47.7666",
        "8","CAN_002","-51.6048","47.7666",
        "8","CAN_002","-51.6048","52.1051",
        "8","CAN_011","-64.1148","48.1072",
        "8","CAN_011","-64.1148","43.7851",
        "8","CAN_011","-55.6719","43.7851",
        "8","CAN_011","-55.6719","48.1072",
        "8","CAN_012","-64.1372","52.1039",
        "8","CAN_012","-64.1372","47.7641",
        "8","CAN_012","-55.1729","47.7641",
        "8","CAN_012","-55.1729","52.1039",
        "8","CAN_013","-64.2153","56.1114",
        "8","CAN_013","-64.2153","51.7697",
        "8","CAN_013","-54.8957","51.7697",
        "8","CAN_013","-54.8957","56.1114",
        "8","CAN_014","-64.1825","60.0655",
        "8","CAN_014","-64.1825","55.7539",
        "8","CAN_014","-59.5227","55.7539",
        "8","CAN_014","-59.5227","60.0655",
        "8","CAN_016","-64.8086","68.0575",
        "8","CAN_016","-64.8086","64.8086",
        "8","CAN_016","-60.2083","64.8086",
        "8","CAN_016","-60.2083","68.0575",
        "8","CAN_020","-66.6258","44.097",
        "8","CAN_020","-66.6258","42.7506",
        "8","CAN_020","-64.2146","42.7506",
        "8","CAN_020","-64.2146","44.097",
        "8","CAN_021","-72.2224","48.1145",
        "8","CAN_021","-72.2224","43.7607",
        "8","CAN_021","-63.6065","43.7607",
        "8","CAN_021","-63.6065","48.1145",
        "8","CAN_022","-72.1846","52.1072",
        "8","CAN_022","-72.1846","47.7768",
        "8","CAN_022","-63.5422","47.7768",
        "8","CAN_022","-63.5422","52.1072",
        "8","CAN_023","-72.2618","56.0968",
        "8","CAN_023","-72.2618","51.7634",
        "8","CAN_023","-63.4413","51.7634",
        "8","CAN_023","-63.4413","56.0968",
        "8","CAN_024","-72.4037","60.0994",
        "8","CAN_024","-72.4037","55.7703",
        "8","CAN_024","-63.3277","55.7703",
        "8","CAN_024","-63.3277","60.0994",
        "8","CAN_025","-72.6204","64.1112",
        "8","CAN_025","-72.6204","59.7672",
        "8","CAN_025","-62.972","59.7672",
        "8","CAN_025","-62.972","64.1112",
        "8","CAN_026","-72.9534","68.1137",
        "8","CAN_026","-72.9534","63.7485",
        "8","CAN_026","-62.6107","63.7485",
        "8","CAN_026","-62.6107","68.1137",
        "8","CAN_027","-72.4271","72.0883",
        "8","CAN_027","-72.4271","67.5387",
        "8","CAN_027","-62.7836","67.5387",
        "8","CAN_027","-62.7836","72.0883",
        "8","CAN_030","-80.0992","44.0897",
        "8","CAN_030","-80.0992","41.7711",
        "8","CAN_030","-75.7145","41.7711",
        "8","CAN_030","-75.7145","44.0897",
        "8","CAN_031","-80.1349","48.093",
        "8","CAN_031","-80.1349","43.6389",
        "8","CAN_031","-71.622","43.6389",
        "8","CAN_031","-71.622","48.093",
        "8","CAN_032","-80.1542","52.1089",
        "8","CAN_032","-80.1542","47.7734",
        "8","CAN_032","-71.5062","47.7734",
        "8","CAN_032","-71.5062","52.1089",
        "8","CAN_033","-80.3459","56.1128",
        "8","CAN_033","-80.3459","51.7608",
        "8","CAN_033","-71.3577","51.7608",
        "8","CAN_033","-71.3577","56.1128",
        "8","CAN_034","-80.3332","60.1347",
        "8","CAN_034","-80.3332","55.7581",
        "8","CAN_034","-71.171","55.7581",
        "8","CAN_034","-71.171","60.1347",
        "8","CAN_035","-79.3422","64.2079",
        "8","CAN_035","-79.3422","59.748",
        "8","CAN_035","-70.8745","59.748",
        "8","CAN_035","-70.8745","64.2079",
        "8","CAN_036","-79.0757","68.1103",
        "8","CAN_036","-79.0757","63.7564",
        "8","CAN_036","-70.5287","63.7564",
        "8","CAN_036","-70.5287","68.1103",
        "8","CAN_037","-80.4198","72.1372",
        "8","CAN_037","-80.4198","67.5375",
        "8","CAN_037","-70.913","67.5375",
        "8","CAN_037","-70.913","72.1372",
        "8","CAN_038","-80.5806","74.1411",
        "8","CAN_038","-80.5806","71.7355",
        "8","CAN_038","-70.8876","71.7355",
        "8","CAN_038","-70.8876","74.1411",
        "8","CAN_039","-82.1586","80.1233",
        "8","CAN_039","-82.1586","75.7104",
        "8","CAN_039","-68.6655","75.7104",
        "8","CAN_039","-68.6655","80.1233",
        "8","CAN_040","-84.15","44.0595",
        "8","CAN_040","-84.15","40.7642",
        "8","CAN_040","-79.5168","40.7642",
        "8","CAN_040","-79.5168","44.0595",
        "8","CAN_041","-86.1356","48.0983",
        "8","CAN_041","-86.1356","43.7934",
        "8","CAN_041","-79.5625","43.7934",
        "8","CAN_041","-79.5625","48.0983",
        "8","CAN_042","-88.1286","52.1015",
        "8","CAN_042","-88.1286","47.7641",
        "8","CAN_042","-79.5629","47.7641",
        "8","CAN_042","-79.5629","52.1015",
        "8","CAN_043","-88.2073","56.0967",
        "8","CAN_043","-88.2073","51.7723",
        "8","CAN_043","-79.5437","51.7723",
        "8","CAN_043","-79.5437","56.0967",
        "8","CAN_044","-82.195","60.0594",
        "8","CAN_044","-82.195","57.8096",
        "8","CAN_044","-79.3902","57.8096",
        "8","CAN_044","-79.3902","60.0594",
        "8","CAN_045","-88.6085","64.1206",
        "8","CAN_045","-88.6085","61.2888",
        "8","CAN_045","-78.0502","61.2888",
        "8","CAN_045","-78.0502","64.1206",
        "8","CAN_046","-88.7905","68.1173",
        "8","CAN_046","-88.7905","63.752",
        "8","CAN_046","-78.899","63.752",
        "8","CAN_046","-78.899","68.1173",
        "8","CAN_047","-88.7291","72.0924",
        "8","CAN_047","-88.7291","67.3568",
        "8","CAN_047","-78.9895","67.3568",
        "8","CAN_047","-78.9895","72.0924",
        "8","CAN_048","-88.7814","76.0991",
        "8","CAN_048","-88.7814","71.7775",
        "8","CAN_048","-77.8782","71.7775",
        "8","CAN_048","-77.8782","76.0991",
        "8","CAN_049","-90.3176","80.0959",
        "8","CAN_049","-90.3176","75.7687",
        "8","CAN_049","-76.8333","75.7687",
        "8","CAN_049","-76.8333","80.0959",
        "8","CAN_052","-96.1659","52.0953",
        "8","CAN_052","-96.1659","47.763",
        "8","CAN_052","-87.554","47.763",
        "8","CAN_052","-87.554","52.0953",
        "8","CAN_053","-96.2706","56.0931",
        "8","CAN_053","-96.2706","51.7755",
        "8","CAN_053","-87.4157","51.7755",
        "8","CAN_053","-87.4157","56.0931",
        "8","CAN_054","-96.4169","60.0884",
        "8","CAN_054","-96.4169","55.6009",
        "8","CAN_054","-86.6804","55.6009",
        "8","CAN_054","-86.6804","60.0884",
        "8","CAN_055","-96.6958","64.1129",
        "8","CAN_055","-96.6958","59.7623",
        "8","CAN_055","-86.9385","59.7623",
        "8","CAN_055","-86.9385","64.1129",
        "8","CAN_056","-96.9507","68.1107",
        "8","CAN_056","-96.9507","63.751",
        "8","CAN_056","-86.5816","63.751",
        "8","CAN_056","-86.5816","68.1107",
        "8","CAN_057","-96.4668","72.1487",
        "8","CAN_057","-96.4668","67.7134",
        "8","CAN_057","-86.8119","67.7134",
        "8","CAN_057","-86.8119","72.1487",
        "8","CAN_058","-96.9661","76.1201",
        "8","CAN_058","-96.9661","71.7141",
        "8","CAN_058","-86.3307","71.7141",
        "8","CAN_058","-86.3307","76.1201",
        "8","CAN_059","-98.2094","80.1281",
        "8","CAN_059","-98.2094","75.704",
        "8","CAN_059","-84.4982","75.704",
        "8","CAN_059","-84.4982","80.1281",
        "8","CAN_062","-104.244","52.0959",
        "8","CAN_062","-104.244","48.7681",
        "8","CAN_062","-95.5186","48.7681",
        "8","CAN_062","-95.5186","52.0959",
        "8","CAN_063","-104.214","56.1044",
        "8","CAN_063","-104.214","51.7742",
        "8","CAN_063","-95.3946","51.7742",
        "8","CAN_063","-95.3946","56.1044",
        "8","CAN_064","-104.368","60.0985",
        "8","CAN_064","-104.368","55.7642",
        "8","CAN_064","-95.1605","55.7642",
        "8","CAN_064","-95.1605","60.0985",
        "8","CAN_065","-104.663","64.1127",
        "8","CAN_065","-104.663","59.755",
        "8","CAN_065","-94.863","59.755",
        "8","CAN_065","-94.863","64.1127",
        "8","CAN_066","-104.854","68.1152",
        "8","CAN_066","-104.854","63.7529",
        "8","CAN_066","-94.5073","63.7529",
        "8","CAN_066","-94.5073","68.1152",
        "8","CAN_067","-104.361","72.1003",
        "8","CAN_067","-104.361","67.6054",
        "8","CAN_067","-94.8986","67.6054",
        "8","CAN_067","-94.8986","72.1003",
        "8","CAN_068","-104.754","76.1246",
        "8","CAN_068","-104.754","71.7",
        "8","CAN_068","-94.2327","71.7",
        "8","CAN_068","-94.2327","76.1246",
        "8","CAN_069","-105.737","80.108",
        "8","CAN_069","-105.737","75.72",
        "8","CAN_069","-92.7763","75.72",
        "8","CAN_069","-92.7763","80.108",
        "8","CAN_072","-112.14","52.1004",
        "8","CAN_072","-112.14","48.7487",
        "8","CAN_072","-103.561","48.7487",
        "8","CAN_072","-103.561","52.1004",
        "8","CAN_073","-112.206","56.0933",
        "8","CAN_073","-112.206","51.7704",
        "8","CAN_073","-103.428","51.7704",
        "8","CAN_073","-103.428","56.0933",
        "8","CAN_074","-112.348","60.096",
        "8","CAN_074","-112.348","55.7672",
        "8","CAN_074","-103.219","55.7672",
        "8","CAN_074","-103.219","60.096",
        "8","CAN_075","-112.641","64.1151",
        "8","CAN_075","-112.641","59.7418",
        "8","CAN_075","-102.991","59.7418",
        "8","CAN_075","-102.991","64.1151",
        "8","CAN_076","-112.815","68.1134",
        "8","CAN_076","-112.815","63.7568",
        "8","CAN_076","-102.625","63.7568",
        "8","CAN_076","-102.625","68.1134",
        "8","CAN_077","-112.416","72.0928",
        "8","CAN_077","-112.416","67.7664",
        "8","CAN_077","-102.689","67.7664",
        "8","CAN_077","-102.689","72.0928",
        "8","CAN_078","-112.938","76.0965",
        "8","CAN_078","-112.938","71.7536",
        "8","CAN_078","-102.129","71.7536",
        "8","CAN_078","-102.129","76.0965",
        "8","CAN_079","-113.505","80.097",
        "8","CAN_079","-113.505","75.7638",
        "8","CAN_079","-100.489","75.7638",
        "8","CAN_079","-100.489","80.097",
        "8","CAN_082","-120.176","52.1154",
        "8","CAN_082","-120.176","48.7501",
        "8","CAN_082","-111.545","48.7501",
        "8","CAN_082","-111.545","52.1154",
        "8","CAN_083","-120.267","56.0944",
        "8","CAN_083","-120.267","51.7575",
        "8","CAN_083","-111.403","51.7575",
        "8","CAN_083","-111.403","56.0944",
        "8","CAN_084","-120.413","60.0913",
        "8","CAN_084","-120.413","55.7689",
        "8","CAN_084","-111.235","55.7689",
        "8","CAN_084","-111.235","60.0913",
        "8","CAN_085","-120.69","64.1137",
        "8","CAN_085","-120.69","59.7606",
        "8","CAN_085","-110.979","59.7606",
        "8","CAN_085","-110.979","64.1137",
        "8","CAN_086","-121.028","68.1143",
        "8","CAN_086","-121.028","63.7546",
        "8","CAN_086","-110.589","63.7546",
        "8","CAN_086","-110.589","68.1143",
        "8","CAN_087","-120.506","72.1468",
        "8","CAN_087","-120.506","67.6176",
        "8","CAN_087","-110.901","67.6176",
        "8","CAN_087","-110.901","72.1468",
        "8","CAN_088","-120.944","76.1368",
        "8","CAN_088","-120.944","71.7116",
        "8","CAN_088","-110.065","71.7116",
        "8","CAN_088","-110.065","76.1368",
        "8","CAN_089","-121.287","79.121",
        "8","CAN_089","-121.287","75.7223",
        "8","CAN_089","-109.418","75.7223",
        "8","CAN_089","-109.418","79.121",
        "8","CAN_092","-128.131","52.1112",
        "8","CAN_092","-128.131","47.7618",
        "8","CAN_092","-119.529","47.7618",
        "8","CAN_092","-119.529","52.1112",
        "8","CAN_093","-128.264","56.1045",
        "8","CAN_093","-128.264","51.7577",
        "8","CAN_093","-119.359","51.7577",
        "8","CAN_093","-119.359","56.1045",
        "8","CAN_094","-128.365","60.0931",
        "8","CAN_094","-128.365","55.7643",
        "8","CAN_094","-119.168","55.7643",
        "8","CAN_094","-119.168","60.0931",
        "8","CAN_095","-128.608","64.1182",
        "8","CAN_095","-128.608","59.7374",
        "8","CAN_095","-118.89","59.7374",
        "8","CAN_095","-118.89","64.1182",
        "8","CAN_096","-128.878","68.2978",
        "8","CAN_096","-128.878","63.7568",
        "8","CAN_096","-117.701","63.7568",
        "8","CAN_096","-117.701","68.2978",
        "8","CAN_097","-128.526","72.1358",
        "8","CAN_097","-128.526","67.5391",
        "8","CAN_097","-118.911","67.5391",
        "8","CAN_097","-118.911","72.1358",
        "8","CAN_098","-128.621","76.0812",
        "8","CAN_098","-128.621","71.7273",
        "8","CAN_098","-118.884","71.7273",
        "8","CAN_098","-118.884","76.0812",
        "8","CAN_099","-124.944","77.0864",
        "8","CAN_099","-124.944","75.7746",
        "8","CAN_099","-118.975","75.7746",
        "8","CAN_099","-118.975","77.0864",
        "8","CAN_102","-130.11","52.0583",
        "8","CAN_102","-130.11","49.8098",
        "8","CAN_102","-127.587","49.8098",
        "8","CAN_102","-127.587","52.0583",
        "8","CAN_103","-134.316","56.0749",
        "8","CAN_103","-134.316","51.7618",
        "8","CAN_103","-127.464","51.7618",
        "8","CAN_103","-127.464","56.0749",
        "8","CAN_104","-136.37","60.1264",
        "8","CAN_104","-136.37","55.7637",
        "8","CAN_104","-127.271","55.7637",
        "8","CAN_104","-127.271","60.1264",
        "8","CAN_105","-136.565","64.1132",
        "8","CAN_105","-136.565","59.7677",
        "8","CAN_105","-126.982","59.7677",
        "8","CAN_105","-126.982","64.1132",
        "8","CAN_106","-136.821","68.1059",
        "8","CAN_106","-136.821","63.7547",
        "8","CAN_106","-126.627","63.7547",
        "8","CAN_106","-126.627","68.1059",
        "8","CAN_107","-136.981","71.0873",
        "8","CAN_107","-136.981","67.7497",
        "8","CAN_107","-127.352","67.7497",
        "8","CAN_107","-127.352","71.0873",
        "8","CAN_114","-140.374","60.096",
        "8","CAN_114","-140.374","58.7671",
        "8","CAN_114","-135.497","58.7671",
        "8","CAN_114","-135.497","60.096",
        "8","CAN_115","-141.198","64.0987",
        "8","CAN_115","-141.198","59.5939",
        "8","CAN_115","-134.99","59.5939",
        "8","CAN_115","-134.99","64.0987",
        "8","CAN_116","-141.32","68.0943",
        "8","CAN_116","-141.32","63.7589",
        "8","CAN_116","-134.634","63.7589",
        "8","CAN_116","-134.634","68.0943",
        "8","CAN_117","-144.467","70.1348",
        "8","CAN_117","-144.467","67.704",
        "8","CAN_117","-134.784","67.704",
        "8","CAN_117","-134.784","70.1348",
        "8","CAN_120","-74.2486","83.3307",
        "8","CAN_120","-74.2486","79.3301",
        "8","CAN_120","-52.1838","79.3301",
        "8","CAN_120","-52.1838","83.3307",
        "8","CAN_340","-91.6532","83.419",
        "8","CAN_340","-91.6532","79.6912",
        "8","CAN_340","-69.5067","79.6912",
        "8","CAN_340","-69.5067","83.419",
        "8","CAN_560","-105.269","82.1024",
        "8","CAN_560","-105.269","79.765",
        "8","CAN_560","-85.4058","79.765",
        "8","CAN_560","-85.4058","82.1024",
        "9","HoustonHeli","-95.7459","30.3279",
        "9","HoustonHeli","-95.7459","29.192",
        "9","HoustonHeli","-94.8043","29.192",
        "9","HoustonHeli","-94.8043","30.3279",
        "9","LosAngelesHeli","-119.408","34.5121",
        "9","LosAngelesHeli","-119.408","33.4028",
        "9","LosAngelesHeli","-116.873","33.4028",
        "9","LosAngelesHeli","-116.873","34.5121",
        "9","NewYorkHeli","-74.9892","41.2852",
        "9","NewYorkHeli","-74.9892","40.4193",
        "9","NewYorkHeli","-71.7762","40.4193",
        "9","NewYorkHeli","-71.7762","41.2852",
        "9","BaltimoreHeli","-77.4981","39.5532",
        "9","BaltimoreHeli","-77.4981","38.9311",
        "9","BaltimoreHeli","-75.9196","38.9311",
        "9","BaltimoreHeli","-75.9196","39.5532",
        "9","BostonHeli","-71.6703","42.8008",
        "9","BostonHeli","-71.6703","41.9707",
        "9","BostonHeli","-70.3129","41.9707",
        "9","BostonHeli","-70.3129","42.8008",
        "9","ChicagoHeli","-88.8356","42.4141",
        "9","ChicagoHeli","-88.8356","41.5524",
        "9","ChicagoHeli","-87.0943","41.5524",
        "9","ChicagoHeli","-87.0943","42.4141",
        "9","Dallas-FtWorthHeli","-97.8465","33.28",
        "9","Dallas-FtWorthHeli","-97.8465","32.4195",
        "9","Dallas-FtWorthHeli","-96.3146","32.4195",
        "9","Dallas-FtWorthHeli","-96.3146","33.28",
        "9","DetroitHeli","-84.5149","42.7545",
        "9","DetroitHeli","-84.5149","41.8897",
        "9","DetroitHeli","-82.765","41.8897",
        "9","DetroitHeli","-82.765","42.7545",
        "9","GrandCanyonGeneralAviationHeli","-114.512","36.7586",
        "9","GrandCanyonGeneralAviationHeli","-114.512","35.5832",
        "9","GrandCanyonGeneralAviationHeli","-111.281","35.5832",
        "9","GrandCanyonGeneralAviationHeli","-111.281","36.7586",
        "9","WashingtonHeli","-78.0641","39.1489",
        "9","WashingtonHeli","-78.0641","38.428",
        "9","WashingtonHeli","-76.4016","38.428",
        "9","WashingtonHeli","-76.4016","39.1489",
	"13","AtlantaFLY","-85.3059","34.3123",
	"13","AtlantaFLY","-85.3059","32.977",
	"13","AtlantaFLY","-83.5943","32.977",
	"13","AtlantaFLY","-83.5943","34.3123",
	"13","Baltimore-WashingtonFLY","-78.3434","39.8038",
	"13","Baltimore-WashingtonFLY","-78.3434","38.1823",
	"13","Baltimore-WashingtonFLY","-75.7541","38.1823",
	"13","Baltimore-WashingtonFLY","-75.7541","39.8038",
	"13","CharlotteFLY","-81.8296","35.9018",
	"13","CharlotteFLY","-81.8296","34.5827",
	"13","CharlotteFLY","-80.0851","34.5827",
	"13","CharlotteFLY","-80.0851","35.9018",
	"13","ChicagoFLY","-89.0853","42.4994",
	"13","ChicagoFLY","-89.0853","41.4563",
	"13","ChicagoFLY","-87.1554","41.4563",
	"13","ChicagoFLY","-87.1554","42.4994",
	"13","CincinnatiFLY","-85.5418","40.0801",
	"13","CincinnatiFLY","-85.5418","38.4711",
	"13","CincinnatiFLY","-83.6914","38.4711",
	"13","CincinnatiFLY","-83.6914","40.0801",
	"13","Dallas-FtWorthFLY","-98.2249","33.6733",
	"13","Dallas-FtWorthFLY","-98.2249","32.0499",
	"13","Dallas-FtWorthFLY","-95.8753","32.0499",
	"13","Dallas-FtWorthFLY","-95.8753","33.6733",
	"13","DenverFLY","-105.587","40.5717",
	"13","DenverFLY","-105.587","39.2426",
	"13","DenverFLY","-103.723","39.2426",
	"13","DenverFLY","-103.723","40.5717",
	"13","DetroitFLY","-84.2867","42.8606",
	"13","DetroitFLY","-84.2867","41.6292",
	"13","DetroitFLY","-82.3775","41.6292",
	"13","DetroitFLY","-82.3775","42.8606",
	"13","HoustonFLY","-96.146","30.51",
	"13","HoustonFLY","-96.146","29.0892",
	"13","HoustonFLY","-94.5123","29.0892",
	"13","HoustonFLY","-94.5123","30.51",
	"13","LasVegasFLY","-115.621","36.7353",
	"13","LasVegasFLY","-115.621","35.6955",
	"13","LasVegasFLY","-113.863","35.6955",
	"13","LasVegasFLY","-113.863","36.7353",
	"13","LosAngelesFLY","-119.151","34.5233",
	"13","LosAngelesFLY","-119.151","33.4099",
	"13","LosAngelesFLY","-116.791","33.4099",
	"13","LosAngelesFLY","-116.791","34.5233",
	"13","MiamiFLY","-81.1825","26.7674",
	"13","MiamiFLY","-81.1825","25.1148",
	"13","MiamiFLY","-79.32","25.1148",
	"13","MiamiFLY","-79.32","26.7674",
	"13","NewOrleansFLY","-91.2863","30.5872",
	"13","NewOrleansFLY","-91.2863","29.5568",
	"13","NewOrleansFLY","-89.6571","29.5568",
	"13","NewOrleansFLY","-89.6571","30.5872",
	"13","OrlandoFLY","-82.0368","29.2311",
	"13","OrlandoFLY","-82.0368","27.8287",
	"13","OrlandoFLY","-80.1406","27.8287",
	"13","OrlandoFLY","-80.1406","29.2311",
    "13","PhoenixFLY","-112.8704026","34.0841324",
    "13","PhoenixFLY","-112.855721","32.782496",
    "13","PhoenixFLY","-111.1819443","32.7827780",
    "13","PhoenixFLY","-111.1675846","34.0844160",
	"13","SaltLakeCityFLY","-112.909","41.4207",
	"13","SaltLakeCityFLY","-112.909","40.1204",
	"13","SaltLakeCityFLY","-111.016","40.1204",
	"13","SaltLakeCityFLY","-111.016","41.4207",
	"13","SanDiegoFLY","-117.982","33.6318",
	"13","SanDiegoFLY","-117.982","32.4948",
	"13","SanDiegoFLY","-116.287","32.4948",
	"13","SanDiegoFLY","-116.287","33.6318",
	"13","SanFranciscoFLY","-123.164","38.2685",
	"13","SanFranciscoFLY","-123.164","37.0071",
	"13","SanFranciscoFLY","-121.37","37.0071",
	"13","SanFranciscoFLY","-121.37","38.2685",
	"13","SeattleFLY","-123.198","48.0644",
	"13","SeattleFLY","-123.198","46.7498",
	"13","SeattleFLY","-121.529","46.7498",
	"13","SeattleFLY","-121.529","48.0644",
	"13","StLouisFLY","-91.0778","39.2754",
	"13","StLouisFLY","-91.0778","38.1819",
	"13","StLouisFLY","-89.6215","38.1819",
	"13","StLouisFLY","-89.6215","39.2754",
	"13","TampaFLY","-83.0969","28.5978",
	"13","TampaFLY","-83.0969","27.2913",
	"13","TampaFLY","-81.8278","27.2913",
	"13","TampaFLY","-81.8278","28.5978"
    };


    public float zoomRatio(String id1, String id2) {
        int idi1 = Integer.valueOf(id1);
        int idi2 = Integer.valueOf(id2);

        return (float)Math.pow(2, mZooms[idi2] - mZooms[idi1]);
    }



}
