"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.makeQueries = makeQueries;
var _errors = require("../helpers/errors");
var _waitFor = _interopRequireDefault(require("../waitFor"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
const deprecatedKeys = ['timeout', 'interval', 'stackTraceError'];

// The WaitForOptions has been moved to the second option param of findBy* methods with the adding of TextMatchOptions
// To make the migration easier and avoid a breaking change, keep reading this options from the first param but warn
function extractDeprecatedWaitForOptions(options) {
  if (!options) {
    return undefined;
  }
  const waitForOptions = {
    timeout: options.timeout,
    interval: options.interval,
    stackTraceError: options.stackTraceError
  };
  deprecatedKeys.forEach(key => {
    const option = options[key];
    if (option) {
      // eslint-disable-next-line no-console
      console.warn(`Use of option "${key}" in a findBy* query options (2nd parameter) is deprecated. Please pass this option in the waitForOptions (3rd parameter). 
Example: 

  findByText(text, {}, { ${key}: ${option.toString()} })`);
    }
  });
  return waitForOptions;
}
function makeQueries(queryAllByQuery, getMissingError, getMultipleError) {
  function getAllByQuery(instance) {
    return function getAllFn(predicate, options) {
      const results = queryAllByQuery(instance)(predicate, options);
      if (results.length === 0) {
        throw new _errors.ErrorWithStack(getMissingError(predicate, options), getAllFn);
      }
      return results;
    };
  }
  function queryByQuery(instance) {
    return function singleQueryFn(predicate, options) {
      const results = queryAllByQuery(instance)(predicate, options);
      if (results.length > 1) {
        throw new _errors.ErrorWithStack(getMultipleError(predicate, options), singleQueryFn);
      }
      if (results.length === 0) {
        return null;
      }
      return results[0];
    };
  }
  function getByQuery(instance) {
    return function getFn(predicate, options) {
      const results = queryAllByQuery(instance)(predicate, options);
      if (results.length > 1) {
        throw new _errors.ErrorWithStack(getMultipleError(predicate, options), getFn);
      }
      if (results.length === 0) {
        throw new _errors.ErrorWithStack(getMissingError(predicate, options), getFn);
      }
      return results[0];
    };
  }
  function findAllByQuery(instance) {
    return function findAllFn(predicate, queryOptions, waitForOptions = {}) {
      const deprecatedWaitForOptions = extractDeprecatedWaitForOptions(queryOptions);
      return (0, _waitFor.default)(() => getAllByQuery(instance)(predicate, queryOptions), {
        ...deprecatedWaitForOptions,
        ...waitForOptions
      });
    };
  }
  function findByQuery(instance) {
    return function findFn(predicate, queryOptions, waitForOptions = {}) {
      const deprecatedWaitForOptions = extractDeprecatedWaitForOptions(queryOptions);
      return (0, _waitFor.default)(() => getByQuery(instance)(predicate, queryOptions), {
        ...deprecatedWaitForOptions,
        ...waitForOptions
      });
    };
  }
  return {
    getBy: getByQuery,
    getAllBy: getAllByQuery,
    queryBy: queryByQuery,
    queryAllBy: queryAllByQuery,
    findBy: findByQuery,
    findAllBy: findAllByQuery
  };
}
//# sourceMappingURL=makeQueries.js.map