"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var CSVError_1 = __importDefault(require("./CSVError"));
var set_1 = __importDefault(require("lodash/set"));
var numReg = /^[-+]?[0-9]*\.?[0-9]+([eE][-+]?[0-9]+)?$/;
function default_1(csvRows, conv) {
    var res = [];
    for (var i = 0, len = csvRows.length; i < len; i++) {
        var r = processRow(csvRows[i], conv, i);
        if (r) {
            res.push(r);
        }
    }
    return res;
}
exports.default = default_1;
;
function processRow(row, conv, index) {
    if (conv.parseParam.checkColumn && conv.parseRuntime.headers && row.length !== conv.parseRuntime.headers.length) {
        throw (CSVError_1.default.column_mismatched(conv.parseRuntime.parsedLineNumber + index));
    }
    var headRow = conv.parseRuntime.headers || [];
    var resultRow = convertRowToJson(row, headRow, conv);
    if (resultRow) {
        return resultRow;
    }
    else {
        return null;
    }
}
function convertRowToJson(row, headRow, conv) {
    var hasValue = false;
    var resultRow = {};
    for (var i = 0, len = row.length; i < len; i++) {
        var item = row[i];
        if (conv.parseParam.ignoreEmpty && item === '') {
            continue;
        }
        hasValue = true;
        var head = headRow[i];
        if (!head || head === "") {
            head = headRow[i] = "field" + (i + 1);
        }
        var convFunc = getConvFunc(head, i, conv);
        if (convFunc) {
            var convRes = convFunc(item, head, resultRow, row, i);
            if (convRes !== undefined) {
                setPath(resultRow, head, convRes, conv, i);
            }
        }
        else {
            // var flag = getFlag(head, i, param);
            // if (flag === 'omit') {
            //   continue;
            // }
            if (conv.parseParam.checkType) {
                var convertFunc = checkType(item, head, i, conv);
                item = convertFunc(item);
            }
            if (item !== undefined) {
                setPath(resultRow, head, item, conv, i);
            }
        }
    }
    if (hasValue) {
        return resultRow;
    }
    else {
        return null;
    }
}
var builtInConv = {
    "string": stringType,
    "number": numberType,
    "omit": function () { }
};
function getConvFunc(head, i, conv) {
    if (conv.parseRuntime.columnConv[i] !== undefined) {
        return conv.parseRuntime.columnConv[i];
    }
    else {
        var flag = conv.parseParam.colParser[head];
        if (flag === undefined) {
            return conv.parseRuntime.columnConv[i] = null;
        }
        if (typeof flag === "object") {
            flag = flag.cellParser || "string";
        }
        if (typeof flag === "string") {
            flag = flag.trim().toLowerCase();
            var builtInFunc = builtInConv[flag];
            if (builtInFunc) {
                return conv.parseRuntime.columnConv[i] = builtInFunc;
            }
            else {
                return conv.parseRuntime.columnConv[i] = null;
            }
        }
        else if (typeof flag === "function") {
            return conv.parseRuntime.columnConv[i] = flag;
        }
        else {
            return conv.parseRuntime.columnConv[i] = null;
        }
    }
}
function setPath(resultJson, head, value, conv, headIdx) {
    if (!conv.parseRuntime.columnValueSetter[headIdx]) {
        if (conv.parseParam.flatKeys) {
            conv.parseRuntime.columnValueSetter[headIdx] = flatSetter;
        }
        else {
            if (head.indexOf(".") > -1) {
                var headArr = head.split(".");
                var jsonHead = true;
                while (headArr.length > 0) {
                    var headCom = headArr.shift();
                    if (headCom.length === 0) {
                        jsonHead = false;
                        break;
                    }
                }
                if (!jsonHead || conv.parseParam.colParser[head] && conv.parseParam.colParser[head].flat) {
                    conv.parseRuntime.columnValueSetter[headIdx] = flatSetter;
                }
                else {
                    conv.parseRuntime.columnValueSetter[headIdx] = jsonSetter;
                }
            }
            else {
                conv.parseRuntime.columnValueSetter[headIdx] = flatSetter;
            }
        }
    }
    if (conv.parseParam.nullObject === true && value === "null") {
        value = null;
    }
    conv.parseRuntime.columnValueSetter[headIdx](resultJson, head, value);
    // flatSetter(resultJson, head, value);
}
function flatSetter(resultJson, head, value) {
    resultJson[head] = value;
}
function jsonSetter(resultJson, head, value) {
    set_1.default(resultJson, head, value);
}
function checkType(item, head, headIdx, conv) {
    if (conv.parseRuntime.headerType[headIdx]) {
        return conv.parseRuntime.headerType[headIdx];
    }
    else if (head.indexOf('number#!') > -1) {
        return conv.parseRuntime.headerType[headIdx] = numberType;
    }
    else if (head.indexOf('string#!') > -1) {
        return conv.parseRuntime.headerType[headIdx] = stringType;
    }
    else if (conv.parseParam.checkType) {
        return conv.parseRuntime.headerType[headIdx] = dynamicType;
    }
    else {
        return conv.parseRuntime.headerType[headIdx] = stringType;
    }
}
function numberType(item) {
    var rtn = parseFloat(item);
    if (isNaN(rtn)) {
        return item;
    }
    return rtn;
}
function stringType(item) {
    return item.toString();
}
function dynamicType(item) {
    var trimed = item.trim();
    if (trimed === "") {
        return stringType(item);
    }
    if (numReg.test(trimed)) {
        return numberType(item);
    }
    else if (trimed.length === 5 && trimed.toLowerCase() === "false" || trimed.length === 4 && trimed.toLowerCase() === "true") {
        return booleanType(item);
    }
    else if (trimed[0] === "{" && trimed[trimed.length - 1] === "}" || trimed[0] === "[" && trimed[trimed.length - 1] === "]") {
        return jsonType(item);
    }
    else {
        return stringType(item);
    }
}
function booleanType(item) {
    var trimed = item.trim();
    if (trimed.length === 5 && trimed.toLowerCase() === "false") {
        return false;
    }
    else {
        return true;
    }
}
function jsonType(item) {
    try {
        return JSON.parse(item);
    }
    catch (e) {
        return item;
    }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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