import 'dart:async';

import 'package:animations/animations.dart';
import 'package:chopper/chopper.dart' show Response;
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:infinite_scroll_pagination/infinite_scroll_pagination.dart';
import 'package:intl/intl.dart';
import 'package:logging/logging.dart';
import 'package:provider/provider.dart';
import 'package:waterflyiii/auth.dart';
import 'package:waterflyiii/extensions.dart';
import 'package:waterflyiii/generated/l10n/app_localizations.dart';
import 'package:waterflyiii/generated/swagger_fireflyiii_api/firefly_iii.swagger.dart';
import 'package:waterflyiii/pages/home.dart';
import 'package:waterflyiii/pages/home/transactions/filter.dart';
import 'package:waterflyiii/pages/transaction.dart';
import 'package:waterflyiii/pages/transaction/delete.dart';
import 'package:waterflyiii/settings.dart';
import 'package:waterflyiii/stock.dart';
import 'package:waterflyiii/timezonehandler.dart';
import 'package:waterflyiii/widgets/listview_pagedchildbuilder.dart';

class TransactionSum {
  double withdrawals = 0;
  double deposits = 0;
  double transfers = 0;

  double get total => deposits - withdrawals;
}

class HomeTransactions extends StatefulWidget {
  const HomeTransactions({super.key, this.filters});

  final TransactionFilters? filters;

  @override
  State<HomeTransactions> createState() => _HomeTransactionsState();
}

class _HomeTransactionsState extends State<HomeTransactions>
    with AutomaticKeepAliveClientMixin {
  final Logger log = Logger("Pages.Home.Transaction");

  final int _numberOfPostsPerRequest = 50;
  late PagingState<int, TransactionRead> _pagingState;

  DateTime? _lastDate;
  List<int> _rowsWithDate = <int>[];
  late TransStock _stock;
  late TimeZoneHandler _tzHandler;

  TransactionSum _txSum = TransactionSum();

  final TransactionFilters _filters = TransactionFilters();
  final ValueNotifier<bool> _tagsHidden = ValueNotifier<bool>(false);
  final Map<String, double> _runningBalancesByTransactionId =
      <String, double>{};
  double? _lastCalculatedBalance;

  bool _isRevenueOrExpense(ShortAccountTypeProperty? type) {
    return type == ShortAccountTypeProperty.revenue ||
        type == ShortAccountTypeProperty.expense;
  }

  @override
  void initState() {
    super.initState();

    _tzHandler = context.read<FireflyService>().tzHandler;
    _pagingState = PagingState<int, TransactionRead>();
    _tagsHidden.value = context.read<SettingsProvider>().hideTags;

    // Only add filter button when in own tab
    if (widget.filters == null) {
      WidgetsBinding.instance.addPostFrameCallback((_) {
        context.read<PageActions>().set(widget.key!, <Widget>[
          ValueListenableBuilder<bool>(
            valueListenable: _tagsHidden,
            builder:
                (BuildContext context, bool value, _) => IconButton(
                  icon: const Icon(Icons.bookmarks_outlined),
                  selectedIcon: Icon(
                    Icons.bookmarks,
                    color: Theme.of(context).colorScheme.primary,
                  ),
                  isSelected: !value,
                  tooltip: S.of(context).homeTransactionsSettingsShowTags,
                  onPressed: () {
                    final SettingsProvider settings =
                        context.read<SettingsProvider>();
                    settings.hideTags = !settings.hideTags;
                    _tagsHidden.value = settings.hideTags;
                  },
                ),
          ),
          ChangeNotifierProvider<TransactionFilters>.value(
            value: _filters,
            builder:
                (BuildContext context, _) => IconButton(
                  icon: const Icon(Icons.filter_alt_outlined),
                  selectedIcon: Icon(
                    Icons.filter_alt,
                    color: Theme.of(context).colorScheme.primary,
                  ),
                  isSelected: context.watch<TransactionFilters>().hasFilters,
                  tooltip: S.of(context).homeTransactionsActionFilter,
                  onPressed: () async {
                    final TransactionFilters oldFilters = _filters.copyWith();
                    final SettingsProvider settings =
                        context.read<SettingsProvider>();
                    final bool oldShowFutureTXs = settings.showFutureTXs;
                    final TransactionDateFilter oldTransactionDateFilter =
                        settings.transactionDateFilter;
                    final bool? ok = await showDialog<bool>(
                      context: context,
                      builder:
                          (BuildContext context) => FilterDialog(
                            // passed by reference -> auto updated
                            filters: _filters,
                          ),
                    );
                    if (ok == null || !ok) {
                      if (settings.showFutureTXs != oldShowFutureTXs) {
                        settings.showFutureTXs = oldShowFutureTXs;
                        _txSum = TransactionSum();
                        setState(() {
                          _pagingState = _pagingState.reset();
                        });
                      }
                      if (settings.transactionDateFilter !=
                          oldTransactionDateFilter) {
                        await settings.setTransactionDateFilter(
                          oldTransactionDateFilter,
                        );
                        _txSum = TransactionSum();
                        setState(() {
                          _pagingState = _pagingState.reset();
                        });
                      }
                      _filters.account = oldFilters.account;
                      _filters.budget = oldFilters.budget;
                      _filters.category = oldFilters.category;
                      _filters.currency = oldFilters.currency;
                      _filters.text = oldFilters.text;
                      _filters.bill = oldFilters.bill;
                      _filters.tags = oldFilters.tags;

                      return;
                    }
                    if (oldFilters == _filters &&
                        settings.showFutureTXs == oldShowFutureTXs &&
                        settings.transactionDateFilter ==
                            oldTransactionDateFilter) {
                      return;
                    }
                    _filters.updateFilters();
                    _rowsWithDate = <int>[];
                    _lastDate = null;
                    _txSum = TransactionSum();
                    setState(() {
                      _pagingState = _pagingState.reset();
                    });
                  },
                ),
          ),
        ]);
      });
    }

    _stock = context.read<FireflyService>().transStock!;
    _stock.addListener(notifRefresh);
  }

  @override
  void dispose() {
    _stock.removeListener(notifRefresh);

    super.dispose();
  }

  void notifRefresh() {
    setState(() {
      _lastCalculatedBalance = null;
      _pagingState = _pagingState.reset();
    });
  }

  Future<void> _fetchPage() async {
    if (_pagingState.isLoading) return;

    final TransStock? stock = context.read<FireflyService>().transStock;
    if (stock == null) {
      throw Exception("Stock not available");
    }

    setState(() {
      _pagingState = _pagingState.copyWith(isLoading: true, error: null);
    });

    try {
      late List<TransactionRead> transactionList;

      final int pageKey = (_pagingState.keys?.last ?? 0) + 1;
      log.finest(
        "Getting page $pageKey (${_pagingState.pages?.length} pages loaded)",
      );

      if (widget.filters != null) {
        _filters.account = widget.filters!.account;
        _filters.text = widget.filters!.text;
        _filters.currency = widget.filters!.currency;
        _filters.category = widget.filters!.category;
        _filters.budget = widget.filters!.budget;
        _filters.bill = widget.filters!.bill;
        _filters.tags = widget.filters!.tags;
        _filters.updateFilters();
      }

      // Get start date
      late DateTime startDate;
      final DateTime now = _tzHandler.sNow().clearTime();
      switch (context.read<SettingsProvider>().transactionDateFilter) {
        case TransactionDateFilter.currentMonth:
          startDate = now.copyWith(day: 1);
          break;
        case TransactionDateFilter.currentYear:
          startDate = now.copyWith(month: 1, day: 1);
          break;
        case TransactionDateFilter.last30Days:
          startDate = now.subtract(const Duration(days: 30));
          break;
        case TransactionDateFilter.lastYear:
          startDate = now.copyWith(year: now.year - 1);
          break;
        default:
          // Don't use 0 seconds, as DST & stuff might throw an error:
          // "The start must be a date after 1970-01-02."
          startDate = DateTime.fromMillisecondsSinceEpoch(
            365 * 24 * 60 * 60 * 1000,
          );
          break;
      }

      // Faster than searching for an account, and also has cache (stock) behind
      // This search should never have additional filters!
      if (widget.filters?.account != null) {
        transactionList = await stock.getAccount(
          id: _filters.account!.id,
          page: pageKey,
          limit: _numberOfPostsPerRequest,
          type: TransactionTypeFilter.all,
          start:
              context.read<SettingsProvider>().showFutureTXs
                  ? null
                  : DateFormat('yyyy-MM-dd', 'en_US').format(startDate),
          end:
              context.read<SettingsProvider>().showFutureTXs
                  ? null
                  : DateFormat('yyyy-MM-dd', 'en_US').format(now),
        );
      } else if (_filters.hasFilters) {
        String query = _filters.text ?? "";
        if (_filters.account != null) {
          query = "account_id:${_filters.account!.id} $query";
        }
        if (_filters.currency != null) {
          query = "currency_is:${_filters.currency!.attributes.code} $query";
        }
        if (_filters.category != null) {
          query =
              (_filters.category!.id == "-1")
                  ? "has_no_category:true $query"
                  : "category_is:\"${_filters.category!.attributes.name}\" $query";
        }
        if (_filters.budget != null) {
          query =
              (_filters.budget!.id == "-1")
                  ? "has_no_budget:true $query"
                  : "budget_is:\"${_filters.budget!.attributes.name}\" $query";
        }
        if (_filters.bill != null) {
          query =
              (_filters.bill!.id == "-1")
                  ? "has_no_bill:true $query"
                  : "bill_is:\"${_filters.bill!.attributes.name}\" $query";
        }
        if (_filters.tags != null) {
          for (String tag in _filters.tags!.tags) {
            query = "tag_is:\"$tag\" $query";
          }
        }
        query =
            "date_after:${DateFormat('yyyy-MM-dd', 'en_US').format(startDate)} $query";
        if (!context.read<SettingsProvider>().showFutureTXs) {
          query = "date_before:today $query";
        }
        log.fine(() => "Search query: $query");
        transactionList = await stock.getSearch(
          query: query,
          page: pageKey,
          limit: _numberOfPostsPerRequest,
        );
      } else {
        transactionList = await stock.get(
          page: pageKey,
          limit: _numberOfPostsPerRequest,
          type: TransactionTypeFilter.all,
          end:
              context.read<SettingsProvider>().showFutureTXs
                  ? null
                  : DateFormat('yyyy-MM-dd', 'en_US').format(now),
          start: DateFormat('yyyy-MM-dd', 'en_US').format(startDate),
        );
      }

      if (_filters.account != null) {
        final AccountRead account = _filters.account!;
        // Attempt to retrieve the opening balance
        double balance =
            _lastCalculatedBalance ??
            double.tryParse(account.attributes.currentBalance!) ??
            0.0;
        // If the account is a revenue/expense account, we need to invert the balance
        if (_lastCalculatedBalance == null &&
            _isRevenueOrExpense(account.attributes.type)) {
          balance *= -1;
        }
        for (TransactionRead item in transactionList) {
          _runningBalancesByTransactionId[item.id] = balance;
          // Attempt to retrieve the transaction total amount
          for (TransactionSplit tx in item.attributes.transactions) {
            final double amount = double.tryParse(tx.amount) ?? 0.0;
            // Should never be the case
            if (amount == 0.0) {
              continue;
            }

            if (tx.type == TransactionTypeProperty.withdrawal) {
              balance += amount;
            } else if (tx.type == TransactionTypeProperty.transfer) {
              if (tx.destinationId == account.id) {
                balance -= amount;
              } else {
                balance += amount;
              }
            } else {
              balance -= amount;
            }
          }
          _lastCalculatedBalance = balance;
        }
      }

      for (TransactionRead item in transactionList) {
        for (TransactionSplit tx in item.attributes.transactions) {
          switch (tx.type) {
            case TransactionTypeProperty.deposit:
              _txSum.deposits += double.parse(tx.pcAmount ?? tx.amount);
              break;
            case TransactionTypeProperty.withdrawal:
              _txSum.withdrawals += double.parse(tx.pcAmount ?? tx.amount);
              break;
            case TransactionTypeProperty.transfer:
              _txSum.transfers += double.parse(tx.pcAmount ?? tx.amount);
              break;
            case TransactionTypeProperty.openingBalance:
            case TransactionTypeProperty.reconciliation:
            case TransactionTypeProperty.swaggerGeneratedUnknown:
            // do nothing, ignore
          }
        }
      }

      final bool isLastPage = transactionList.length < _numberOfPostsPerRequest;

      if (mounted) {
        setState(() {
          _pagingState = _pagingState.copyWith(
            pages: <List<TransactionRead>>[
              ...?_pagingState.pages,
              transactionList,
            ],
            keys: <int>[...?_pagingState.keys, pageKey],
            hasNextPage: !isLastPage,
            isLoading: false,
            error: null,
          );
        });
      }
    } catch (e, stackTrace) {
      log.severe("_fetchPage()", e, stackTrace);
      if (mounted) {
        setState(() {
          _pagingState = _pagingState.copyWith(error: e, isLoading: false);
        });
      }
    }
  }

  @override
  bool get wantKeepAlive => true;

  @override
  Widget build(BuildContext context) {
    log.finest(() => "build()");
    super.build(context);

    return RefreshIndicator.adaptive(
      onRefresh:
          () => Future<void>.sync(() {
            _rowsWithDate = <int>[];
            _lastDate = null;
            _txSum = TransactionSum();
            context.read<FireflyService>().transStock!.clear();
            setState(() {
              _lastCalculatedBalance = null;
              _pagingState = _pagingState.reset();
            });
          }),
      child: PagedListView<int, TransactionRead>(
        state: _pagingState,
        fetchNextPage: _fetchPage,
        builderDelegate: customPagedChildBuilderDelegate<TransactionRead>(
          itemBuilder: transactionRowBuilder,
          noMoreItemsIndicatorBuilder: (BuildContext context) {
            final CurrencyRead defaultCurrency =
                context.read<FireflyService>().defaultCurrency;
            return Padding(
              padding: const EdgeInsetsGeometry.symmetric(horizontal: 8),
              child: Column(
                children: <Widget>[
                  const Divider(),
                  Table(
                    //border: TableBorder.all(), // :DEBUG:
                    columnWidths: const <int, TableColumnWidth>{
                      0: FlexColumnWidth(),
                      1: FlexColumnWidth(),
                      2: FlexColumnWidth(),
                    },
                    children: <TableRow>[
                      TableRow(
                        children: <Widget>[
                          Text(
                            S.of(context).transactionTypeDeposit,
                            style: Theme.of(context).textTheme.bodyLarge!
                                .copyWith(fontWeight: FontWeight.bold),
                          ),
                          Text(
                            S.of(context).transactionTypeWithdrawal,
                            style: Theme.of(context).textTheme.bodyLarge!
                                .copyWith(fontWeight: FontWeight.bold),
                          ),
                          Text(
                            S.of(context).transactionTypeTransfer,
                            style: Theme.of(context).textTheme.bodyLarge!
                                .copyWith(fontWeight: FontWeight.bold),
                          ),
                        ],
                      ),

                      TableRow(
                        children: <Widget>[
                          Text(
                            defaultCurrency.fmt(_txSum.deposits),
                            style: Theme.of(
                              context,
                            ).textTheme.bodyMedium!.copyWith(
                              color: Colors.green,
                              fontWeight: FontWeight.bold,
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                          Text(
                            defaultCurrency.fmt(_txSum.withdrawals),
                            style: Theme.of(
                              context,
                            ).textTheme.bodyMedium!.copyWith(
                              color: Colors.red,
                              fontWeight: FontWeight.bold,
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                          Text(
                            defaultCurrency.fmt(_txSum.transfers),
                            style: Theme.of(
                              context,
                            ).textTheme.bodyMedium!.copyWith(
                              color: Colors.blue,
                              fontWeight: FontWeight.bold,
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                        ],
                      ),
                      const TableRow(
                        children: <Widget>[
                          SizedBox(height: 8),
                          SizedBox.shrink(),
                          SizedBox.shrink(),
                        ],
                      ),
                      TableRow(
                        children: <Widget>[
                          const SizedBox.shrink(),
                          Align(
                            alignment: Alignment.centerRight,
                            child: Text(
                              "${S.of(context).generalSum}:  ",
                              style: Theme.of(context).textTheme.bodyLarge!
                                  .copyWith(fontWeight: FontWeight.bold),
                            ),
                          ),
                          Text(
                            defaultCurrency.fmt(_txSum.total),
                            style: Theme.of(
                              context,
                            ).textTheme.bodyLarge!.copyWith(
                              color:
                                  _txSum.total < 0 ? Colors.red : Colors.green,
                              fontWeight: FontWeight.bold,
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                        ],
                      ),
                    ],
                  ),
                  // Padding for FAB
                  const SizedBox(height: 76),
                ],
              ),
            );
          },
        ),
      ),
      //itemExtent: 80,
    );
  }

  Widget transactionRowBuilder(
    BuildContext context,
    TransactionRead item,
    int index,
  ) {
    final List<TransactionSplit> transactions = item.attributes.transactions;
    if (transactions.isEmpty) {
      return Text(S.of(context).homeTransactionsEmpty);
    }
    String category = "";
    int categories = 0;
    String notes = "";
    bool hasAttachments = false;
    double amount = 0.0;
    final Map<String, double> foreignAmounts = <String, double>{};
    final Map<String, CurrencyRead> foreignCurrencies =
        <String, CurrencyRead>{};
    String foreignText = "";
    String sourceName = "";
    String destinationName = "";
    List<String> tags = <String>[];
    late bool reconciled;
    for (TransactionSplit trans in transactions) {
      if (trans.categoryName?.isNotEmpty ?? false) {
        categories++;
        if (category.isEmpty) {
          category = trans.categoryName!;
        }
      }
      if (trans.notes?.isNotEmpty ?? false) {
        if (notes.isNotEmpty) {
          notes += "\n";
        }
        notes += trans.notes!.trim();
      }
      if (trans.hasAttachments ?? false) {
        hasAttachments = true;
      }
      if (trans.tags?.isNotEmpty ?? false) {
        tags = trans.tags!;
      }
      amount += double.parse(trans.amount);
      if (trans.foreignAmount?.isNotEmpty ?? false) {
        final double foreignAmount = double.parse(trans.foreignAmount!);
        final String foreignSymbol =
            trans.foreignCurrencySymbol ?? trans.foreignCurrencyCode ?? "";
        if (foreignAmount != 0 && foreignSymbol.isNotEmpty) {
          foreignAmounts[foreignSymbol] =
              (foreignAmounts[foreignSymbol] ?? 0) + foreignAmount;
          foreignCurrencies[foreignSymbol] = CurrencyRead(
            id: trans.foreignCurrencyId ?? "0",
            type: "currencies",
            attributes: CurrencyProperties(
              code: trans.foreignCurrencyCode ?? "",
              name: "",
              symbol: trans.foreignCurrencySymbol ?? "",
              decimalPlaces: trans.foreignCurrencyDecimalPlaces,
            ),
          );
        }
      }
      if (sourceName == "") {
        sourceName = trans.sourceName!;
      } else if (sourceName != trans.sourceName!) {
        sourceName = S.of(context).generalMultiple;
      }
      if (destinationName == "") {
        destinationName = trans.destinationName!;
      } else if (destinationName != trans.destinationName!) {
        destinationName = S.of(context).generalMultiple;
      }
    }
    if (categories > 1) {
      category = S.of(context).homeTransactionsMultipleCategories(categories);
    }

    // Title
    late String title;
    if (item.attributes.groupTitle?.isNotEmpty ?? false) {
      title = item.attributes.groupTitle!;
    } else {
      title = transactions.first.description;
    }
    // Subtitle
    final List<InlineSpan> subtitle = <InlineSpan>[];
    if (hasAttachments) {
      subtitle.add(
        const WidgetSpan(
          baseline: TextBaseline.ideographic,
          alignment: PlaceholderAlignment.middle,
          child: Padding(
            padding: EdgeInsets.only(right: 2),
            child: Icon(Icons.attachment),
          ),
        ),
      );
    }
    if (transactions.first.type == TransactionTypeProperty.transfer) {
      subtitle.add(
        TextSpan(text: "(${S.of(context).transactionTypeTransfer})"),
      );
      subtitle.add(
        const WidgetSpan(
          baseline: TextBaseline.ideographic,
          alignment: PlaceholderAlignment.middle,
          child: Padding(
            padding: EdgeInsets.only(right: 2),
            child: Icon(Icons.arrow_right_alt),
          ),
        ),
      );
    }
    subtitle.add(
      TextSpan(
        text:
            (transactions.first.type == TransactionTypeProperty.withdrawal ||
                    transactions.first.type == TransactionTypeProperty.transfer)
                ? destinationName
                : sourceName,
      ),
    );
    subtitle.add(const TextSpan(text: "\n"));
    if (category.isNotEmpty) {
      subtitle.add(
        TextSpan(
          text: category,
          style: const TextStyle(fontStyle: FontStyle.italic),
        ),
      );
    }
    if (notes.isNotEmpty) {
      subtitle.add(
        TextSpan(
          text: "\n$notes",
          style: const TextStyle(fontStyle: FontStyle.italic),
        ),
      );
    }

    reconciled = transactions.first.reconciled ?? false;
    final CurrencyRead currency = CurrencyRead(
      id: transactions.first.currencyId ?? "0",
      type: "currencies",
      attributes: CurrencyProperties(
        code: transactions.first.currencyCode ?? "",
        name: transactions.first.currencyName ?? "",
        symbol: transactions.first.currencySymbol ?? "",
        decimalPlaces: transactions.first.currencyDecimalPlaces,
      ),
    );
    if (foreignAmounts.isNotEmpty) {
      foreignAmounts.forEach((String cur, double amount) {
        if (foreignCurrencies.containsKey(cur)) {
          foreignText += "${foreignCurrencies[cur]!.fmt(amount)} ";
        }
      });
      foreignText += " ";
    }

    // Account balance
    late double balance;
    if (_filters.account != null) {
      if (item.attributes.transactions.first.sourceBalanceAfter != null) {
        balance =
            double.tryParse(
              item.attributes.transactions.first.destinationId ==
                      _filters.account?.id
                  ? item.attributes.transactions.first.destinationBalanceAfter!
                  : item.attributes.transactions.first.sourceBalanceAfter!,
            ) ??
            0;
        if (_filters.account!.attributes.type !=
                ShortAccountTypeProperty.asset &&
            balance != 0) {
          balance *= -1;
        }
      } else {
        balance = _runningBalancesByTransactionId[item.id] ?? 0;
      }
    }

    Widget transactionWidget = OpenContainer(
      openBuilder:
          (BuildContext context, Function closedContainer) =>
              TransactionPage(transaction: item),
      openColor: Theme.of(context).cardColor,
      closedColor: Theme.of(context).cardColor,
      closedShape: const RoundedRectangleBorder(
        borderRadius: BorderRadius.only(
          topLeft: Radius.circular(16),
          bottomLeft: Radius.circular(16),
        ),
      ),
      closedElevation: 0,
      closedBuilder:
          (BuildContext context, Function openContainer) => GestureDetector(
            onLongPressStart: (LongPressStartDetails details) async {
              final Size screenSize = MediaQuery.of(context).size;
              final Offset offset = details.globalPosition;
              unawaited(HapticFeedback.vibrate());
              final Function? func = await showMenu<Function>(
                context: context,
                position: RelativeRect.fromLTRB(
                  offset.dx,
                  offset.dy,
                  screenSize.width - offset.dx,
                  screenSize.height - offset.dy,
                ),
                items: <PopupMenuEntry<Function>>[
                  PopupMenuItem<Function>(
                    value: () async {
                      final bool? ok = await Navigator.push(
                        context,
                        MaterialPageRoute<bool>(
                          builder:
                              (BuildContext context) => TransactionPage(
                                transaction: item,
                                clone: true,
                              ),
                        ),
                      );
                      if (ok ?? false) {
                        _rowsWithDate = <int>[];
                        _lastDate = null;
                        _txSum = TransactionSum();
                        if (context.mounted) {
                          context.read<FireflyService>().transStock!.clear();
                        }
                      }
                      setState(() {
                        _lastCalculatedBalance = null;
                        _pagingState = _pagingState.reset();
                      });
                    },
                    child: Row(
                      children: <Widget>[
                        const Icon(Icons.copy),
                        const SizedBox(width: 12),
                        Text(S.of(context).transactionDuplicate),
                      ],
                    ),
                  ),
                  const PopupMenuDivider(),
                  PopupMenuItem<Function>(
                    value: () async {
                      final FireflyIii api = context.read<FireflyService>().api;
                      final bool? ok = await showDialog<bool>(
                        context: context,
                        builder:
                            (BuildContext context) =>
                                const DeletionConfirmDialog(),
                      );
                      if (!(ok ?? false)) {
                        return;
                      }

                      await api.v1TransactionsIdDelete(id: item.id);
                      _rowsWithDate = <int>[];
                      _lastDate = null;
                      _txSum = TransactionSum();
                      if (context.mounted) {
                        context.read<FireflyService>().transStock!.clear();
                      }
                      setState(() {
                        _lastCalculatedBalance = null;
                        _pagingState = _pagingState.reset();
                      });
                    },
                    child: Row(
                      children: <Widget>[
                        const Icon(Icons.delete),
                        const SizedBox(width: 12),
                        Text(
                          MaterialLocalizations.of(context).deleteButtonTooltip,
                        ),
                      ],
                    ),
                  ),
                ],
                clipBehavior: Clip.hardEdge,
              );
              if (func == null) {
                return;
              }
              func();
            },
            child: ListTile(
              leading: CircleAvatar(
                foregroundColor: Colors.white,
                backgroundColor: transactions.first.type.color,
                child: Icon(transactions.first.type.icon),
              ),
              title: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: <Widget>[
                  // Front part
                  Expanded(
                    child: Text(
                      title,
                      maxLines: 1,
                      overflow: TextOverflow.ellipsis,
                    ),
                  ),
                  const SizedBox(width: 8),
                  // Trailing part
                  RichText(
                    textAlign: TextAlign.end,
                    maxLines: 1,
                    text: TextSpan(
                      style: Theme.of(context).textTheme.bodyMedium,
                      children: <InlineSpan>[
                        if (foreignText.isNotEmpty)
                          TextSpan(
                            text: foreignText,
                            style: Theme.of(
                              context,
                            ).textTheme.bodySmall!.copyWith(
                              color: Colors.blue,
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                        TextSpan(
                          text: currency.fmt(amount),
                          style: Theme.of(
                            context,
                          ).textTheme.titleMedium!.copyWith(
                            color:
                                transactions.first.type !=
                                        TransactionTypeProperty.reconciliation
                                    ? transactions.first.type.color
                                    : (transactions.first.sourceType ==
                                        AccountTypeProperty
                                            .reconciliationAccount)
                                    ? Colors.green
                                    : Colors.red,
                            fontFeatures: const <FontFeature>[
                              FontFeature.tabularFigures(),
                            ],
                          ),
                        ),
                      ],
                    ),
                  ),
                ],
              ),
              subtitle: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: <Widget>[
                  // Front part
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: <Widget>[
                        RichText(
                          overflow: TextOverflow.ellipsis,
                          maxLines: 2,
                          text: TextSpan(
                            style: Theme.of(context).textTheme.bodyMedium,
                            children: subtitle,
                          ),
                        ),
                        if (!context.watch<SettingsProvider>().hideTags &&
                            tags.isNotEmpty) ...<Widget>[
                          Wrap(
                            children:
                                tags
                                    .map(
                                      (String tag) => Card(
                                        child: Padding(
                                          padding: const EdgeInsets.all(6.0),
                                          child: Row(
                                            mainAxisSize: MainAxisSize.min,
                                            children: <Widget>[
                                              const Icon(
                                                Icons.label_outline,
                                                size: 16,
                                              ),
                                              const SizedBox(width: 5),
                                              Flexible(
                                                child: RichText(
                                                  overflow: TextOverflow.fade,
                                                  text: TextSpan(
                                                    style:
                                                        Theme.of(
                                                          context,
                                                        ).textTheme.bodyMedium,
                                                    text: tag,
                                                  ),
                                                ),
                                              ),
                                            ],
                                          ),
                                        ),
                                      ),
                                    )
                                    .toList(),
                          ),
                        ],
                      ],
                    ),
                  ),
                  // Trailing part
                  RichText(
                    textAlign: TextAlign.end,
                    maxLines: 1,
                    text: TextSpan(
                      style: Theme.of(context).textTheme.bodyMedium,
                      children: <InlineSpan>[
                        if (reconciled)
                          const WidgetSpan(
                            baseline: TextBaseline.ideographic,
                            alignment: PlaceholderAlignment.middle,
                            child: Padding(
                              padding: EdgeInsets.only(right: 2),
                              child: Icon(Icons.check),
                            ),
                          ),
                        if (_filters.account != null)
                          TextSpan(
                            text: currency.fmt(balance),
                            style: Theme.of(
                              context,
                            ).textTheme.bodyMedium!.copyWith(
                              fontFeatures: const <FontFeature>[
                                FontFeature.tabularFigures(),
                              ],
                            ),
                          ),
                        if (_filters.account == null)
                          TextSpan(
                            text: switch (transactions.first.type) {
                              TransactionTypeProperty.deposit =>
                                destinationName,
                              TransactionTypeProperty.openingBalance => "",
                              TransactionTypeProperty.reconciliation => "",
                              _ => sourceName,
                            },
                          ),
                      ],
                    ),
                  ),
                ],
              ),
              isThreeLine: true,
              shape: const RoundedRectangleBorder(
                borderRadius: BorderRadius.only(
                  topLeft: Radius.circular(16),
                  bottomLeft: Radius.circular(16),
                ),
              ),
            ),
          ),
      onClosed: (bool? refresh) async {
        if (_filters.account != null) {
          // Reset last balance calculated
          final FireflyIii api = context.read<FireflyService>().api;
          // Retrieve the account to get the current balance
          final Response<AccountSingle> respAccount = await api.v1AccountsIdGet(
            id: _filters.account!.id,
          );
          apiThrowErrorIfEmpty(respAccount, mounted ? context : null);
          final AccountRead account = respAccount.body!.data;
          _lastCalculatedBalance =
              double.tryParse(account.attributes.currentBalance!) ?? 0.0;
          // If the account is a revenue/expense account, we need to invert the balance
          if (_isRevenueOrExpense(account.attributes.type)) {
            _lastCalculatedBalance =
                _lastCalculatedBalance != null
                    ? _lastCalculatedBalance! * -1
                    : 0;
          }
        }

        if (refresh ?? false == true) {
          _rowsWithDate = <int>[];
          _lastDate = null;
          _txSum = TransactionSum();
          if (context.mounted) {
            context.read<FireflyService>().transStock!.clear();
          }
        }
        setState(() {
          _lastCalculatedBalance = null;
          _pagingState = _pagingState.reset();
        });
      },
    );

    // Date
    final DateTime date = _tzHandler.sTime(transactions.first.date).toLocal();
    // Show Date Banner when:
    // 1. _lastDate is not set (= first element)
    // 2. _lastDate has a different day than current date (= date changed) and
    //    is an earlier date (= scrolling down)
    // 3. index indicates we have to.
    if (_lastDate == null ||
        (_lastDate!.clearTime() != date.clearTime() &&
            date.clearTime().isBefore(_lastDate!.clearTime())) ||
        _rowsWithDate.contains(index)) {
      // Add date row
      _rowsWithDate.add(index);
      transactionWidget = Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: <Widget>[
          Padding(
            padding: const EdgeInsets.fromLTRB(16, 16, 0, 8),
            child: Text(
              DateFormat.yMMMMEEEEd().format(date),
              style: Theme.of(context).textTheme.labelLarge,
            ),
          ),
          transactionWidget,
        ],
      );
      _lastDate = date;
    }

    return transactionWidget;
  }
}
