// ignore_for_file: deprecated_member_use_from_same_package

import 'dart:convert' show Encoding, latin1, utf8;
import 'dart:typed_data' show ByteBuffer, Uint8List;

import 'package:euc/jis.dart';
import 'package:qs_dart/qs_dart.dart';
import 'package:qs_dart/src/utils.dart';
import 'package:test/test.dart';

import '../fixtures/data/empty_test_cases.dart';
import '../fixtures/dummy_enum.dart';

// Custom class that is neither a Map nor an Iterable
class CustomObject {
  CustomObject(this.value);

  final String value;

  String operator [](String key) {
    if (key == 'prop') return value;
    throw UnsupportedError('Only prop supported');
  }
}

void main() {
  group('encode', () {
    test('Default parameter initializations in _encode method', () {
      // This test targets lines 30-32 in encode.dart
      // We need to call QS.encode with null values for the parameters that have default initializations
      final result = QS.encode(
        {'a': 'b'},
        const EncodeOptions(
          // Force the code to use the default initializations
          listFormat: null,
          commaRoundTrip: null,
          format: Format.rfc3986,
        ),
      );
      expect(result, 'a=b');

      // Try another approach with a list to trigger the generateArrayPrefix default
      final result2 = QS.encode(
        {
          'a': ['b', 'c']
        },
        const EncodeOptions(
          // Force the code to use the default initializations
          listFormat: null,
          commaRoundTrip: null,
        ),
      );
      expect(result2, 'a%5B0%5D=b&a%5B1%5D=c');

      // Try with comma format to trigger the commaRoundTrip default
      final result3 = QS.encode(
        {
          'a': ['b', 'c']
        },
        const EncodeOptions(
          listFormat: ListFormat.comma,
          commaRoundTrip: null,
        ),
      );
      expect(result3, 'a=b%2Cc');
    });

    test('Default DateTime serialization', () {
      // This test targets line 60 in encode.dart
      // We need to call QS.encode with a DateTime and null serializeDate
      final dateTime = DateTime.utc(2023, 1, 1);
      final result = QS.encode(
        {'date': dateTime},
        const EncodeOptions(
          encode: false,
          serializeDate:
              null, // Force the code to use the default serialization
        ),
      );
      expect(result, 'date=2023-01-01T00:00:00.000Z');

      // Try another approach with a list of DateTimes
      final result2 = QS.encode(
        {
          'dates': [dateTime, dateTime]
        },
        const EncodeOptions(
          encode: false,
          serializeDate: null,
          listFormat: ListFormat.comma,
        ),
      );
      expect(
          result2, 'dates=2023-01-01T00:00:00.000Z,2023-01-01T00:00:00.000Z');
    });

    test('filter callback can expand custom objects into maps', () {
      final customObj = CustomObject('test');

      final result = QS.encode(
        {'obj': customObj},
        EncodeOptions(
          encode: false,
          filter: (prefix, value) =>
              value is CustomObject ? {'prop': value.value} : value,
        ),
      );

      expect(result, equals('obj[prop]=test'));
    });
    test('encodes a query string map', () {
      expect(QS.encode({'a': 'b'}), equals('a=b'));
      expect(QS.encode({'a': 1}), equals('a=1'));
      expect(QS.encode({'a': 1, 'b': 2}), equals('a=1&b=2'));
      expect(QS.encode({'a': 'A_Z'}), equals('a=A_Z'));
      expect(QS.encode({'a': '€'}), equals('a=%E2%82%AC'));
      expect(QS.encode({'a': ''}), equals('a=%EE%80%80'));
      expect(QS.encode({'a': 'א'}), equals('a=%D7%90'));
      expect(QS.encode({'a': '𐐷'}), equals('a=%F0%90%90%B7'));
    });

    test('encodes with default parameter values', () {
      // Test with ListFormat.comma but without setting commaRoundTrip
      // This should trigger the default initialization of commaRoundTrip
      const customOptions = EncodeOptions(
        listFormat: ListFormat.comma,
        encode: false,
      );

      // This should use the default commaRoundTrip value (false)
      expect(
        QS.encode({
          'a': ['b']
        }, customOptions),
        equals('a=b'),
      );

      // Test with explicitly set commaRoundTrip to true
      final customOptionsWithCommaRoundTrip = const EncodeOptions(
        listFormat: ListFormat.comma,
        commaRoundTrip: true,
        encode: false,
      );

      // This should append [] to single-item lists
      expect(
        QS.encode({
          'a': ['b']
        }, customOptionsWithCommaRoundTrip),
        equals('a[]=b'),
      );
    });

    test('encodes a list', () {
      expect(QS.encode([1234]), equals('0=1234'));
      expect(
        QS.encode(['lorem', 1234, 'ipsum']),
        equals('0=lorem&1=1234&2=ipsum'),
      );
    });

    test('encodes falsy values', () {
      expect(QS.encode({}), equals(''));
      expect(QS.encode(null), equals(''));
      expect(
        QS.encode(null, const EncodeOptions(strictNullHandling: true)),
        equals(''),
      );
      expect(QS.encode(false), equals(''));
      expect(QS.encode(0), equals(''));
    });

    test('encodes bigints', () {
      final BigInt three = BigInt.from(3);
      String encodeWithN(dynamic value, {Encoding? charset, Format? format}) {
        final String result = Utils.encode(value, format: format);
        return value is BigInt ? '${result}n' : result;
      }

      expect(QS.encode(three), equals(''));
      expect(QS.encode([three]), equals('0=3'));
      expect(
        QS.encode([three], EncodeOptions(encoder: encodeWithN)),
        equals('0=3n'),
      );
      expect(QS.encode({'a': three}), equals('a=3'));
      expect(
        QS.encode({'a': three}, EncodeOptions(encoder: encodeWithN)),
        equals('a=3n'),
      );
      expect(
        QS.encode(
            {
              'a': [three]
            },
            const EncodeOptions(
                encodeValuesOnly: true, listFormat: ListFormat.brackets)),
        equals('a[]=3'),
      );
      expect(
        QS.encode(
          {
            'a': [three],
          },
          EncodeOptions(
            encodeValuesOnly: true,
            encoder: encodeWithN,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=3n'),
      );
    });

    test(
      'encodes dot in key of map when encodeDotInKeys and allowDots is provided',
      () {
        expect(
          QS.encode(
            {
              'name.obj': {'first': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: false, encodeDotInKeys: false),
          ),
          equals('name.obj%5Bfirst%5D=John&name.obj%5Blast%5D=Doe'),
          reason: 'with allowDots false and encodeDotInKeys false',
        );

        expect(
          QS.encode(
            {
              'name.obj': {'first': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: true, encodeDotInKeys: false),
          ),
          equals('name.obj.first=John&name.obj.last=Doe'),
          reason: 'with allowDots true and encodeDotInKeys false',
        );

        expect(
          QS.encode(
            {
              'name.obj': {'first': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: false, encodeDotInKeys: true),
          ),
          equals('name%252Eobj%5Bfirst%5D=John&name%252Eobj%5Blast%5D=Doe'),
          reason: 'with allowDots false and encodeDotInKeys true',
        );

        expect(
          QS.encode(
            {
              'name.obj': {'first': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: true, encodeDotInKeys: true),
          ),
          equals('name%252Eobj.first=John&name%252Eobj.last=Doe'),
          reason: 'with allowDots true and encodeDotInKeys true',
        );

        expect(
          QS.encode(
            {
              'name.obj.subobject': {'first.godly.name': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: true, encodeDotInKeys: false),
          ),
          equals(
            'name.obj.subobject.first.godly.name=John&name.obj.subobject.last=Doe',
          ),
          reason: 'with allowDots true and encodeDotInKeys false',
        );

        expect(
          QS.encode(
            {
              'name.obj.subobject': {'first.godly.name': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: false, encodeDotInKeys: true),
          ),
          equals(
            'name%252Eobj%252Esubobject%5Bfirst.godly.name%5D=John&name%252Eobj%252Esubobject%5Blast%5D=Doe',
          ),
          reason: 'with allowDots false and encodeDotInKeys true',
        );

        expect(
          QS.encode(
            {
              'name.obj.subobject': {'first.godly.name': 'John', 'last': 'Doe'}
            },
            const EncodeOptions(allowDots: true, encodeDotInKeys: true),
          ),
          equals(
            'name%252Eobj%252Esubobject.first%252Egodly%252Ename=John&name%252Eobj%252Esubobject.last=Doe',
          ),
          reason: 'with allowDots true and encodeDotInKeys true',
        );
      },
    );

    test(
      'should encode dot in key of map, and automatically set allowDots to `true` when encodeDotInKeys is true and allowDots in undefined',
      () {
        expect(
          QS.encode(
            {
              'name.obj.subobject': {
                'first.godly.name': 'John',
                'last': 'Doe',
              },
            },
            const EncodeOptions(encodeDotInKeys: true),
          ),
          equals(
            'name%252Eobj%252Esubobject.first%252Egodly%252Ename=John&name%252Eobj%252Esubobject.last=Doe',
          ),
          reason: 'with allowDots undefined and encodeDotInKeys true',
        );
      },
    );

    test(
      'should encode dot in key of map when encodeDotInKeys and allowDots is provided, and nothing else when encodeValuesOnly is provided',
      () {
        expect(
          QS.encode(
            {
              'name.obj': {
                'first': 'John',
                'last': 'Doe',
              },
            },
            const EncodeOptions(
              encodeDotInKeys: true,
              allowDots: true,
              encodeValuesOnly: true,
            ),
          ),
          equals('name%2Eobj.first=John&name%2Eobj.last=Doe'),
        );

        expect(
          QS.encode(
            {
              'name.obj.subobject': {
                'first.godly.name': 'John',
                'last': 'Doe',
              },
            },
            const EncodeOptions(
              allowDots: true,
              encodeDotInKeys: true,
              encodeValuesOnly: true,
            ),
          ),
          equals(
            'name%2Eobj%2Esubobject.first%2Egodly%2Ename=John&name%2Eobj%2Esubobject.last=Doe',
          ),
        );
      },
    );

    test('adds query prefix', () {
      expect(
        QS.encode({'a': 'b'}, const EncodeOptions(addQueryPrefix: true)),
        equals('?a=b'),
      );
    });

    test(
      'with query prefix, outputs blank string given an empty map',
      () {
        expect(
          QS.encode({}, const EncodeOptions(addQueryPrefix: true)),
          equals(''),
        );
      },
    );

    test(
      'encodes nested falsy values',
      () {
        expect(
          QS.encode({
            'a': {
              'b': {'c': null}
            }
          }),
          equals('a%5Bb%5D%5Bc%5D='),
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': {'c': null}
              }
            },
            const EncodeOptions(strictNullHandling: true),
          ),
          equals('a%5Bb%5D%5Bc%5D'),
        );

        expect(
          QS.encode({
            'a': {
              'b': {'c': false}
            }
          }),
          equals('a%5Bb%5D%5Bc%5D=false'),
        );
      },
    );

    test(
      'encodes a nested map',
      () {
        expect(
          QS.encode({
            'a': {'b': 'c'}
          }),
          equals('a%5Bb%5D=c'),
        );

        expect(
          QS.encode({
            'a': {
              'b': {
                'c': {'d': 'e'}
              }
            }
          }),
          equals('a%5Bb%5D%5Bc%5D%5Bd%5D=e'),
        );
      },
    );

    test(
      'encodes a nested map with dots notation',
      () {
        expect(
          QS.encode(
            {
              'a': {'b': 'c'}
            },
            const EncodeOptions(allowDots: true),
          ),
          equals('a.b=c'),
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': {
                  'c': {'d': 'e'}
                }
              }
            },
            const EncodeOptions(allowDots: true),
          ),
          equals('a.b.c.d=e'),
        );
      },
    );

    test(
      'encodes a list value',
      () {
        expect(
          QS.encode(
            {
              'a': ['b', 'c', 'd']
            },
            const EncodeOptions(listFormat: ListFormat.indices),
          ),
          equals('a%5B0%5D=b&a%5B1%5D=c&a%5B2%5D=d'),
        );

        expect(
          QS.encode(
            {
              'a': ['b', 'c', 'd']
            },
            const EncodeOptions(listFormat: ListFormat.brackets),
          ),
          equals('a%5B%5D=b&a%5B%5D=c&a%5B%5D=d'),
        );

        expect(
          QS.encode(
            {
              'a': ['b', 'c', 'd']
            },
            const EncodeOptions(listFormat: ListFormat.comma),
          ),
          equals('a=b%2Cc%2Cd'),
        );

        expect(
          QS.encode(
            {
              'a': ['b', 'c', 'd']
            },
            const EncodeOptions(
                listFormat: ListFormat.comma, commaRoundTrip: true),
          ),
          equals('a=b%2Cc%2Cd'),
        );

        expect(
          QS.encode({
            'a': ['b', 'c', 'd']
          }),
          equals('a%5B0%5D=b&a%5B1%5D=c&a%5B2%5D=d'),
        );
      },
    );

    test(
      'omits nulls when asked',
      () {
        expect(
          QS.encode(
              {'a': 'b', 'c': null}, const EncodeOptions(skipNulls: true)),
          equals('a=b'),
        );

        expect(
          QS.encode(
            {
              'a': {'b': 'c', 'd': null}
            },
            const EncodeOptions(skipNulls: true),
          ),
          equals('a%5Bb%5D=c'),
        );
      },
    );

    test(
      'omits list indices when asked',
      () {
        expect(
          QS.encode(
            {
              'a': ['b', 'c', 'd']
            },
            const EncodeOptions(indices: false),
          ),
          equals('a=b&a=c&a=d'),
        );
      },
    );

    test(
      'omits map key/value pair when value is empty list',
      () {
        expect(
          QS.encode({'a': [], 'b': 'zz'}),
          equals('b=zz'),
        );
      },
    );

    test(
      'should not omit map key/value pair when value is empty list and when asked',
      () {
        expect(QS.encode({'a': [], 'b': 'zz'}), equals('b=zz'));

        expect(
          QS.encode({'a': [], 'b': 'zz'},
              const EncodeOptions(allowEmptyLists: false)),
          equals('b=zz'),
        );

        expect(
          QS.encode(
              {'a': [], 'b': 'zz'}, const EncodeOptions(allowEmptyLists: true)),
          equals('a[]&b=zz'),
        );
      },
    );

    test(
      'allowEmptyLists + strictNullHandling',
      () {
        expect(
          QS.encode(
            {'testEmptyList': []},
            const EncodeOptions(
              strictNullHandling: true,
              allowEmptyLists: true,
            ),
          ),
          equals('testEmptyList[]'),
        );
      },
    );

    group('encodes a list value with one item vs multiple items', () {
      test(
        'non-list item',
        () {
          expect(
            QS.encode(
              {'a': 'c'},
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.indices,
              ),
            ),
            equals('a=c'),
          );

          expect(
            QS.encode(
              {'a': 'c'},
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.brackets,
              ),
            ),
            equals('a=c'),
          );

          expect(
            QS.encode(
              {'a': 'c'},
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
              ),
            ),
            equals('a=c'),
          );

          expect(
            QS.encode(
              {'a': 'c'},
              const EncodeOptions(encodeValuesOnly: true),
            ),
            equals('a=c'),
          );
        },
      );

      test(
        'list with a single item',
        () {
          expect(
            QS.encode(
              {
                'a': ['c']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.indices,
              ),
            ),
            equals('a[0]=c'),
          );

          expect(
            QS.encode(
              {
                'a': ['c']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.brackets,
              ),
            ),
            equals('a[]=c'),
          );

          expect(
            QS.encode(
              {
                'a': ['c']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
              ),
            ),
            equals('a=c'),
          );

          expect(
            QS.encode(
              {
                'a': ['c']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
                commaRoundTrip: true,
              ),
            ),
            equals('a[]=c'),
          );

          expect(
            QS.encode(
              {
                'a': ['c']
              },
              const EncodeOptions(encodeValuesOnly: true),
            ),
            equals('a[0]=c'),
          );
        },
      );

      test(
        'list with multiple items',
        () {
          expect(
            QS.encode(
              {
                'a': ['c', 'd']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.indices,
              ),
            ),
            equals('a[0]=c&a[1]=d'),
          );

          expect(
            QS.encode(
              {
                'a': ['c', 'd']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.brackets,
              ),
            ),
            equals('a[]=c&a[]=d'),
          );

          expect(
            QS.encode(
              {
                'a': ['c', 'd']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
              ),
            ),
            equals('a=c,d'),
          );

          expect(
            QS.encode(
              {
                'a': ['c', 'd']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
                commaRoundTrip: true,
              ),
            ),
            equals('a=c,d'),
          );

          expect(
            QS.encode(
              {
                'a': ['c', 'd']
              },
              const EncodeOptions(encodeValuesOnly: true),
            ),
            equals('a[0]=c&a[1]=d'),
          );
        },
      );

      test(
        'list with multiple items with a comma inside',
        () {
          expect(
            QS.encode(
              {
                'a': ['c,d', 'e']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
              ),
            ),
            equals('a=c%2Cd,e'),
          );

          expect(
            QS.encode(
              {
                'a': ['c,d', 'e']
              },
              const EncodeOptions(listFormat: ListFormat.comma),
            ),
            equals('a=c%2Cd%2Ce'),
          );

          expect(
            QS.encode(
              {
                'a': ['c,d', 'e']
              },
              const EncodeOptions(
                encodeValuesOnly: true,
                listFormat: ListFormat.comma,
                commaRoundTrip: true,
              ),
            ),
            equals('a=c%2Cd,e'),
          );

          expect(
            QS.encode(
              {
                'a': ['c,d', 'e']
              },
              const EncodeOptions(
                listFormat: ListFormat.comma,
                commaRoundTrip: true,
              ),
            ),
            equals('a=c%2Cd%2Ce'),
          );
        },
      );
    });

    test('encodes a nested list value', () {
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd']
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[b][0]=c&a[b][1]=d'),
      );

      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd']
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[b][]=c&a[b][]=d'),
      );

      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd']
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a[b]=c,d'),
      );

      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd']
            }
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a[b][0]=c&a[b][1]=d'),
      );
    });

    test('encodes comma and empty list values', () {
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('a[0]=,&a[1]=&a[2]=c,d%'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
        ),
        equals('a[]=,&a[]=&a[]=c,d%'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.comma),
        ),
        equals('a=,,,c,d%'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
        ),
        equals('a=,&a=&a=c,d%'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=%2C&a[]=&a[]=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C,,c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&a=&a=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=%2C&a[1]=&a[2]=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a%5B%5D=%2C&a%5B%5D=&a%5B%5D=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C%2C%2Cc%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&a=&a=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%']
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a%5B0%5D=%2C&a%5B1%5D=&a%5B2%5D=c%2Cd%25'),
      );
    });

    test('encodes comma and empty non-list values', () {
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('a=,&b=&c=c,d%'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
        ),
        equals('a=,&b=&c=c,d%'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(encode: false, listFormat: ListFormat.comma),
        ),
        equals('a=,&b=&c=c,d%'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
        ),
        equals('a=,&b=&c=c,d%'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
              encode: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
    });

    test(
      'encodes a nested list value with dots notation',
      () {
        expect(
          QS.encode(
            {
              'a': {
                'b': ['c', 'd']
              }
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a.b[0]=c&a.b[1]=d'),
          reason: 'indices: encodes with dots + indices',
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': ['c', 'd']
              }
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a.b[]=c&a.b[]=d'),
          reason: 'brackets: encodes with dots + brackets',
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': ['c', 'd']
              }
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
            ),
          ),
          equals('a.b=c,d'),
          reason: 'comma: encodes with dots + comma',
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': ['c', 'd']
              }
            },
            const EncodeOptions(allowDots: true, encodeValuesOnly: true),
          ),
          equals('a.b[0]=c&a.b[1]=d'),
          reason: 'default: encodes with dots + indices',
        );
      },
    );

    test('encodes a map inside a list', () {
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'}
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0][b]=c'),
        reason: 'indices => indices',
      );

      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'}
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a[b]=c'),
        reason: 'repeat => repeat',
      );

      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'}
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[][b]=c'),
        reason: 'brackets => brackets',
      );

      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'}
            ]
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a[0][b]=c'),
        reason: 'default => indices',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1]
                }
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0][b][c][0]=1'),
        reason: 'indices => indices',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1]
                }
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a[b][c]=1'),
        reason: 'repeat => repeat',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1]
                }
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[][b][c][]=1'),
        reason: 'brackets => brackets',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1]
                }
              }
            ]
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a[0][b][c][0]=1'),
        reason: 'default => indices',
      );
    });

    test(
      'encodes a list with mixed maps and primitives',
      () {
        expect(
          QS.encode(
            {
              'a': [
                {'b': 1},
                2,
                3
              ]
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a[0][b]=1&a[1]=2&a[2]=3'),
          reason: 'indices => indices',
        );

        expect(
          QS.encode(
            {
              'a': [
                {'b': 1},
                2,
                3
              ]
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a[][b]=1&a[]=2&a[]=3'),
          reason: 'brackets => brackets',
        );

        expect(
          QS.encode(
            {
              'a': [
                {'b': 1},
                2,
                3
              ]
            },
            const EncodeOptions(encodeValuesOnly: true),
          ),
          equals('a[0][b]=1&a[1]=2&a[2]=3'),
          reason: 'default => indices',
        );
      },
    );

    test(
      'encodes a map inside a list with dots notation',
      () {
        expect(
          QS.encode(
            {
              'a': [
                {'b': 'c'}
              ]
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a[0].b=c'),
          reason: 'indices => indices',
        );

        expect(
          QS.encode(
            {
              'a': [
                {'b': 'c'}
              ]
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a[].b=c'),
          reason: 'brackets => brackets',
        );

        expect(
          QS.encode(
            {
              'a': [
                {'b': 'c'}
              ]
            },
            const EncodeOptions(allowDots: true, encodeValuesOnly: true),
          ),
          equals('a[0].b=c'),
          reason: 'default => indices',
        );

        expect(
          QS.encode(
            {
              'a': [
                {
                  'b': {
                    'c': [1]
                  }
                }
              ]
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a[0].b.c[0]=1'),
          reason: 'indices => indices',
        );

        expect(
          QS.encode(
            {
              'a': [
                {
                  'b': {
                    'c': [1]
                  }
                }
              ]
            },
            const EncodeOptions(
              allowDots: true,
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a[].b.c[]=1'),
          reason: 'brackets => brackets',
        );

        expect(
          QS.encode(
            {
              'a': [
                {
                  'b': {
                    'c': [1]
                  }
                }
              ]
            },
            const EncodeOptions(allowDots: true, encodeValuesOnly: true),
          ),
          equals('a[0].b.c[0]=1'),
          reason: 'default => indices',
        );
      },
    );

    test(
      'does not omit map keys when indices = false',
      () {
        expect(
          QS.encode(
            {
              'a': [
                {'b': 'c'}
              ]
            },
            const EncodeOptions(indices: false),
          ),
          equals('a%5Bb%5D=c'),
        );
      },
    );

    test('uses indices notation for lists when indices=true', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c']
          },
          const EncodeOptions(indices: true),
        ),
        equals('a%5B0%5D=b&a%5B1%5D=c'),
      );
    });

    test(
      'uses indices notation for lists when no listFormat is specified',
      () {
        expect(
          QS.encode({
            'a': ['b', 'c']
          }),
          equals('a%5B0%5D=b&a%5B1%5D=c'),
        );
      },
    );

    test(
      'uses indices notation for lists when listFormat=indices',
      () {
        expect(
          QS.encode(
            {
              'a': ['b', 'c']
            },
            const EncodeOptions(listFormat: ListFormat.indices),
          ),
          equals('a%5B0%5D=b&a%5B1%5D=c'),
        );
      },
    );

    test(
      'uses repeat notation for lists when listFormat=repeat',
      () {
        expect(
          QS.encode(
            {
              'a': ['b', 'c']
            },
            const EncodeOptions(listFormat: ListFormat.repeat),
          ),
          equals('a=b&a=c'),
        );
      },
    );

    test(
      'uses brackets notation for lists when listFormat=brackets',
      () {
        expect(
          QS.encode(
            {
              'a': ['b', 'c']
            },
            const EncodeOptions(listFormat: ListFormat.brackets),
          ),
          equals('a%5B%5D=b&a%5B%5D=c'),
        );
      },
    );

    test(
      'encodes a complicated map',
      () {
        expect(
          QS.encode({
            'a': {'b': 'c', 'd': 'e'}
          }),
          equals('a%5Bb%5D=c&a%5Bd%5D=e'),
        );
      },
    );

    test(
      'encodes an empty value',
      () {
        expect(
          QS.encode({'a': ''}),
          equals('a='),
        );

        expect(
          QS.encode({'a': null}, const EncodeOptions(strictNullHandling: true)),
          equals('a'),
        );

        expect(
          QS.encode({'a': '', 'b': ''}),
          equals('a=&b='),
        );

        expect(
          QS.encode(
            {'a': null, 'b': ''},
            const EncodeOptions(strictNullHandling: true),
          ),
          equals('a&b='),
        );

        expect(
          QS.encode({
            'a': {'b': ''}
          }),
          equals('a%5Bb%5D='),
        );

        expect(
          QS.encode(
            {
              'a': {'b': null}
            },
            const EncodeOptions(strictNullHandling: true),
          ),
          equals('a%5Bb%5D'),
        );

        expect(
          QS.encode(
            {
              'a': {'b': null}
            },
            const EncodeOptions(strictNullHandling: false),
          ),
          equals('a%5Bb%5D='),
        );
      },
    );

    group('encodes an empty list in different listFormat', () {
      test('default parameters', () {
        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(encode: false),
          ),
          equals('b[0]=&c=c'),
        );
      });

      test('listFormat default', () {
        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(encode: false, listFormat: ListFormat.indices),
          ),
          equals('b[0]=&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
          ),
          equals('b[]=&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
          ),
          equals('b=&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(encode: false, listFormat: ListFormat.comma),
          ),
          equals('b=&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.comma,
              commaRoundTrip: true,
            ),
          ),
          equals('b[]=&c=c'),
        );
      });

      test('with strictNullHandling', () {
        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.brackets,
              strictNullHandling: true,
            ),
          ),
          equals('b[]&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.repeat,
              strictNullHandling: true,
            ),
          ),
          equals('b&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.comma,
              strictNullHandling: true,
            ),
          ),
          equals('b&c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.comma,
              strictNullHandling: true,
              commaRoundTrip: true,
            ),
          ),
          equals('b[]&c=c'),
        );
      });

      test('with skipNulls', () {
        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.indices,
              skipNulls: true,
            ),
          ),
          equals('c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.brackets,
              skipNulls: true,
            ),
          ),
          equals('c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.repeat,
              skipNulls: true,
            ),
          ),
          equals('c=c'),
        );

        expect(
          QS.encode(
            {
              'a': [],
              'b': [null],
              'c': 'c'
            },
            const EncodeOptions(
              encode: false,
              listFormat: ListFormat.comma,
              skipNulls: true,
            ),
          ),
          equals('c=c'),
        );
      });
    });

    test(
      'encodes a null map',
      () {
        final Map<String, dynamic> obj = {};
        obj['a'] = 'b';
        expect(QS.encode(obj), equals('a=b'));
      },
    );

    test(
      'returns an empty string for invalid input',
      () {
        expect(QS.encode(null), equals(''));
        expect(QS.encode(false), equals(''));
        expect(QS.encode(''), equals(''));
      },
    );

    test(
      'encodes a map with a null map as a child',
      () {
        final Map<String, dynamic> obj = {
          'a': {},
        };
        obj['a']['b'] = 'c';
        expect(QS.encode(obj), equals('a%5Bb%5D=c'));
      },
    );

    test(
      'url encodes values',
      () {
        expect(QS.encode({'a': 'b c'}), equals('a=b%20c'));
      },
    );

    test(
      'encodes a date',
      () {
        final DateTime now = DateTime.now();
        final String str = 'a=${Uri.encodeComponent(now.toIso8601String())}';
        expect(QS.encode({'a': now}), equals(str));
      },
    );

    test(
      'encodes the weird map from qs',
      () {
        expect(
          QS.encode({'my weird field': '~q1!2"\'w\$5&7/z8)?'}),
          equals('my%20weird%20field=~q1%212%22%27w%245%267%2Fz8%29%3F'),
        );
      },
    );

    test(
      'encodes boolean values',
      () {
        expect(QS.encode({'a': true}), equals('a=true'));
        expect(
          QS.encode({
            'a': {'b': true}
          }),
          equals('a%5Bb%5D=true'),
        );
        expect(QS.encode({'b': false}), equals('b=false'));
        expect(
          QS.encode({
            'b': {'c': false}
          }),
          equals('b%5Bc%5D=false'),
        );
      },
    );

    test(
      'encodes buffer values',
      () {
        expect(
          QS.encode({'a': utf8.encode('test').buffer}),
          equals('a=test'),
        );
        expect(
          QS.encode({
            'a': {'b': utf8.encode('test').buffer}
          }),
          equals('a%5Bb%5D=test'),
        );
      },
    );

    test(
      'encodes a map using an alternative delimiter',
      () {
        expect(
          QS.encode(
            {'a': 'b', 'c': 'd'},
            const EncodeOptions(delimiter: ';'),
          ),
          equals('a=b;c=d'),
        );
      },
    );

    test('does not crash when parsing circular references', () {
      final Map<String, dynamic> a = <String, dynamic>{};
      a['b'] = a;

      expect(
        () => QS.encode(
          {'foo[bar]': 'baz', 'foo[baz]': a},
        ),
        throwsA(isA<RangeError>()),
      );

      final Map<String, dynamic> circular = <String, dynamic>{'a': 'value'};
      circular['a'] = circular;
      expect(
        () => QS.encode(circular),
        throwsA(isA<RangeError>()),
      );

      final List<String> arr = ['a'];
      expect(
        () => QS.encode({'x': arr, 'y': arr}),
        returnsNormally,
      );
    });

    test('non-circular duplicated references can still work', () {
      final Map<String, dynamic> hourOfDay = {'function': 'hour_of_day'};

      final Map<String, dynamic> p1 = {
        'function': 'gte',
        'arguments': [hourOfDay, 0]
      };

      final Map<String, dynamic> p2 = {
        'function': 'lte',
        'arguments': [hourOfDay, 23]
      };

      expect(
        QS.encode(
          {
            'filters': {
              r'$and': [p1, p2]
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals(
          r'filters[$and][0][function]=gte&filters[$and][0][arguments][0][function]=hour_of_day&filters[$and][0][arguments][1]=0&filters[$and][1][function]=lte&filters[$and][1][arguments][0][function]=hour_of_day&filters[$and][1][arguments][1]=23',
        ),
      );

      expect(
        QS.encode(
          {
            'filters': {
              r'$and': [p1, p2]
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals(
          r'filters[$and][][function]=gte&filters[$and][][arguments][][function]=hour_of_day&filters[$and][][arguments][]=0&filters[$and][][function]=lte&filters[$and][][arguments][][function]=hour_of_day&filters[$and][][arguments][]=23',
        ),
      );

      expect(
        QS.encode(
          {
            'filters': {
              r'$and': [p1, p2]
            }
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals(
          r'filters[$and][function]=gte&filters[$and][arguments][function]=hour_of_day&filters[$and][arguments]=0&filters[$and][function]=lte&filters[$and][arguments][function]=hour_of_day&filters[$and][arguments]=23',
        ),
      );
    });

    test(
      'selects properties when filter=list',
      () {
        expect(
          QS.encode(
            {'a': 'b'},
            const EncodeOptions(filter: ['a']),
          ),
          equals('a=b'),
        );

        expect(
          QS.encode(
            {'a': 1},
            const EncodeOptions(filter: []),
          ),
          equals(''),
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': [1, 2, 3, 4],
                'c': 'd'
              },
              'c': 'f'
            },
            const EncodeOptions(
              filter: ['a', 'b', 0, 2],
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a%5Bb%5D%5B0%5D=1&a%5Bb%5D%5B2%5D=3'),
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': [1, 2, 3, 4],
                'c': 'd'
              },
              'c': 'f'
            },
            const EncodeOptions(
              filter: ['a', 'b', 0, 2],
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a%5Bb%5D%5B%5D=1&a%5Bb%5D%5B%5D=3'),
        );

        expect(
          QS.encode(
            {
              'a': {
                'b': [1, 2, 3, 4],
                'c': 'd'
              },
              'c': 'f'
            },
            const EncodeOptions(filter: ['a', 'b', 0, 2]),
          ),
          equals('a%5Bb%5D%5B0%5D=1&a%5Bb%5D%5B2%5D=3'),
        );
      },
    );

    test(
      'supports custom representations when filter=function',
      () {
        int calls = 0;
        final Map<String, dynamic> obj = {
          'a': 'b',
          'c': 'd',
          'e': {
            'f': DateTime.fromMillisecondsSinceEpoch(1257894000000),
          },
        };

        dynamic filterFunc(String prefix, dynamic value) {
          calls += 1;
          if (calls == 1) {
            expect(prefix, equals(''));
            expect(value, equals(obj));
          } else if (prefix == 'c') {
            expect(value, equals('d'));
            return null;
          } else if (value is DateTime) {
            expect(prefix, equals('e[f]'));
            return value.millisecondsSinceEpoch;
          }
          return value;
        }

        expect(
          QS.encode(obj, EncodeOptions(filter: filterFunc)),
          equals('a=b&c=&e%5Bf%5D=1257894000000'),
        );
        expect(calls, 5);
      },
    );

    test(
      'can disable uri encoding',
      () {
        expect(QS.encode({'a': 'b'}, const EncodeOptions(encode: false)),
            equals('a=b'));
        expect(
          QS.encode(
            {
              'a': {'b': 'c'}
            },
            const EncodeOptions(encode: false),
          ),
          equals('a[b]=c'),
        );
        expect(
          QS.encode(
            {'a': 'b', 'c': null},
            const EncodeOptions(encode: false, strictNullHandling: true),
          ),
          equals('a=b&c'),
        );
      },
    );

    test(
      'can sort the keys',
      () {
        int sort(a, b) => a.compareTo(b);
        expect(
          QS.encode(
            {'a': 'c', 'z': 'y', 'b': 'f'},
            EncodeOptions(sort: sort),
          ),
          equals('a=c&b=f&z=y'),
        );
        expect(
          QS.encode(
            {
              'a': 'c',
              'z': {'j': 'a', 'i': 'b'},
              'b': 'f'
            },
            EncodeOptions(sort: sort),
          ),
          equals('a=c&b=f&z%5Bi%5D=b&z%5Bj%5D=a'),
        );
      },
    );

    test(
      'can sort the keys at depth 3 or more too',
      () {
        int sort(a, b) => a.compareTo(b);
        expect(
          QS.encode(
            {
              'a': 'a',
              'z': {
                'zj': {'zjb': 'zjb', 'zja': 'zja'},
                'zi': {'zib': 'zib', 'zia': 'zia'}
              },
              'b': 'b'
            },
            EncodeOptions(sort: sort, encode: false),
          ),
          equals(
            'a=a&b=b&z[zi][zia]=zia&z[zi][zib]=zib&z[zj][zja]=zja&z[zj][zjb]=zjb',
          ),
        );
        expect(
          QS.encode(
            {
              'a': 'a',
              'z': {
                'zj': {'zjb': 'zjb', 'zja': 'zja'},
                'zi': {'zib': 'zib', 'zia': 'zia'}
              },
              'b': 'b'
            },
            const EncodeOptions(sort: null, encode: false),
          ),
          equals(
            'a=a&z[zj][zjb]=zjb&z[zj][zja]=zja&z[zi][zib]=zib&z[zi][zia]=zia&b=b',
          ),
        );
      },
    );

    test('can encode with custom encoding', () {
      String encode(dynamic str, {Encoding? charset, Format? format}) {
        if ((str as String?)?.isNotEmpty ?? false) {
          final Uint8List buf = Uint8List.fromList(ShiftJIS().encode(str!));
          final List<String> result = [
            for (int i = 0; i < buf.length; ++i) buf[i].toRadixString(16)
          ];
          return '%${result.join('%')}';
        }
        return '';
      }

      expect(
        QS.encode(
          {'県': '大阪府', '': ''},
          EncodeOptions(encoder: encode),
        ),
        equals('%8c%a7=%91%e5%8d%e3%95%7b&='),
      );
    });

    test('receives the default encoder as a second argument', () {
      Map<String, dynamic> obj = {
        'a': 1,
        'b': DateTime.now(),
        'c': true,
        'd': [1]
      };
      QS.encode(
        obj,
        EncodeOptions(
          encoder: (str, {Encoding? charset, Format? format}) {
            expect(str, anyOf(isA<String>(), isA<int>(), isA<bool>()));
            return '';
          },
        ),
      );
    });

    test('can use custom encoder for a buffer map', () {
      final ByteBuffer buf = Uint8List.fromList([1]).buffer;
      expect(
        QS.encode(
          {'a': buf},
          EncodeOptions(
            encoder: (buffer, {Encoding? charset, Format? format}) {
              if (buffer is String) {
                return buffer;
              }
              return String.fromCharCode(buffer.asUint8List()[0] + 97);
            },
          ),
        ),
        equals('a=b'),
      );

      expect(
        QS.encode(
          {
            'a': utf8.encode('a b').buffer,
          },
          EncodeOptions(
            encoder: (buffer, {Encoding? charset, Format? format}) =>
                buffer is ByteBuffer
                    ? utf8.decode(buffer.asUint8List())
                    : buffer,
          ),
        ),
        equals('a=a b'),
      );
    });

    test('serializeDate option', () {
      final date = DateTime.now();
      expect(
        QS.encode({'a': date}),
        equals('a=${date.toIso8601String().replaceAll(':', '%3A')}'),
      );

      expect(
        QS.encode(
          {'a': date},
          EncodeOptions(
              serializeDate: (d) => d.millisecondsSinceEpoch.toString()),
        ),
        equals('a=${date.millisecondsSinceEpoch}'),
      );

      final specificDate = DateTime.fromMillisecondsSinceEpoch(6);
      expect(
        QS.encode(
          {'a': specificDate},
          EncodeOptions(
            serializeDate: (d) => (d.millisecondsSinceEpoch * 7).toString(),
          ),
        ),
        equals('a=42'),
      );

      expect(
        QS.encode(
          {
            'a': [date]
          },
          EncodeOptions(
            serializeDate: (d) => d.millisecondsSinceEpoch.toString(),
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=${date.millisecondsSinceEpoch}'),
      );

      expect(
        QS.encode(
          {
            'a': [date]
          },
          EncodeOptions(
            serializeDate: (d) => d.millisecondsSinceEpoch.toString(),
            listFormat: ListFormat.comma,
            commaRoundTrip: true,
          ),
        ),
        equals('a%5B%5D=${date.millisecondsSinceEpoch}'),
      );
    });

    test('RFC 1738 serialization', () {
      expect(
        QS.encode({'a': 'b c'}, const EncodeOptions(format: Format.rfc1738)),
        equals('a=b+c'),
      );
      expect(
        QS.encode({'a b': 'c d'}, const EncodeOptions(format: Format.rfc1738)),
        equals('a+b=c+d'),
      );
      expect(
        QS.encode(
          {'a b': utf8.encode('a b').buffer},
          const EncodeOptions(format: Format.rfc1738),
        ),
        equals('a+b=a+b'),
      );

      expect(
        QS.encode(
          {'foo(ref)': 'bar'},
          const EncodeOptions(format: Format.rfc1738),
        ),
        equals('foo(ref)=bar'),
      );
    });

    test('RFC 3986 spaces serialization', () {
      expect(
        QS.encode({'a': 'b c'}, const EncodeOptions(format: Format.rfc3986)),
        equals('a=b%20c'),
      );
      expect(
        QS.encode({'a b': 'c d'}, const EncodeOptions(format: Format.rfc3986)),
        equals('a%20b=c%20d'),
      );
      expect(
        QS.encode(
          {'a b': utf8.encode('a b').buffer},
          const EncodeOptions(format: Format.rfc3986),
        ),
        equals('a%20b=a%20b'),
      );
    });

    test('Backward compatibility to RFC 3986', () {
      expect(QS.encode({'a': 'b c'}), equals('a=b%20c'));
      expect(
        QS.encode(
          {'a b': utf8.encode('a b').buffer},
        ),
        equals('a%20b=a%20b'),
      );
    });

    test('encodeValuesOnly', () {
      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e=f'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=b&c[0]=d&c[1]=e%3Df&f[0][0]=g&f[1][0]=h'),
        reason: 'encodeValuesOnly + indices',
      );

      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e=f'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=b&c[]=d&c[]=e%3Df&f[][]=g&f[][]=h'),
        reason: 'encodeValuesOnly + brackets',
      );

      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e=f'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=b&c=d&c=e%3Df&f=g&f=h'),
        reason: 'encodeValuesOnly + repeat',
      );

      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=b&c%5B0%5D=d&c%5B1%5D=e&f%5B0%5D%5B0%5D=g&f%5B1%5D%5B0%5D=h'),
        reason: 'no encodeValuesOnly + indices',
      );

      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=b&c%5B%5D=d&c%5B%5D=e&f%5B%5D%5B%5D=g&f%5B%5D%5B%5D=h'),
        reason: 'no encodeValuesOnly + brackets',
      );

      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e'],
            'f': [
              ['g'],
              ['h']
            ]
          },
          const EncodeOptions(
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=b&c=d&c=e&f=g&f=h'),
        reason: 'no encodeValuesOnly + repeat',
      );
    });

    test('encodeValuesOnly - strictNullHandling', () {
      expect(
        QS.encode(
          {
            'a': {'b': null}
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            strictNullHandling: true,
          ),
        ),
        equals('a[b]'),
      );
    });

    test('respects a charset of iso-8859-1', () {
      expect(
        QS.encode(
          {'æ': 'æ'},
          const EncodeOptions(charset: latin1),
        ),
        equals('%E6=%E6'),
      );
    });

    test(
      'encodes unrepresentable chars as numeric entities in iso-8859-1 mode',
      () {
        expect(
          QS.encode(
            {'a': '☺'},
            const EncodeOptions(charset: latin1),
          ),
          equals('a=%26%239786%3B'),
        );
      },
    );

    test(
      'respects an explicit charset of utf-8 (the default)',
      () {
        expect(
          QS.encode(
            {'a': 'æ'},
            const EncodeOptions(charset: utf8),
          ),
          equals('a=%C3%A6'),
        );
      },
    );

    test('`charsetSentinel` option', () {
      expect(
        QS.encode(
          {'a': 'æ'},
          const EncodeOptions(charsetSentinel: true, charset: utf8),
        ),
        equals('utf8=%E2%9C%93&a=%C3%A6'),
      );

      expect(
        QS.encode(
          {'a': 'æ'},
          const EncodeOptions(charsetSentinel: true, charset: latin1),
        ),
        equals('utf8=%26%2310003%3B&a=%E6'),
      );
    });

    test(
      'does not mutate the options argument',
      () {
        final EncodeOptions options = const EncodeOptions();
        QS.encode({}, options);
        expect(options, equals(const EncodeOptions()));
      },
    );

    test(
      'strictNullHandling works with custom filter',
      () {
        final options = EncodeOptions(
          strictNullHandling: true,
          filter: (String prefix, dynamic value) => value,
        );
        expect(QS.encode({'key': null}, options), equals('key'));
      },
    );

    test('objects inside lists', () {
      final Map<String, dynamic> obj = {
        'a': {
          'b': {'c': 'd', 'e': 'f'}
        }
      };
      final Map<String, dynamic> withList = {
        'a': {
          'b': [
            {'c': 'd', 'e': 'f'}
          ]
        }
      };

      expect(
        QS.encode(obj, const EncodeOptions(encode: false)),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no list, no listFormat',
      );

      expect(
        QS.encode(
          obj,
          const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no list, bracket',
      );

      expect(
        QS.encode(
          obj,
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no list, indices',
      );

      expect(
        QS.encode(
          obj,
          const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no list, repeat',
      );

      expect(
        QS.encode(
          obj,
          const EncodeOptions(encode: false, listFormat: ListFormat.comma),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no list, comma',
      );

      expect(
        QS.encode(
          withList,
          const EncodeOptions(encode: false),
        ),
        equals('a[b][0][c]=d&a[b][0][e]=f'),
        reason: 'list, no listFormat',
      );

      expect(
        QS.encode(
          withList,
          const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
        ),
        equals('a[b][][c]=d&a[b][][e]=f'),
        reason: 'list, bracket',
      );

      expect(
        QS.encode(
          withList,
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('a[b][0][c]=d&a[b][0][e]=f'),
        reason: 'list, indices',
      );

      expect(
        QS.encode(
          withList,
          const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'list, repeat',
      );
    });

    test('encodes lists with nulls', () {
      expect(
        QS.encode(
          {
            'a': [null, '2', null, null, '1']
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=&a[1]=2&a[2]=&a[3]=&a[4]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [null, '2', null, null, '1']
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=&a[]=2&a[]=&a[]=&a[]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [null, '2', null, null, '1']
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=&a=2&a=&a=&a=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              {
                'b': [
                  null,
                  null,
                  {'c': '1'}
                ]
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=&a[1][b][0]=&a[1][b][1]=&a[1][b][2][c]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              {
                'b': [
                  null,
                  null,
                  {'c': '1'}
                ]
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=&a[][b][]=&a[][b][]=&a[][b][][c]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              {
                'b': [
                  null,
                  null,
                  {'c': '1'}
                ]
              }
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=&a[b]=&a[b]=&a[b][c]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              [
                null,
                [
                  null,
                  null,
                  {'c': '1'}
                ]
              ]
            ]
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=&a[1][0]=&a[1][1][0]=&a[1][1][1]=&a[1][1][2][c]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              [
                null,
                [
                  null,
                  null,
                  {'c': '1'}
                ]
              ]
            ]
          },
          const EncodeOptions(
              encodeValuesOnly: true, listFormat: ListFormat.brackets),
        ),
        equals('a[]=&a[][]=&a[][][]=&a[][][]=&a[][][][c]=1'),
      );

      expect(
        QS.encode(
          {
            'a': [
              null,
              [
                null,
                [
                  null,
                  null,
                  {'c': '1'}
                ]
              ]
            ]
          },
          const EncodeOptions(
              encodeValuesOnly: true, listFormat: ListFormat.repeat),
        ),
        equals('a=&a=&a=&a=&a[c]=1'),
      );
    });

    test('encodes url', () {
      expect(
        QS.encode(
          {'url': 'https://example.com?foo=bar&baz=qux'},
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('url=https%3A%2F%2Fexample.com%3Ffoo%3Dbar%26baz%3Dqux'),
      );

      expect(
        QS.encode(
          {
            'url': Uri.https('example.com', '/some/path', {
              'foo': 'bar',
              'baz': 'qux',
            })
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals(
            'url=https%3A%2F%2Fexample.com%2Fsome%2Fpath%3Ffoo%3Dbar%26baz%3Dqux'),
      );
    });

    test('encodes Spatie map', () {
      expect(
        QS.encode(
          {
            'filters': {
              r'$or': [
                {
                  'date': {
                    r'$eq': '2020-01-01',
                  }
                },
                {
                  'date': {
                    r'$eq': '2020-01-02',
                  }
                }
              ],
              'author': {
                'name': {
                  r'$eq': 'John doe',
                },
              }
            }
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.brackets),
        ),
        equals(
          r'filters[$or][][date][$eq]=2020-01-01&filters[$or][][date][$eq]=2020-01-02&filters[author][name][$eq]=John doe',
        ),
      );

      expect(
        QS.encode(
          {
            'filters': {
              r'$or': [
                {
                  'date': {
                    r'$eq': '2020-01-01',
                  }
                },
                {
                  'date': {
                    r'$eq': '2020-01-02',
                  }
                }
              ],
              'author': {
                'name': {
                  r'$eq': 'John doe',
                },
              }
            }
          },
          const EncodeOptions(listFormat: ListFormat.brackets),
        ),
        equals(
          'filters%5B%24or%5D%5B%5D%5Bdate%5D%5B%24eq%5D=2020-01-01&filters%5B%24or%5D%5B%5D%5Bdate%5D%5B%24eq%5D=2020-01-02&filters%5Bauthor%5D%5Bname%5D%5B%24eq%5D=John%20doe',
        ),
      );
    });
  });

  group('encodes empty keys', () {
    for (Map<String, dynamic> element in emptyTestCases) {
      test(
        'encodes a map with empty string key with ${element['input']}',
        () {
          expect(
            QS.encode(
              element['withEmptyKeys'],
              const EncodeOptions(
                  encode: false, listFormat: ListFormat.indices),
            ),
            equals(element['stringifyOutput']['indices']),
            reason: 'test case: ${element['input']}, indices',
          );

          expect(
            QS.encode(
              element['withEmptyKeys'],
              const EncodeOptions(
                  encode: false, listFormat: ListFormat.brackets),
            ),
            equals(element['stringifyOutput']['brackets']),
            reason: 'test case: ${element['input']}, brackets',
          );

          expect(
            QS.encode(
              element['withEmptyKeys'],
              const EncodeOptions(encode: false, listFormat: ListFormat.repeat),
            ),
            equals(element['stringifyOutput']['repeat']),
            reason: 'test case: ${element['input']}, repeat',
          );
        },
      );
    }

    test('edge case with map/lists', () {
      expect(
        QS.encode(
          {
            '': {
              '': [2, 3]
            }
          },
          const EncodeOptions(encode: false),
        ),
        equals('[][0]=2&[][1]=3'),
      );

      expect(
        QS.encode(
          {
            '': {
              '': [2, 3],
              'a': 2
            }
          },
          const EncodeOptions(encode: false),
        ),
        equals('[][0]=2&[][1]=3&[a]=2'),
      );

      expect(
        QS.encode(
          {
            '': {
              '': [2, 3]
            }
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('[][0]=2&[][1]=3'),
      );

      expect(
        QS.encode(
          {
            '': {
              '': [2, 3],
              'a': 2
            }
          },
          const EncodeOptions(encode: false, listFormat: ListFormat.indices),
        ),
        equals('[][0]=2&[][1]=3&[a]=2'),
      );
    });

    test('encodes non-String keys', () {
      expect(
        QS.encode(
          {
            'a': 'b',
            'false': {},
          },
          const EncodeOptions(
            filter: ['a', false, null],
            allowDots: true,
            encodeDotInKeys: true,
          ),
        ),
        equals('a=b'),
      );
    });
  });

  group('encode non-Strings', () {
    test('encodes a null value', () {
      expect(QS.encode({'a': null}), equals('a='));
    });

    test('encodes a boolean value', () {
      expect(QS.encode({'a': true}), equals('a=true'));
      expect(QS.encode({'a': false}), equals('a=false'));
    });

    test('encodes a number value', () {
      expect(QS.encode({'a': 0}), equals('a=0'));
      expect(QS.encode({'a': 1}), equals('a=1'));
      expect(QS.encode({'a': 1.1}), equals('a=1.1'));
    });

    test('encodes a buffer value', () {
      expect(QS.encode({'a': utf8.encode('test').buffer}), equals('a=test'));
    });

    test('encodes a date value', () {
      final DateTime now = DateTime.now();
      final String str = 'a=${Uri.encodeComponent(now.toIso8601String())}';
      expect(QS.encode({'a': now}), equals(str));
    });

    test('encodes a Duration', () {
      final Duration duration = const Duration(
          days: 1,
          hours: 2,
          minutes: 3,
          seconds: 4,
          milliseconds: 5,
          microseconds: 6);
      final String str = 'a=${Uri.encodeComponent(duration.toString())}';
      expect(QS.encode({'a': duration}), equals(str));
    });

    test(
      'encodes a BigInt',
      () {
        final BigInt bigInt = BigInt.from(1234567890123456);
        final String str = 'a=${Uri.encodeComponent(bigInt.toString())}';
        expect(QS.encode({'a': bigInt}), equals(str));
      },
    );

    test('encodes a list value', () {
      expect(
          QS.encode({
            'a': [1, 2, 3]
          }),
          equals('a%5B0%5D=1&a%5B1%5D=2&a%5B2%5D=3'));
    });

    test('encodes a map value', () {
      expect(
          QS.encode({
            'a': {'b': 'c'}
          }),
          equals('a%5Bb%5D=c'));
    });

    test('encodes a Uri', () {
      expect(
        QS.encode({'a': Uri.parse('https://example.com?foo=bar&baz=qux')}),
        equals('a=https%3A%2F%2Fexample.com%3Ffoo%3Dbar%26baz%3Dqux'),
      );
    });

    test('encodes a map with a null map as a child', () {
      final Map<String, dynamic> obj = {
        'a': {},
      };
      obj['a']['b'] = 'c';
      expect(QS.encode(obj), equals('a%5Bb%5D=c'));
    });

    test('encodes a map with an enum as a child', () {
      final Map<String, dynamic> obj = {
        'a': DummyEnum.lorem,
        'b': 'foo',
        'c': 1,
        'd': 1.234,
        'e': true,
      };
      expect(
        QS.encode(obj),
        equals('a=lorem&b=foo&c=1&d=1.234&e=true'),
      );
    });

    // does not encode
    // Symbol
    test('does not encode a Symbol', () {
      expect(QS.encode({'a': #a}), equals(''));
    });

    // Record
    test('does not encode a Record', () {
      expect(QS.encode({'a': ('b', 'c')}), equals(''));

      ({int a, String b}) rec = (a: 1, b: 'a');
      expect(QS.encode({'a': rec}), equals(''));
    });

    // Future
    test('does not encode a Future', () {
      expect(QS.encode({'a': Future.value('b')}), equals(''));
    });

    // Undefined
    test('does not encode a Undefined', () {
      expect(QS.encode({'a': const Undefined()}), equals(''));
    });
  });

  /// Copied and adapted from https://github.com/luffynando/dart_api_query/blob/main/test/unit/qs/stringify_test.dart
  group('dart_api_query tests', () {
    test('encodes a query string object', () {
      expect(QS.encode({'a': 'b'}), equals('a=b'));
      expect(QS.encode({'a': 1}), equals('a=1'));
      expect(QS.encode({'a': 1, 'b': 2}), equals('a=1&b=2'));
      expect(QS.encode({'a': 'A_Z'}), equals('a=A_Z'));
      expect(QS.encode({'a': '€'}), equals('a=%E2%82%AC'));
      expect(QS.encode({'a': ''}), equals('a=%EE%80%80'));
      expect(QS.encode({'a': 'א'}), equals('a=%D7%90'));
      expect(QS.encode({'a': '𐐷'}), equals('a=%F0%90%90%B7'));
    });

    test('encodes falsy values', () {
      expect(QS.encode(null), equals(''));
      expect(
        QS.encode(null, const EncodeOptions(strictNullHandling: true)),
        equals(''),
      );
      expect(QS.encode(false), equals(''));
      expect(QS.encode(0), equals(''));
    });

    test('encodes big ints', () {
      final three = BigInt.from(3);
      String encodeWithN(
        dynamic value, {
        Encoding? charset,
        Format? format,
      }) {
        final String result =
            Utils.encode(value.toString(), charset: charset ?? utf8);
        return value is BigInt ? '${result}n' : result;
      }

      expect(QS.encode(three), equals(''));
      expect(QS.encode([three]), equals('0=3'));
      expect(
        QS.encode([three], EncodeOptions(encoder: encodeWithN)),
        equals('0=3n'),
      );
      expect(QS.encode({'a': three}), equals('a=3'));
      expect(
        QS.encode(
          {'a': three},
          EncodeOptions(encoder: encodeWithN),
        ),
        equals('a=3n'),
      );
      expect(
        QS.encode(
          {
            'a': [three],
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=3'),
      );
      expect(
        QS.encode(
          {
            'a': [three],
          },
          EncodeOptions(
            encodeValuesOnly: true,
            encoder: encodeWithN,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=3n'),
      );
    });

    test('adds query prefix', () {
      expect(
        QS.encode({'a': 'b'}, const EncodeOptions(addQueryPrefix: true)),
        equals('?a=b'),
      );
    });

    test('with query prefix, outputs blank string given an empty object', () {
      expect(
        QS.encode(
          <String, dynamic>{},
          const EncodeOptions(addQueryPrefix: true),
        ),
        equals(''),
      );
    });

    test('encodes nested falsy values', () {
      expect(
        QS.encode({
          'a': {
            'b': {'c': null},
          },
        }),
        equals('a%5Bb%5D%5Bc%5D='),
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': {'c': null},
            },
          },
          const EncodeOptions(strictNullHandling: true),
        ),
        equals('a%5Bb%5D%5Bc%5D'),
      );
      expect(
        QS.encode({
          'a': {
            'b': {'c': false},
          },
        }),
        equals('a%5Bb%5D%5Bc%5D=false'),
      );
    });

    test('encodes a nested object', () {
      expect(
        QS.encode({
          'a': {'b': 'c'},
        }),
        equals('a%5Bb%5D=c'),
      );
      expect(
        QS.encode({
          'a': {
            'b': {
              'c': {'d': 'e'},
            },
          },
        }),
        equals('a%5Bb%5D%5Bc%5D%5Bd%5D=e'),
      );
    });

    test('encodes a nested object with dots notation', () {
      expect(
        QS.encode(
          {
            'a': {'b': 'c'},
          },
          const EncodeOptions(allowDots: true),
        ),
        equals('a.b=c'),
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': {
                'c': {'d': 'e'},
              },
            },
          },
          const EncodeOptions(allowDots: true),
        ),
        equals('a.b.c.d=e'),
      );
    });

    test('encodes an array value', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c', 'd'],
          },
          const EncodeOptions(listFormat: ListFormat.indices),
        ),
        equals('a%5B0%5D=b&a%5B1%5D=c&a%5B2%5D=d'),
        reason: 'indices => indices',
      );
      expect(
        QS.encode(
          {
            'a': ['b', 'c', 'd'],
          },
          const EncodeOptions(listFormat: ListFormat.brackets),
        ),
        equals('a%5B%5D=b&a%5B%5D=c&a%5B%5D=d'),
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode(
          {
            'a': ['b', 'c', 'd'],
          },
          const EncodeOptions(listFormat: ListFormat.comma),
        ),
        equals('a=b%2Cc%2Cd'),
        reason: 'comma => comma',
      );
      expect(
        QS.encode({
          'a': ['b', 'c', 'd'],
        }),
        equals('a%5B0%5D=b&a%5B1%5D=c&a%5B2%5D=d'),
        reason: 'default => indices',
      );
    });

    test('omits nulls when asked', () {
      expect(
        QS.encode(
          {'a': 'b', 'c': null},
          const EncodeOptions(skipNulls: true),
        ),
        equals('a=b'),
      );
    });

    test('omits nested null when asked', () {
      expect(
        QS.encode(
          {
            'a': {'b': 'c', 'd': null},
          },
          const EncodeOptions(skipNulls: true),
        ),
        equals('a%5Bb%5D=c'),
      );
    });

    test('omits array indices when asked', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c', 'd'],
          },
          const EncodeOptions(indices: false),
        ),
        equals('a=b&a=c&a=d'),
      );
    });

    group('encodes an array value with one item vs multiple items', () {
      test('non-array item', () {
        expect(
          QS.encode(
            {'a': 'c'},
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a=c'),
        );
        expect(
          QS.encode(
            {'a': 'c'},
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a=c'),
        );
        expect(
          QS.encode(
            {'a': 'c'},
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
            ),
          ),
          equals('a=c'),
        );
        expect(
          QS.encode(
            {'a': 'c'},
            const EncodeOptions(encodeValuesOnly: true),
          ),
          equals('a=c'),
        );
      });

      test('array with a single item', () {
        expect(
          QS.encode(
            {
              'a': ['c'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a[0]=c'),
        );
        expect(
          QS.encode(
            {
              'a': ['c'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a[]=c'),
        );
        expect(
          QS.encode(
            {
              'a': ['c'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
            ),
          ),
          equals('a=c'),
        );
        expect(
          QS.encode(
            {
              'a': ['c'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
              commaRoundTrip: true,
            ),
          ),
          equals('a[]=c'),
        );
        expect(
          QS.encode(
            {
              'a': ['c'],
            },
            const EncodeOptions(encodeValuesOnly: true),
          ),
          equals('a[0]=c'),
        );
      });

      test('array with multiple items', () {
        expect(
          QS.encode(
            {
              'a': ['c', 'd'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.indices,
            ),
          ),
          equals('a[0]=c&a[1]=d'),
        );
        expect(
          QS.encode(
            {
              'a': ['c', 'd'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.brackets,
            ),
          ),
          equals('a[]=c&a[]=d'),
        );
        expect(
          QS.encode(
            {
              'a': ['c', 'd'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
            ),
          ),
          equals('a=c,d'),
        );
        expect(
          QS.encode(
            {
              'a': ['c', 'd'],
            },
            const EncodeOptions(encodeValuesOnly: true),
          ),
          equals('a[0]=c&a[1]=d'),
        );
      });

      test('array with multiple items with a comma inside', () {
        expect(
          QS.encode(
            {
              'a': ['c,d', 'e'],
            },
            const EncodeOptions(
              encodeValuesOnly: true,
              listFormat: ListFormat.comma,
            ),
          ),
          equals('a=c%2Cd,e'),
        );
        expect(
          QS.encode(
            {
              'a': ['c,d', 'e'],
            },
            const EncodeOptions(listFormat: ListFormat.comma),
          ),
          equals('a=c%2Cd%2Ce'),
        );
      });
    });

    test('encodes a nested array value', () {
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[b][0]=c&a[b][1]=d'),
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[b][]=c&a[b][]=d'),
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a[b]=c,d'),
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a[b][0]=c&a[b][1]=d'),
      );
    });

    test('encodes comma and empty array values', () {
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=,&a[1]=&a[2]=c,d%'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=,&a[]=&a[]=c,d%'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=,,,c,d%'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=,&a=&a=c,d%'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0]=%2C&a[1]=&a[2]=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[]=%2C&a[]=&a[]=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C,,c%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&a=&a=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a%5B0%5D=%2C&a%5B1%5D=&a%5B2%5D=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a%5B%5D=%2C&a%5B%5D=&a%5B%5D=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C%2C%2Cc%2Cd%25'),
      );
      expect(
        QS.encode(
          {
            'a': [',', '', 'c,d%'],
          },
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&a=&a=c%2Cd%25'),
      );
    });

    test('encodes comma and empty non-array values', () {
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=,&b=&c=c,d%'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=,&b=&c=c,d%'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=,&b=&c=c,d%'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=,&b=&c=c,d%'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );

      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
      expect(
        QS.encode(
          {'a': ',', 'b': '', 'c': 'c,d%'},
          const EncodeOptions(
            encode: true,
            encodeValuesOnly: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('a=%2C&b=&c=c%2Cd%25'),
      );
    });

    test('encodes a nested array value with dots notation', () {
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            allowDots: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a.b[0]=c&a.b[1]=d'),
        reason: 'indices: encodes with dots + indices',
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            allowDots: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a.b[]=c&a.b[]=d'),
        reason: 'brackets: encodes with dots + brackets',
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            allowDots: true,
            encodeValuesOnly: true,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a.b=c,d'),
        reason: 'comma: encodes with dots + comma',
      );
      expect(
        QS.encode(
          {
            'a': {
              'b': ['c', 'd'],
            },
          },
          const EncodeOptions(
            allowDots: true,
            encodeValuesOnly: true,
          ),
        ),
        equals('a.b[0]=c&a.b[1]=d'),
        reason: 'default: encodes with dots + indices',
      );
    });

    test('encodes an object inside an array', () {
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(listFormat: ListFormat.indices),
        ),
        equals('a%5B0%5D%5Bb%5D=c'), // a[0][b]=c
        reason: 'indices => brackets',
      );
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(listFormat: ListFormat.brackets),
        ),
        equals('a%5B%5D%5Bb%5D=c'), // a[][b]=c
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode({
          'a': [
            {'b': 'c'},
          ],
        }),
        equals('a%5B0%5D%5Bb%5D=c'),
        reason: 'default => indices',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1],
                },
              }
            ],
          },
          const EncodeOptions(listFormat: ListFormat.indices),
        ),
        equals('a%5B0%5D%5Bb%5D%5Bc%5D%5B0%5D=1'),
        reason: 'indices => indices',
      );
      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1],
                },
              }
            ],
          },
          const EncodeOptions(listFormat: ListFormat.brackets),
        ),
        equals('a%5B%5D%5Bb%5D%5Bc%5D%5B%5D=1'),
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode({
          'a': [
            {
              'b': {
                'c': [1],
              },
            }
          ],
        }),
        equals('a%5B0%5D%5Bb%5D%5Bc%5D%5B0%5D=1'),
        reason: 'default => indices',
      );
    });

    test('encodes an array with mixed objects and primitives', () {
      expect(
        QS.encode(
          {
            'a': [
              {'b': 1},
              2,
              3,
            ],
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0][b]=1&a[1]=2&a[2]=3'),
        reason: 'indices => indices',
      );
      expect(
        QS.encode(
          {
            'a': [
              {'b': 1},
              2,
              3,
            ],
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[][b]=1&a[]=2&a[]=3'),
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode(
          {
            'a': [
              {'b': 1},
              2,
              3,
            ],
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a[0][b]=1&a[1]=2&a[2]=3'),
        reason: 'default => indices',
      );
    });

    test('encodes an object inside an array with dots notation', () {
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0].b=c'),
        reason: 'indices => indices',
      );
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[].b=c'),
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
          ),
        ),
        equals('a[0].b=c'),
        reason: 'default => indices',
      );

      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1],
                },
              }
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[0].b.c[0]=1'),
        reason: 'indices => indices',
      );
      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1],
                },
              }
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[].b.c[]=1'),
        reason: 'brackets => brackets',
      );
      expect(
        QS.encode(
          {
            'a': [
              {
                'b': {
                  'c': [1],
                },
              }
            ],
          },
          const EncodeOptions(
            allowDots: true,
            encode: false,
          ),
        ),
        equals('a[0].b.c[0]=1'),
        reason: 'default => indices',
      );
    });

    test('does not omit object keys when indices = false', () {
      expect(
        QS.encode(
          {
            'a': [
              {'b': 'c'},
            ],
          },
          const EncodeOptions(indices: false),
        ),
        equals('a%5Bb%5D=c'),
      );
    });

    test('uses indices notation for arrays when indices=true', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c'],
          },
          const EncodeOptions(indices: true),
        ),
        equals('a%5B0%5D=b&a%5B1%5D=c'),
      );
    });

    test('uses indices notation for arrays when no listFormat is specified',
        () {
      expect(
        QS.encode({
          'a': ['b', 'c'],
        }),
        equals('a%5B0%5D=b&a%5B1%5D=c'),
      );
    });

    test('uses indices notation for arrays when listFormat=indices', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c'],
          },
          const EncodeOptions(listFormat: ListFormat.indices),
        ),
        equals('a%5B0%5D=b&a%5B1%5D=c'),
      );
    });

    test('uses repeat notation for arrays when no listFormat=repeat', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c'],
          },
          const EncodeOptions(listFormat: ListFormat.repeat),
        ),
        equals('a=b&a=c'),
      );
    });

    test('uses brackets notation for arrays when no listFormat=brackets', () {
      expect(
        QS.encode(
          {
            'a': ['b', 'c'],
          },
          const EncodeOptions(listFormat: ListFormat.brackets),
        ),
        equals('a%5B%5D=b&a%5B%5D=c'),
      );
    });

    test('encodes a complicated object', () {
      expect(
        QS.encode({
          'a': {'b': 'c', 'd': 'e'},
        }),
        equals('a%5Bb%5D=c&a%5Bd%5D=e'),
      );
    });

    test('encodes an empty value', () {
      expect(QS.encode({'a': ''}), equals('a='));
      expect(
        QS.encode(
          {'a': null},
          const EncodeOptions(strictNullHandling: true),
        ),
        equals('a'),
      );

      expect(QS.encode({'a': '', 'b': ''}), equals('a=&b='));
      expect(
        QS.encode(
          {'a': null, 'b': ''},
          const EncodeOptions(strictNullHandling: true),
        ),
        equals('a&b='),
      );

      expect(
        QS.encode({
          'a': {'b': ''},
        }),
        equals('a%5Bb%5D='),
      );
      expect(
        QS.encode(
          {
            'a': {'b': null},
          },
          const EncodeOptions(strictNullHandling: true),
        ),
        equals('a%5Bb%5D'),
      );
      expect(
        QS.encode(
          {
            'a': {'b': null},
          },
          const EncodeOptions(strictNullHandling: false),
        ),
        equals('a%5Bb%5D='),
      );
    });

    test('encodes an empty array in different listFormat', () {
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(encode: false),
        ),
        equals('b[0]=&c=c'),
      );
      // listFormat default
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('b[0]=&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('b[]=&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.repeat,
          ),
        ),
        equals('b=&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('b=&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
            commaRoundTrip: true,
          ),
        ),
        equals('b[]=&c=c'),
      );
      // with strictNullHandling
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
            strictNullHandling: true,
          ),
        ),
        equals('b[0]&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
            strictNullHandling: true,
          ),
        ),
        equals('b[]&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.repeat,
            strictNullHandling: true,
          ),
        ),
        equals('b&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
            strictNullHandling: true,
          ),
        ),
        equals('b&c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
            commaRoundTrip: true,
            strictNullHandling: true,
          ),
        ),
        equals('b[]&c=c'),
      );
      // with skipNulls
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
            skipNulls: true,
          ),
        ),
        equals('c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
            skipNulls: true,
          ),
        ),
        equals('c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.repeat,
            skipNulls: true,
          ),
        ),
        equals('c=c'),
      );
      expect(
        QS.encode(
          {
            'a': <dynamic>[],
            'b': [null],
            'c': 'c',
          },
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
            skipNulls: true,
          ),
        ),
        equals('c=c'),
      );
    });

    test('returns an empty string for invalid input', () {
      expect(QS.encode(false), equals(''));
      expect(QS.encode(null), equals(''));
      expect(QS.encode(''), equals(''));
    });

    test('url encodes values', () {
      expect(QS.encode({'a': 'b c'}), equals('a=b%20c'));
    });

    test('encodes a date', () {
      final now = DateTime.now();
      final str = 'a=${Uri.encodeComponent(now.toIso8601String())}';
      expect(QS.encode({'a': now}), equals(str));
    });

    test('encodes the weird object from qs', () {
      expect(
        QS.encode({'my weird field': '~q1!2"\'w\$5&7/z8)?'}),
        equals('my%20weird%20field=~q1%212%22%27w%245%267%2Fz8%29%3F'),
      );
    });

    test('encodes boolean values', () {
      expect(QS.encode({'a': true}), equals('a=true'));
      expect(
        QS.encode({
          'a': {'b': true},
        }),
        equals('a%5Bb%5D=true'),
      );
      expect(QS.encode({'b': false}), equals('b=false'));
      expect(
        QS.encode({
          'b': {'c': false},
        }),
        equals('b%5Bc%5D=false'),
      );
    });

    test('encodes buffer values', () {
      expect(QS.encode({'a': StringBuffer('test')}), equals('a=test'));
      expect(
        QS.encode({
          'a': {'b': StringBuffer('test')},
        }),
        equals('a%5Bb%5D=test'),
      );
    });

    test('encodes an object using an alternative delimiter', () {
      expect(
        QS.encode(
          {'a': 'b', 'c': 'd'},
          const EncodeOptions(delimiter: ';'),
        ),
        equals('a=b;c=d'),
      );
    });

    test('does not crash when parsing circular references', () {
      final a = <String, dynamic>{};
      a['b'] = a;
      expect(
        () => QS.encode({'foo[bar]': 'baz', 'foo[baz]': a}),
        throwsA(
          predicate(
            (e) => e is RangeError && e.message == 'Cyclic object value',
          ),
        ),
      );

      final circular = <String, dynamic>{'a': 'value'};
      circular['a'] = circular;
      expect(
        () => QS.encode(circular),
        throwsA(isA<RangeError>()),
      );

      final arr = ['a'];
      expect(
        () => QS.encode({'x': arr, 'y': arr}),
        isNot(throwsRangeError),
        reason: 'non-cyclic values do not throw',
      );
    });

    test('non-circular duplicated references can still work', () {
      final hourOfDay = {'function': 'hour_of_day'};
      final p1 = {
        'function': 'gte',
        'arguments': [hourOfDay, 0],
      };
      final p2 = {
        'function': 'lte',
        'arguments': [hourOfDay, 23],
      };

      expect(
        QS.encode(
          {
            'filters': {
              r'$and': [p1, p2],
            },
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals(
          [
            r'filters[$and][0][function]=',
            r'gte&filters[$and][0][arguments][0][function]=',
            r'hour_of_day&filters[$and][0][arguments][1]=',
            r'0&filters[$and][1][function]=',
            r'lte&filters[$and][1][arguments][0][function]=',
            r'hour_of_day&filters[$and][1][arguments][1]=23',
          ].join(),
        ),
      );
    });

    test('can disable uri encoding', () {
      expect(
        QS.encode({'a': 'b'}, const EncodeOptions(encode: false)),
        equals('a=b'),
      );
      expect(
        QS.encode(
          {
            'a': {'b': 'c'},
          },
          const EncodeOptions(encode: false),
        ),
        equals('a[b]=c'),
      );
      expect(
        QS.encode(
          {'a': 'b', 'c': null},
          const EncodeOptions(
            strictNullHandling: true,
            encode: false,
          ),
        ),
        equals('a=b&c'),
      );
    });

    test('can sort the keys', () {
      int sort(dynamic a, dynamic b) => a.toString().compareTo(b.toString());

      expect(
        QS.encode(
          {'a': 'c', 'z': 'y', 'b': 'f'},
          EncodeOptions(sort: sort),
        ),
        equals('a=c&b=f&z=y'),
      );
      expect(
        QS.encode(
          {
            'a': 'c',
            'z': {'j': 'a', 'i': 'b'},
            'b': 'f',
          },
          EncodeOptions(sort: sort),
        ),
        equals('a=c&b=f&z%5Bi%5D=b&z%5Bj%5D=a'),
      );
    });

    test('can sort the keys at depth 3 or more too', () {
      int sort(dynamic a, dynamic b) => a.toString().compareTo(b.toString());

      expect(
        QS.encode(
          {
            'a': 'a',
            'z': {
              'zj': {'zjb': 'zjb', 'zja': 'zja'},
              'zi': {'zib': 'zib', 'zia': 'zia'},
            },
            'b': 'b',
          },
          EncodeOptions(
            sort: sort,
            encode: false,
          ),
        ),
        equals(
          'a=a&b=b&z[zi][zia]=zia&z[zi][zib]=zib&z[zj][zja]=zja&z[zj][zjb]=zjb',
        ),
      );
      expect(
        QS.encode(
          {
            'a': 'a',
            'z': {
              'zj': {'zjb': 'zjb', 'zja': 'zja'},
              'zi': {'zib': 'zib', 'zia': 'zia'},
            },
            'b': 'b',
          },
          const EncodeOptions(encode: false),
        ),
        equals(
          'a=a&z[zj][zjb]=zjb&z[zj][zja]=zja&z[zi][zib]=zib&z[zi][zia]=zia&b=b',
        ),
      );
    });

    test('serializeDate option', () {
      final date = DateTime.now();
      expect(
        QS.encode({'a': date}),
        equals(
          'a=${date.toIso8601String().replaceAll(':', '%3A')}',
        ),
      );

      final specificDate = DateTime.fromMillisecondsSinceEpoch(6);
      expect(
        QS.encode(
          {'a': specificDate},
          EncodeOptions(
            serializeDate: (DateTime d) =>
                (d.millisecondsSinceEpoch * 7).toString(),
          ),
        ),
        equals('a=42'),
        reason: 'custom serializeDate function called',
      );

      expect(
        QS.encode(
          {
            'a': [date],
          },
          EncodeOptions(
            serializeDate: (DateTime d) => d.millisecondsSinceEpoch.toString(),
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a=${date.millisecondsSinceEpoch}'),
        reason: 'works with listFormat comma',
      );
      expect(
        QS.encode(
          {
            'a': [date],
          },
          EncodeOptions(
            serializeDate: (DateTime d) => d.millisecondsSinceEpoch.toString(),
            listFormat: ListFormat.comma,
            commaRoundTrip: true,
          ),
        ),
        equals('a%5B%5D=${date.millisecondsSinceEpoch}'),
        reason: 'works with listFormat comma',
      );
    });

    test('RFC 1738 serialization', () {
      expect(
        QS.encode(
          {'a': 'b c'},
          const EncodeOptions(
            format: Format.rfc1738,
          ),
        ),
        equals('a=b+c'),
      );
      expect(
        QS.encode(
          {'a b': 'c d'},
          const EncodeOptions(format: Format.rfc1738),
        ),
        equals('a+b=c+d'),
      );
      expect(
        QS.encode(
          {'a b': StringBuffer('a b')},
          const EncodeOptions(format: Format.rfc1738),
        ),
        equals('a+b=a+b'),
      );

      expect(
        QS.encode(
          {'foo(ref)': 'bar'},
          const EncodeOptions(format: Format.rfc1738),
        ),
        equals('foo(ref)=bar'),
      );
    });

    test('RFC 3986 spaces serialization', () {
      expect(
        QS.encode(
          {'a': 'b c'},
          const EncodeOptions(
            format: Format.rfc3986,
          ),
        ),
        equals('a=b%20c'),
      );
      expect(
        QS.encode(
          {'a b': 'c d'},
          const EncodeOptions(format: Format.rfc3986),
        ),
        equals('a%20b=c%20d'),
      );
      expect(
        QS.encode(
          {'a b': StringBuffer('a b')},
          const EncodeOptions(format: Format.rfc3986),
        ),
        equals('a%20b=a%20b'),
      );
    });

    test('Backward compatibility to RFC 3986', () {
      expect(QS.encode({'a': 'b c'}), equals('a=b%20c'));
      expect(QS.encode({'a b': StringBuffer('a b')}), equals('a%20b=a%20b'));
    });

    test('encodeValuesOnly', () {
      expect(
        QS.encode(
          {
            'a': 'b',
            'c': ['d', 'e=f'],
            'f': [
              ['g'],
              ['h'],
            ],
          },
          const EncodeOptions(encodeValuesOnly: true),
        ),
        equals('a=b&c[0]=d&c[1]=e%3Df&f[0][0]=g&f[1][0]=h'),
      );
      expect(
        QS.encode({
          'a': 'b',
          'c': ['d', 'e'],
          'f': [
            ['g'],
            ['h'],
          ],
        }),
        equals('a=b&c%5B0%5D=d&c%5B1%5D=e&f%5B0%5D%5B0%5D=g&f%5B1%5D%5B0%5D=h'),
      );
    });

    test('encodeValuesOnly - strictNullHandling', () {
      expect(
        QS.encode(
          {
            'a': {'b': null},
          },
          const EncodeOptions(
            encodeValuesOnly: true,
            strictNullHandling: true,
          ),
        ),
        equals('a[b]'),
      );
    });

    test('respects a charset of iso-8859-1', () {
      expect(
        QS.encode(
          {'æ': 'æ'},
          const EncodeOptions(charset: latin1),
        ),
        equals('%E6=%E6'),
      );
    });

    test(
      'encodes unrepresentable chars as numeric entities in iso-8859-1 mode',
      () {
        expect(
          QS.encode(
            {'a': '☺'},
            const EncodeOptions(charset: latin1),
          ),
          equals('a=%26%239786%3B'),
        );
      },
    );

    test('respects an explicit charset of utf-8 (the default)', () {
      expect(
        QS.encode(
          {'a': 'æ'},
          const EncodeOptions(charset: utf8),
        ),
        equals('a=%C3%A6'),
      );
    });

    test('adds the right sentinel when instructed to and charset is utf-8', () {
      expect(
        QS.encode(
          {'a': 'æ'},
          const EncodeOptions(
            charsetSentinel: true,
            charset: utf8,
          ),
        ),
        equals('utf8=%E2%9C%93&a=%C3%A6'),
      );
    });

    test('adds the right sentinel when instructed to and charset is iso88591',
        () {
      expect(
        QS.encode(
          {'a': 'æ'},
          const EncodeOptions(
            charsetSentinel: true,
            charset: latin1,
          ),
        ),
        equals('utf8=%26%2310003%3B&a=%E6'),
      );
    });

    test('objects inside arrays', () {
      final obj = {
        'a': {
          'b': {'c': 'd', 'e': 'f'},
        },
      };
      final withArray = {
        'a': {
          'b': [
            {'c': 'd', 'e': 'f'},
          ],
        },
      };

      expect(
        QS.encode(obj, const EncodeOptions(encode: false)),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no array, no listFormat',
      );
      expect(
        QS.encode(
          obj,
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no array, bracket',
      );
      expect(
        QS.encode(
          obj,
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no array, indices',
      );
      expect(
        QS.encode(
          obj,
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.comma,
          ),
        ),
        equals('a[b][c]=d&a[b][e]=f'),
        reason: 'no array, comma',
      );

      expect(
        QS.encode(
          withArray,
          const EncodeOptions(encode: false),
        ),
        equals('a[b][0][c]=d&a[b][0][e]=f'),
        reason: 'array, no listFormat',
      );
      expect(
        QS.encode(
          withArray,
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.brackets,
          ),
        ),
        equals('a[b][][c]=d&a[b][][e]=f'),
        reason: 'array, bracket',
      );
      expect(
        QS.encode(
          withArray,
          const EncodeOptions(
            encode: false,
            listFormat: ListFormat.indices,
          ),
        ),
        equals('a[b][0][c]=d&a[b][0][e]=f'),
        reason: 'array, indices',
      );
    });

    test('edge case with object/arrays', () {
      expect(
        QS.encode(
          {
            '': {
              '': [2, 3],
            },
          },
          const EncodeOptions(encode: false),
        ),
        equals('[][0]=2&[][1]=3'),
      );
      expect(
        QS.encode(
          {
            '': {
              '': [2, 3],
              'a': 2,
            },
          },
          const EncodeOptions(encode: false),
        ),
        equals('[][0]=2&[][1]=3&[a]=2'),
      );
    });
  });

  group('Additional encode coverage', () {
    test('empty list with allowEmptyLists emits key[]', () {
      expect(
        QS.encode({'a': []},
            const EncodeOptions(allowEmptyLists: true, encode: false)),
        'a[]',
      );
    });

    test('commaRoundTrip single item list adds []', () {
      expect(
        QS.encode(
            {
              'a': ['x']
            },
            const EncodeOptions(
                listFormat: ListFormat.comma,
                commaRoundTrip: true,
                encode: false)),
        'a[]=x',
      );
    });

    test('commaCompactNulls drops null entries before joining', () {
      expect(
        QS.encode(
          {
            'a': {
              'b': [true, false, null, true]
            }
          },
          const EncodeOptions(
            listFormat: ListFormat.comma,
            commaCompactNulls: true,
            encode: false,
          ),
        ),
        'a[b]=true,false,true',
      );
    });

    test('commaCompactNulls omits key when all entries are null', () {
      expect(
        QS.encode(
          {
            'a': [null, null]
          },
          const EncodeOptions(
            listFormat: ListFormat.comma,
            commaCompactNulls: true,
            encode: false,
          ),
        ),
        isEmpty,
      );
    });

    test('commaCompactNulls keeps round-trip marker after filtering', () {
      expect(
        QS.encode(
          {
            'a': [null, 'foo']
          },
          const EncodeOptions(
            listFormat: ListFormat.comma,
            commaRoundTrip: true,
            commaCompactNulls: true,
            encode: false,
          ),
        ),
        'a[]=foo',
      );
    });

    test('cycle detection throws RangeError', () {
      final map = <String, dynamic>{};
      map['self'] = map; // self reference
      expect(
        () => QS.encode(map),
        throwsA(isA<RangeError>()),
      );
    });
  });
}
