/*
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.drdisagree.materialcolorutilities.scheme

import com.drdisagree.materialcolorutilities.dynamiccolor.ColorSpec.SpecVersion
import com.drdisagree.materialcolorutilities.dynamiccolor.ColorSpecs
import com.drdisagree.materialcolorutilities.dynamiccolor.DynamicScheme
import com.drdisagree.materialcolorutilities.dynamiccolor.Variant
import com.drdisagree.materialcolorutilities.hct.Hct

/**
 * A scheme that places the source color in Scheme.primaryContainer.
 *
 * Primary Container is the source color, adjusted for color relativity. It maintains constant
 * appearance in light mode and dark mode. This adds ~5 tone in light mode, and subtracts ~5 tone in
 * dark mode.
 *
 * Tertiary Container is an analogous color, specifically, the analog of a color wheel divided into
 * 6, and the precise analog is the one found by increasing hue. This is a scientifically grounded
 * equivalent to rotating hue clockwise by 60 degrees. It also maintains constant appearance.
 */
class SchemeContent(
  sourceColorHct: Hct,
  isDark: Boolean,
  contrastLevel: Double,
  specVersion: SpecVersion = DEFAULT_SPEC_VERSION,
  platform: Platform = DEFAULT_PLATFORM,
) :
  DynamicScheme(
    sourceColorHct,
    Variant.CONTENT,
    isDark,
    contrastLevel,
    platform,
    specVersion,
    ColorSpecs.get(specVersion)
      .getPrimaryPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
    ColorSpecs.get(specVersion)
      .getSecondaryPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
    ColorSpecs.get(specVersion)
      .getTertiaryPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
    ColorSpecs.get(specVersion)
      .getNeutralPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
    ColorSpecs.get(specVersion)
      .getNeutralVariantPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
    ColorSpecs.get(specVersion)
      .getErrorPalette(Variant.CONTENT, sourceColorHct, isDark, platform, contrastLevel),
  )
