package com.dozingcatsoftware.vectorpinball.model;

import com.dozingcatsoftware.vectorpinball.util.FrameRateManager;

/**
 * Class to manage the game thread which updates the game's internal state and draws to the
 * FieldView. Controls the frame rate and attempts to keep it as consistent as possible. Because
 * this class manipulates the Field object in a separate thread, all access to the Field from the
 * game thread and main thread must be synchronized.
 */
public class FieldDriver {

    private final Field field;
    private final Runnable drawFn;

    public FieldDriver(Field field, Runnable drawFn) {
        this.field = field;
        this.drawFn = drawFn;
    }

    // Volatile so game thread sees updates from main thread immediately.
    private volatile boolean running;
    private Thread gameThread;

    FrameRateManager frameRateManager = new FrameRateManager(
            System::nanoTime,
            new double[] {144, 120, 90, 60, 50, 45, 40, 30},
            new double[] {137, 114, 86, 57, 48, 43, 38});
    double averageFPS;

    // Sleep this long when field.hasActiveElements() is false.
    private static final long INACTIVE_FRAME_MSECS = 250;

    private static final long MILLION = 1_000_000;
    private static final long BILLION = MILLION * 1000;

    /** Starts the game thread running. Does not actually start a new game. */
    public synchronized void start() {
        if (running) return;
        running = true;
        gameThread = new Thread(this::threadMain);
        gameThread.start();
    }

    /** Stops the game thread, which will pause updates to the game state and view redraws. */
    public synchronized void stop() {
        // Don't explicitly join() the game thread because that can deadlock.
        // Setting running to false will cause it to exit.
        running = false;
    }

    /**
     * Main loop for the game thread. Repeatedly calls field.tick to advance the game simulation,
     * redraws the field, and sleeps until it's time for the next frame. Dynamically adjusts sleep
     * times in an attempt to maintain a consistent frame rate.
     */
    void threadMain() {
        while (running) {
            frameRateManager.frameStarted();
            boolean fieldActive = true;
            if (field != null) {
                try {
                    synchronized (field) {
                        long nanosPerFrame =
                                (long) (BILLION / frameRateManager.targetFramesPerSecond());
                        long fieldTickNanos = (long) (nanosPerFrame * field.getTargetTimeRatio());
                        // If field isn't doing anything, sleep for a long time.
                        fieldActive = field.hasActiveElements();
                        if (!fieldActive) {
                            fieldTickNanos = (long)
                                    (INACTIVE_FRAME_MSECS * MILLION * field.getTargetTimeRatio());
                        }
                        field.tick(fieldTickNanos, 4);
                    }
                    drawFn.run();
                }
                catch (Exception ex) {
                    ex.printStackTrace();
                }
            }

            // If field is inactive, clear start time history and bail.
            if (!fieldActive) {
                frameRateManager.clearTimestamps();
                setAverageFps(0);
                try {
                    Thread.sleep(INACTIVE_FRAME_MSECS);
                }
                catch (InterruptedException ignored) {
                }
                continue;
            }

            frameRateManager.sleepUntilNextFrame();

            // For debugging, show frames per second and other info.
            if (frameRateManager.getTotalFrames() % 100 == 0) {
                setAverageFps(frameRateManager.currentFramesPerSecond());
            }
        }
    }

    /**
     * Resets the frame rate and forgets any locked rate, called when rendering quality is changed.
     */
    public void resetFrameRate() {
        frameRateManager.resetFrameRate();
    }

    public void setMaxTargetFrameRate(double rate) {
        frameRateManager.setMaxTargetFrameRate(rate);
    }

    public double getAverageFps() {
        return averageFPS;
    }

    public void setAverageFps(double value) {
        averageFPS = value;
    }

    public double getTargetFps() {
        return frameRateManager.targetFramesPerSecond();
    }
}
