package com.dijia1124.plusplusbattery.vm

import android.app.Application
import android.content.Context
import android.net.Uri
import android.os.BatteryManager
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import com.dijia1124.plusplusbattery.data.model.BatteryInfo
import com.dijia1124.plusplusbattery.data.model.CustomEntry
import com.dijia1124.plusplusbattery.data.repository.BatteryInfoRepository
import com.dijia1124.plusplusbattery.data.repository.HistoryInfoRepository
import com.dijia1124.plusplusbattery.data.repository.PrefsRepository
import com.dijia1124.plusplusbattery.data.util.saveCycleCountToHistory
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class BatteryInfoViewModel(application: Application,
                           private val batteryInfoRepository: BatteryInfoRepository = BatteryInfoRepository(
                               application
                           ),
                           private val prefsRepo: PrefsRepository = PrefsRepository(application),
                           private val historyInfoRepository: HistoryInfoRepository = HistoryInfoRepository(
                               application
                           )
) : AndroidViewModel(application) {
    private val context = application.applicationContext

    val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as BatteryManager

    val isRootMode: StateFlow<Boolean> = prefsRepo.isRootModeFlow
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, false)

    fun setRootMode(enabled: Boolean) = viewModelScope.launch {
        prefsRepo.setRootMode(enabled)
    }

    val showSwitchOnDashboard: StateFlow<Boolean> = prefsRepo.showSwitchOnDashboardFlow
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, true)

    fun setShowSwitchOnDashboard(show: Boolean) = viewModelScope.launch {
        prefsRepo.setShowSwitchOnDashboard(show)
    }

    val showOplusFields: StateFlow<Boolean> = prefsRepo.showOplusFields
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, true)

    val isDualBatt: StateFlow<Boolean> = batteryInfoRepository.isDualBattFlow
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, false)

    fun setDualBatt(enabled: Boolean) = viewModelScope.launch {
        batteryInfoRepository.setDualBatt(enabled)
    }

    val isMultiply: StateFlow<Boolean> = batteryInfoRepository.isMultiplyFlow
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, true)

    val selectedMagnitude: StateFlow<Int> = batteryInfoRepository.selectedMagnitudeFlow
        .stateIn(viewModelScope, SharingStarted.Companion.Eagerly, 0)

    fun setMultiplierPrefs(isMultiply: Boolean, magnitude: Int) {
        viewModelScope.launch {
            batteryInfoRepository.setMultiplierPrefs(isMultiply, magnitude)
        }
    }

    val customEntries: StateFlow<List<CustomEntry>> =
        batteryInfoRepository.customEntries
            .stateIn(viewModelScope, SharingStarted.Eagerly, emptyList())

    suspend fun addCustomEntry(entry: CustomEntry) =
        batteryInfoRepository.addCustomEntry(entry)

    suspend fun removeCustomEntry(path: String) =
        batteryInfoRepository.removeCustomEntry(path)

    suspend fun getDisplayBatteryInfo(): List<BatteryInfo> = withContext(Dispatchers.IO) {
        val isRoot = isRootMode.value
        val showOplus = showOplusFields.value
        batteryInfoRepository.getAvailableBatteryInfo(isRoot, showOplus)
    }

    suspend fun saveCycleCount() {
        saveCycleCountToHistory(context, historyInfoRepository)
    }

    fun exportEntries(context: Context, onDone: (Uri) -> Unit) =
        viewModelScope.launch {
            try {
                val uri = batteryInfoRepository.exportEntriesToDownloads(context)
                onDone(uri)
            } catch (e: Exception) {
                onDone(Uri.EMPTY)
            }
        }

    fun importJsonFromUri(uri: Uri) = viewModelScope.launch {
        batteryInfoRepository.importFromUri(getApplication(), uri)
    }

    fun importPreset(name: String) = viewModelScope.launch {
        batteryInfoRepository.importPreset(getApplication(), name)
    }
}