package com.dijia1124.plusplusbattery.ui.screen

import android.Manifest
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.provider.Settings
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.ContextCompat
import androidx.navigation.NavController
import com.dijia1124.plusplusbattery.ui.components.AppScaffold
import androidx.compose.runtime.getValue
import com.dijia1124.plusplusbattery.vm.BatteryMonitorSettingsViewModel
import com.dijia1124.plusplusbattery.R
import androidx.core.net.toUri

@Composable
fun BatteryMonitor(currentTitle: String, navController: NavController, battMonVM: BatteryMonitorSettingsViewModel) {
    AppScaffold(currentTitle) {
        BatteryMonitorContent(navController, battMonVM)
    }
}

@Composable
fun BatteryMonitorContent(
    navController: NavController,
    battMonVM: BatteryMonitorSettingsViewModel
) {
    val scrollState = rememberScrollState()

    Column(
        modifier = Modifier
            .fillMaxWidth()
            .padding(horizontal = 10.dp)
            .verticalScroll(scrollState)
    ) {
        ListItem(
            modifier = Modifier.clickable { navController.navigate("batt_mon_settings") },
            headlineContent = { Text(text = stringResource(R.string.battery_monitor_entry_settings_list_item_title), style = MaterialTheme.typography.bodyLarge) }
        )
        ListItem(
            modifier = Modifier.clickable { navController.navigate("floating_window_settings") },
            headlineContent = { Text(text = stringResource(R.string.floating_window_settings_list_item_title), style = MaterialTheme.typography.bodyLarge) }
        )
        ListItem(
            headlineContent = {BatteryMonitorSwitch(battMonVM)}
        )
        ListItem(
            headlineContent = {FloatingWindowSwitch(battMonVM)}
        )
        ListItem(
            headlineContent = {
                Text(text = stringResource(R.string.disable_battery_optimization), style = MaterialTheme.typography.bodySmall)
            }
        )
    }
}

@Composable
fun FloatingWindowSwitch(battMonVM: BatteryMonitorSettingsViewModel) {
    val context = LocalContext.current
    val isFloatingWindowOn by battMonVM.isFloatingWindowOn.collectAsState()

    val overlayPermissionLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.StartActivityForResult()
    ) { _ ->
        if (Settings.canDrawOverlays(context)) {
            battMonVM.startFloatingWindow()
        } else {
            Toast.makeText(
                context,
                context.getString(R.string.overlay_permission_denied),
                Toast.LENGTH_SHORT
            ).show()
        }
    }

    Row(
        verticalAlignment = Alignment.CenterVertically
    ) {
        Text(text = context.getString(R.string.show_in_floating_window), modifier = Modifier.weight(1f), style = MaterialTheme.typography.bodyLarge)
        Switch(
            checked = isFloatingWindowOn,
            onCheckedChange = { on ->
                if (on) {
                    if (!Settings.canDrawOverlays(context)) {
                        val intent = Intent(
                            Settings.ACTION_MANAGE_OVERLAY_PERMISSION,
                            "package:${context.packageName}".toUri()
                        )
                        overlayPermissionLauncher.launch(intent)
                    } else {
                        battMonVM.startFloatingWindow()
                    }
                } else {
                    battMonVM.stopFloatingWindow()
                }
            }
        )
    }
}

@Composable
fun BatteryMonitorSwitch(battMonVM: BatteryMonitorSettingsViewModel) {
    val context = LocalContext.current
    val isMonitoring by battMonVM.isMonitoring.collectAsState()

    val permissionLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.RequestPermission()
    ) { granted: Boolean ->
        if (granted) {
            battMonVM.startMonitor()
        } else {
            Toast.makeText(
                context,
                context.getString(R.string.notification_permission_denied),
                Toast.LENGTH_SHORT
            ).show()
            battMonVM.stopMonitor()
        }
    }

    Row(
        verticalAlignment = Alignment.CenterVertically
    ) {
        Text(text = context.getString(R.string.show_in_notification_bar), modifier = Modifier.weight(1f), style = MaterialTheme.typography.bodyLarge)
        Switch(
            checked = isMonitoring,
            onCheckedChange = { on ->
                if (on) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                        when {
                            ContextCompat.checkSelfPermission(
                                context,
                                Manifest.permission.POST_NOTIFICATIONS
                            ) == PackageManager.PERMISSION_GRANTED -> {
                                battMonVM.startMonitor()
                            }
                            else -> {
                                permissionLauncher.launch(Manifest.permission.POST_NOTIFICATIONS)
                            }
                        }
                    } else {
                        battMonVM.startMonitor()
                    }
                } else {
                    battMonVM.stopMonitor()
                }
            }
        )
    }
}
