import 'dart:io';
import 'package:daily_you/backup_restore_utils.dart';
import 'package:daily_you/config_provider.dart';
import 'package:daily_you/database/app_database.dart';
import 'package:daily_you/database/image_storage.dart';
import 'package:daily_you/providers/entries_provider.dart';
import 'package:daily_you/widgets/settings_dropdown.dart';
import 'package:daily_you/widgets/settings_icon_action.dart';
import 'package:device_info_plus/device_info_plus.dart';
import 'package:flutter/material.dart';
import 'package:daily_you/l10n/generated/app_localizations.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:provider/provider.dart';

class StorageSettings extends StatefulWidget {
  const StorageSettings({super.key});

  @override
  State<StorageSettings> createState() => _StorageSettingsState();
}

class _StorageSettingsState extends State<StorageSettings> {
  @override
  void initState() {
    super.initState();
  }

  Future<bool> requestStoragePermission() async {
    DeviceInfoPlugin deviceInfoPlugin = DeviceInfoPlugin();
    AndroidDeviceInfo androidInfo = await deviceInfoPlugin.androidInfo;
    if (androidInfo.version.sdkInt < 33) {
      //Legacy Permission
      var status = await Permission.storage.status;
      if (!status.isGranted) {
        status = await Permission.storage.request();
      }
      if (status.isGranted) {
        return true;
      }
      return false;
    }

    //Modern Permission
    return true;
  }

  Future<bool> requestPhotosPermission() async {
    DeviceInfoPlugin deviceInfoPlugin = DeviceInfoPlugin();
    AndroidDeviceInfo androidInfo = await deviceInfoPlugin.androidInfo;
    if (androidInfo.version.sdkInt < 33) {
      //Legacy Permission
      var status = await Permission.storage.status;
      if (!status.isGranted) {
        status = await Permission.storage.request();
      }
      if (status.isGranted) {
        return true;
      }
      return false;
    } else {
      //Modern Photos Permission
      return true;
    }
  }

  Future<void> _showChangeLogFolderWarning() async {
    bool confirmed = false;
    await showDialog(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.warningTitle),
          content:
              Text(AppLocalizations.of(context)!.logFolderWarningDescription),
          actions: [
            TextButton(
              child: Text(MaterialLocalizations.of(context).cancelButtonLabel),
              onPressed: () async {
                Navigator.pop(context);
              },
            ),
            TextButton(
              child: Text(MaterialLocalizations.of(context).okButtonLabel),
              onPressed: () async {
                confirmed = true;
                Navigator.pop(context);
              },
            ),
          ],
        );
      },
    );
    if (confirmed) {
      ValueNotifier<String> statusNotifier = ValueNotifier<String>("");

      BackupRestoreUtils.showLoadingStatus(context, statusNotifier);
      bool locationSet =
          await AppDatabase.instance.selectExternalLocation((status) {
        statusNotifier.value = status;
      });

      Navigator.of(context).pop();

      if (!locationSet) {
        await showDialog(
            context: context,
            builder: (BuildContext context) {
              return AlertDialog(
                  title: Text(AppLocalizations.of(context)!.errorTitle),
                  actions: [
                    TextButton(
                      child:
                          Text(MaterialLocalizations.of(context).okButtonLabel),
                      onPressed: () async {
                        Navigator.pop(context);
                      },
                    ),
                  ],
                  content: Text(
                      AppLocalizations.of(context)!.logFolderErrorDescription));
            });
      }
    }
  }

  Future<void> _attemptImageFolderChange() async {
    ValueNotifier<String> statusNotifier = ValueNotifier<String>("");

    BackupRestoreUtils.showLoadingStatus(context, statusNotifier);
    bool locationSet =
        await ImageStorage.instance.selectExternalLocation((status) {
      statusNotifier.value = status;
    });

    Navigator.of(context).pop();

    if (!locationSet) {
      await showDialog(
          context: context,
          builder: (BuildContext context) {
            return AlertDialog(
                title: Text(AppLocalizations.of(context)!.errorTitle),
                actions: [
                  TextButton(
                    child:
                        Text(MaterialLocalizations.of(context).okButtonLabel),
                    onPressed: () async {
                      Navigator.pop(context);
                    },
                  ),
                ],
                content: Text(
                    AppLocalizations.of(context)!.imageFolderErrorDescription));
          });
    }
  }

  void _showDeleteAllLogsDialog(
      BuildContext context, String requiredText, VoidCallback onConfirm) {
    ThemeData theme = Theme.of(context);
    final TextEditingController controller = TextEditingController();
    final ValueNotifier<bool> isButtonEnabled = ValueNotifier(false);

    showDialog(
      context: context,
      builder: (context) {
        return AlertDialog(
          title: Text(AppLocalizations.of(context)!.settingsDeleteAllLogsTitle),
          content: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(AppLocalizations.of(context)!
                  .settingsDeleteAllLogsDescription),
              Padding(
                padding: const EdgeInsets.only(top: 8.0),
                child: Text(AppLocalizations.of(context)!
                    .settingsDeleteAllLogsPrompt(requiredText)),
              ),
              Padding(
                padding: const EdgeInsets.only(top: 8.0),
                child: Card(
                  child: Padding(
                    padding: const EdgeInsets.only(left: 8.0, right: 8.0),
                    child: TextField(
                      decoration: InputDecoration(
                          hintStyle: TextStyle(fontStyle: FontStyle.italic),
                          hintText: requiredText,
                          border: InputBorder.none),
                      controller: controller,
                      onChanged: (value) {
                        isButtonEnabled.value = value.toLowerCase().trim() ==
                            requiredText.toLowerCase().trim();
                      },
                    ),
                  ),
                ),
              ),
            ],
          ),
          actions: [
            TextButton(
              onPressed: () {
                Navigator.of(context).pop();
              },
              child: Text(MaterialLocalizations.of(context).cancelButtonLabel),
            ),
            ValueListenableBuilder<bool>(
              valueListenable: isButtonEnabled,
              builder: (context, isEnabled, child) {
                return TextButton(
                  onPressed: isEnabled
                      ? () {
                          Navigator.of(context).pop();
                          onConfirm();
                        }
                      : null,
                  child: Text(
                      style: TextStyle(
                          color: isEnabled ? null : theme.disabledColor),
                      MaterialLocalizations.of(context).deleteButtonTooltip),
                );
              },
            ),
          ],
        );
      },
    );
  }

  Future<void> _deleteAllLogs(BuildContext context) async {
    ValueNotifier<String> statusNotifier = ValueNotifier<String>("");

    BackupRestoreUtils.showLoadingStatus(context, statusNotifier);

    await EntriesProvider.instance.deleteAll((status) {
      statusNotifier.value = status;
    });

    Navigator.of(context).pop();
  }

  String _displayNameFromUri(String uriString) {
    try {
      final decoded = Uri.decodeFull(uriString);

      // Everything after the last slash
      final lastSegment = decoded.split('/').last;

      // Get the folder for directory URIs. URIs cannot be turned
      // into full paths.
      if (lastSegment.contains(':')) {
        return lastSegment.split(':').last;
      }
      return lastSegment;
    } catch (e) {
      // Fall back to URI string when it cannot be parsed
      return uriString;
    }
  }

  @override
  Widget build(BuildContext context) {
    final configProvider = Provider.of<ConfigProvider>(context);

    return Scaffold(
      appBar: AppBar(
        title: Text(AppLocalizations.of(context)!.settingsStorageTitle),
        centerTitle: true,
      ),
      body: ListView(
        children: [
          SettingsDropdown<String>(
              title: AppLocalizations.of(context)!.settingsImageQuality,
              value: configProvider.get(ConfigKey.imageQualityLevel),
              options: [
                DropdownMenuItem<String>(
                    value: ImageQuality.noCompression,
                    child: Text(AppLocalizations.of(context)!
                        .imageQualityNoCompression)),
                DropdownMenuItem<String>(
                    value: ImageQuality.high,
                    child:
                        Text(AppLocalizations.of(context)!.imageQualityHigh)),
                DropdownMenuItem<String>(
                    value: ImageQuality.medium,
                    child:
                        Text(AppLocalizations.of(context)!.imageQualityMedium)),
                DropdownMenuItem<String>(
                    value: ImageQuality.low,
                    child: Text(AppLocalizations.of(context)!.imageQualityLow)),
              ],
              onChanged: (String? newValue) {
                if (newValue != null) {
                  configProvider.set(ConfigKey.imageQualityLevel, newValue);
                }
              }),
          FutureBuilder(
              future: AppDatabase.instance.getInternalPath(),
              builder: (context, snapshot) {
                if (snapshot.hasData && snapshot.data != null) {
                  var folderText = snapshot.data!;
                  if (AppDatabase.instance.usingExternalLocation()) {
                    folderText = _displayNameFromUri(
                        configProvider.get(ConfigKey.externalDbUri));
                  }
                  return SettingsIconAction(
                    title: AppLocalizations.of(context)!.settingsLogFolder,
                    hint: folderText,
                    icon: Icon(Icons.folder_rounded),
                    secondaryIcon: Icon(Icons.refresh_rounded),
                    onPressed: () async {
                      if (Platform.isAndroid &&
                          !await requestStoragePermission()) {
                        return;
                      }
                      await _showChangeLogFolderWarning();
                    },
                    onSecondaryPressed: () async {
                      AppDatabase.instance.resetExternalLocation();
                    },
                  );
                }
                return const SizedBox();
              }),
          FutureBuilder(
              future: ImageStorage.instance.getInternalFolder(),
              builder: (context, snapshot) {
                if (snapshot.hasData && snapshot.data != null) {
                  var folderText = snapshot.data!;
                  if (ImageStorage.instance.usingExternalLocation()) {
                    folderText = _displayNameFromUri(
                        configProvider.get(ConfigKey.externalImgUri));
                  }
                  return SettingsIconAction(
                    title: AppLocalizations.of(context)!.settingsImageFolder,
                    hint: folderText,
                    icon: Icon(Icons.folder_rounded),
                    secondaryIcon: Icon(Icons.refresh_rounded),
                    onPressed: () async {
                      if (Platform.isAndroid &&
                          !await requestPhotosPermission()) {
                        return;
                      }
                      await _attemptImageFolderChange();
                    },
                    onSecondaryPressed: () async {
                      ImageStorage.instance.resetImageFolderLocation();
                    },
                  );
                }
                return const SizedBox();
              }),
          SettingsIconAction(
              title: AppLocalizations.of(context)!.settingsDeleteAllLogsTitle,
              icon: Icon(Icons.delete_forever_rounded),
              onPressed: () => _showDeleteAllLogsDialog(
                  context,
                  AppLocalizations.of(context)!.settingsDeleteAllLogsTitle,
                  () => _deleteAllLogs(context))),
        ],
      ),
    );
  }
}
