package com.darkrockstudios.apps.hammer.project.repository

import com.darkrockstudios.apps.hammer.base.ProjectId
import com.darkrockstudios.apps.hammer.base.http.ClientEntityState
import com.darkrockstudios.apps.hammer.dependencyinjection.PROJECTS_SYNC_MANAGER
import com.darkrockstudios.apps.hammer.dependencyinjection.PROJECT_SYNC_MANAGER
import com.darkrockstudios.apps.hammer.project.*
import com.darkrockstudios.apps.hammer.project.synchronizers.*
import com.darkrockstudios.apps.hammer.projects.ProjectsSynchronizationSession
import com.darkrockstudios.apps.hammer.syncsessionmanager.SyncSessionManager
import com.darkrockstudios.apps.hammer.utils.BaseTest
import com.darkrockstudios.apps.hammer.utils.TestClock
import io.mockk.coEvery
import io.mockk.mockk
import io.mockk.slot
import kotlinx.serialization.json.Json
import okio.FileSystem
import okio.fakefilesystem.FakeFileSystem
import org.junit.jupiter.api.BeforeEach
import org.koin.core.qualifier.named
import org.koin.dsl.bind
import org.koin.dsl.module
import kotlin.time.Clock

abstract class ProjectEntityRepositoryBaseTest : BaseTest() {
	protected val userId = 1L
	protected val projectDefinition = ProjectDefinition("Test Project", ProjectId("test-uuid"))

	protected lateinit var fileSystem: FileSystem
	protected lateinit var clock: TestClock

	protected lateinit var projectsSessionManager: SyncSessionManager<Long, ProjectsSynchronizationSession>
	protected lateinit var projectSessionManager: SyncSessionManager<ProjectSyncKey, ProjectSynchronizationSession>

	protected lateinit var sceneSynchronizer: ServerSceneSynchronizer
	protected lateinit var noteSynchronizer: ServerNoteSynchronizer
	protected lateinit var timelineEventSynchronizer: ServerTimelineSynchronizer
	protected lateinit var encyclopediaSynchronizer: ServerEncyclopediaSynchronizer
	protected lateinit var sceneDraftSynchronizer: ServerSceneDraftSynchronizer
	protected lateinit var projectEntityDatasource: ProjectEntityDatasource

	protected lateinit var clientState: ClientEntityState

	@BeforeEach
	override fun setup() {
		super.setup()

		fileSystem = FakeFileSystem()
		clock = TestClock(Clock.System)

		projectsSessionManager = mockk()
		projectSessionManager = mockk()

		sceneSynchronizer = mockk()
		noteSynchronizer = mockk()
		timelineEventSynchronizer = mockk()
		encyclopediaSynchronizer = mockk()
		sceneDraftSynchronizer = mockk()
		projectEntityDatasource = mockk()

		clientState = mockk()

		coEvery {
			sceneSynchronizer.getUpdateSequence(
				userId,
				projectDefinition,
				clientState
			)
		} returns emptyList()
		coEvery {
			noteSynchronizer.getUpdateSequence(
				userId,
				projectDefinition,
				clientState
			)
		} returns emptyList()
		coEvery {
			timelineEventSynchronizer.getUpdateSequence(
				userId,
				projectDefinition,
				clientState
			)
		} returns emptyList()
		coEvery {
			encyclopediaSynchronizer.getUpdateSequence(
				userId,
				projectDefinition,
				clientState
			)
		} returns emptyList()
		coEvery {
			sceneDraftSynchronizer.getUpdateSequence(
				userId,
				projectDefinition,
				clientState
			)
		} returns emptyList()

		val testModule = module {
			single { fileSystem } bind FileSystem::class
			single { Json } bind Json::class
			single { sceneSynchronizer }
			single { noteSynchronizer }
			single { timelineEventSynchronizer }
			single { encyclopediaSynchronizer }
			single { sceneDraftSynchronizer }
			single { clock } bind TestClock::class

			single<SyncSessionManager<Long, ProjectsSynchronizationSession>>(
				named(
					PROJECTS_SYNC_MANAGER
				)
			) {
				projectsSessionManager
			}

			single<SyncSessionManager<ProjectSyncKey, ProjectSynchronizationSession>>(
				named(
					PROJECT_SYNC_MANAGER
				)
			) {
				projectSessionManager
			}
		}
		setupKoin(testModule)
	}

	protected fun mockCreateSession(syncId: String) {
		val createSessionSlot =
			slot<(key: ProjectSyncKey, syncId: String) -> ProjectSynchronizationSession>()
		val key = ProjectSyncKey(userId, projectDefinition)
		coEvery {
			projectSessionManager.createNewSession(
				key,
				capture(createSessionSlot)
			)
		} coAnswers {
			val session = createSessionSlot.captured(key, syncId)
			session.syncId
		}
	}

	protected fun createProjectRepository(): ProjectEntityRepository {
		return ProjectEntityRepository(clock, projectEntityDatasource)
	}
}