package com.darkrockstudios.apps.hammer.common.storyeditor.scenelist

import androidx.compose.foundation.gestures.Orientation
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.material.FixedThreshold
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.MenuOpen
import androidx.compose.material.icons.outlined.Delete
import androidx.compose.material.icons.outlined.Edit
import androidx.compose.material.rememberSwipeableState
import androidx.compose.material.swipeable
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import com.darkrockstudios.apps.hammer.Res
import com.darkrockstudios.apps.hammer.common.compose.resources.get
import com.darkrockstudios.apps.hammer.common.data.SceneItem
import com.darkrockstudios.apps.hammer.scene_list_item_action_delete
import com.darkrockstudios.apps.hammer.scene_list_item_action_menu_description
import com.darkrockstudios.apps.hammer.scene_list_item_action_rename
import kotlin.math.roundToInt

@Composable
actual fun SceneItemActionContainer(
	scene: SceneItem,
	onSceneDeleteClick: (scene: SceneItem) -> Unit,
	onSceneRenameClick: (scene: SceneItem) -> Unit,
	shouldNux: Boolean,
	itemContent: @Composable (modifier: Modifier) -> Unit,
) {
	val hapticFeedback = LocalHapticFeedback.current
	var showMenu by remember { mutableStateOf(false) }
	val swipeableState = rememberSwipeableState(SwipeState.Initial, confirmStateChange = { value ->
		if (value == SwipeState.Swiped) {
			hapticFeedback.performHapticFeedback(HapticFeedbackType.LongPress)
			showMenu = true
		}
		false
	})
	val sizePx = with(LocalDensity.current) { 50.dp.toPx() }
	val anchors = mapOf(0f to SwipeState.Initial, sizePx to SwipeState.Swiped)

	SwipeNux(swipeableState, shouldNux)

	Box(
		modifier = Modifier
			.swipeable(
				state = swipeableState,
				anchors = anchors,
				thresholds = { _, _ -> FixedThreshold(50.dp) },
				orientation = Orientation.Horizontal
			)
	) {
		val alpha =
			if (!(swipeableState.progress.to == SwipeState.Initial && swipeableState.progress.from == SwipeState.Initial)) {
				swipeableState.progress.fraction
			} else {
				0f
			}

		Icon(
			Icons.Filled.MenuOpen,
			contentDescription = Res.string.scene_list_item_action_menu_description.get(),
			modifier = Modifier
				.align(Alignment.CenterStart)
				.alpha(alpha),
			tint = MaterialTheme.colorScheme.onSurface
		)

		Box(
			Modifier
				.offset { IntOffset(swipeableState.offset.value.roundToInt(), 0) }
				.wrapContentSize()
		) {
			itemContent(Modifier)
		}
	}

	DropdownMenu(
		expanded = showMenu,
		onDismissRequest = { showMenu = false }
	) {
		DropdownMenuItem(
			text = { Text(Res.string.scene_list_item_action_delete.get()) },
			onClick = {
				onSceneDeleteClick(scene)
				showMenu = false
			},
			leadingIcon = {
				Icon(
					Icons.Outlined.Delete,
					contentDescription = Res.string.scene_list_item_action_delete.get()
				)
			},
		)

		DropdownMenuItem(
			text = { Text(Res.string.scene_list_item_action_rename.get()) },
			onClick = {
				onSceneRenameClick(scene)
				showMenu = false
			},
			leadingIcon = {
				Icon(
					Icons.Outlined.Edit,
					contentDescription = Res.string.scene_list_item_action_rename.get()
				)
			},
		)
	}
}