package com.darkrockstudios.apps.hammer.common.data.projectbackup

import com.darkrockstudios.apps.hammer.common.data.ProjectDef
import com.darkrockstudios.apps.hammer.common.data.globalsettings.GlobalSettingsRepository
import com.darkrockstudios.apps.hammer.common.data.projectsrepository.ProjectsRepository
import com.darkrockstudios.apps.hammer.common.fileio.HPath
import com.darkrockstudios.apps.hammer.common.fileio.okio.toHPath
import com.darkrockstudios.apps.hammer.common.fileio.okio.toOkioPath
import com.darkrockstudios.apps.hammer.common.util.format
import com.darkrockstudios.apps.hammer.common.util.zip.unzipToDirectory
import com.darkrockstudios.apps.hammer.common.util.zip.zipDirectory
import io.github.aakira.napier.Napier
import kotlinx.datetime.*
import okio.FileNotFoundException
import okio.FileSystem
import okio.Path
import org.koin.core.component.KoinComponent
import kotlin.time.Clock
import kotlin.time.Instant

open class ProjectBackupRepository(
	protected val fileSystem: FileSystem,
	protected val projectsRepository: ProjectsRepository,
	protected val globalSettingsRepository: GlobalSettingsRepository,
	protected val clock: Clock
) : KoinComponent {

	fun getBackupsDirectory(): HPath {
		val dir = (projectsRepository.getProjectsDirectory().toOkioPath() / BACKUP_DIRECTORY)

		if (fileSystem.exists(dir).not()) {
			fileSystem.createDirectories(dir)
		}

		return dir.toHPath()
	}

	fun getBackups(projectDef: ProjectDef): List<ProjectBackupDef> {
		val dir = getBackupsDirectory().toOkioPath()
		return fileSystem.list(dir)
			.filter {
				try {
					fileSystem.metadata(it).isRegularFile
				} catch (_: FileNotFoundException) {
					false
				}
			}
			.mapNotNull { path -> getProjectBackupDef(path) }
			.filter { it.projectDef == projectDef }
			.sortedBy { it.date }
	}

	fun getBackupsForProject(projectDef: ProjectDef): List<ProjectBackupDef> {
		return getBackups(projectDef).filter { backup -> backup.projectDef == projectDef }
	}

	private fun backupNameToProjectName(backupName: String): String {
		return backupName.replace("_", " ")
	}

	protected fun projectNameToBackupName(projectName: String): String {
		return projectName.replace(" ", "_")
	}

	protected fun createNewProjectBackupDef(projectDef: ProjectDef): ProjectBackupDef {
		val path = pathForBackup(projectDef.name, clock.now())

		return ProjectBackupDef(
			path = path,
			projectDef = projectDef,
			date = clock.now()
		)
	}

	private fun pathForBackup(projectName: String, date: Instant): HPath {
		val filename = filenameForBackup(projectName, date)
		val dir = getBackupsDirectory().toOkioPath()
		return (dir / filename).toHPath()
	}

	private fun filenameForBackup(projectName: String, date: Instant): String {
		val backupName = projectNameToBackupName(projectName)
		val dateStr = date.toBackupDate()
		return "$backupName-$dateStr.zip"
	}

	fun deleteBackup(backup: ProjectBackupDef) {
		try {
			val path = backup.path.toOkioPath()
			if (fileSystem.exists(path)) {
				fileSystem.delete(path)
				Napier.i("Deleted backup: ${backup.path.name}")
			} else {
				Napier.w("Backup file not found: ${backup.path.name}")
			}
		} catch (e: Exception) {
			Napier.e("Failed to delete backup: ${backup.path.name}", e)
			throw e
		}
	}

	fun cullBackups(project: ProjectDef) {
		val settings = globalSettingsRepository.globalSettings

		val backups = getBackups(project).toMutableList()

		// Oldest first
		backups.sortBy { it.date }

		// Delete the oldest backups to get under budget
		if (backups.size > settings.maxBackups) {
			val overBudget = backups.size - settings.maxBackups
			Napier.i("Project '${project.name}' is over it's backup budget by $overBudget backups.")
			for (ii in 0 until overBudget) {
				val oldBackup = backups[ii]
				fileSystem.delete(oldBackup.path.toOkioPath())
				Napier.i("Deleted backup: ${oldBackup.path.name}")
			}
		}
	}

	private fun getProjectBackupDef(path: Path): ProjectBackupDef? {
		val match = FILE_NAME_PATTERN.matchEntire(path.name)
		return if (match != null) {
			val backupName = match.groups[1]?.value
			val date = match.groups[2]?.value

			if (backupName != null && date != null) {
				val projectName = backupNameToProjectName(backupName)

				val dateInstant = localDateTime(date).toInstant(TimeZone.UTC)
				val projectDir = projectsRepository.getProjectDirectory(projectName)

				val projectDef = ProjectDef(
					name = projectName,
					path = projectDir
				)

				ProjectBackupDef(
					path = path.toHPath(),
					projectDef = projectDef,
					date = dateInstant
				)
			} else {
				null
			}
		} else {
			null
		}
	}

	open fun supportsBackup(): Boolean = true

	open suspend fun createBackup(projectDef: ProjectDef): ProjectBackupDef? {
		val projectDir = projectsRepository.getProjectDirectory(projectDef.name).toOkioPath()
		val newBackupDef = createNewProjectBackupDef(projectDef)

		return try {
			zipDirectory(
				fileSystem = fileSystem,
				sourceDirectory = projectDir,
				destinationZip = newBackupDef.path.toOkioPath(),
				skipHiddenFiles = false
			)

			cullBackups(projectDef)

			newBackupDef
		} catch (e: Exception) {
			Napier.e("Failed to make backup for project: ${projectDef.name}", e)
			null
		}
	}

	open suspend fun restoreBackup(backupDef: ProjectBackupDef, targetDir: HPath): Boolean {
		return try {
			val targetOkioPath = targetDir.toOkioPath()
			fileSystem.deleteRecursively(targetOkioPath)
			fileSystem.createDirectories(targetOkioPath)

			unzipToDirectory(
				fileSystem = fileSystem,
				zipPath = backupDef.path.toOkioPath(),
				destinationDirectory = targetOkioPath
			)
			true
		} catch (e: Exception) {
			Napier.e("Failed to restore backup: ${backupDef.path.name}", e)
			false
		}
	}

	private fun localDateTime(dateTimeStr: String): LocalDateTime {
		val match = DATE_PATTERN.matchEntire(dateTimeStr) ?: throw IllegalArgumentException("Failed to parse date time")
		val year = match.groupValues[1].toInt()
		val month = match.groupValues[2].toInt()
		val day = match.groupValues[3].toInt()
		val hour = match.groupValues[4].toInt()
		val minute = match.groupValues[5].toInt()
		val second = match.groupValues[6].toInt()

		return LocalDateTime(
			date = LocalDate(
				year = year,
				monthNumber = month,
				dayOfMonth = day
			),
			time = LocalTime(hour, minute, second)
		)
	}

	companion object {
		const val BACKUP_DIRECTORY = ".backups"
		val FILE_NAME_PATTERN = Regex("^([a-zA-Z0-9_]+)-(\\d{4}-\\d{2}-\\d{2}T\\d+Z)\\.zip$")
		val DATE_PATTERN = Regex("^(\\d{4})-(\\d{2})-(\\d{2})T(\\d{2})(\\d{2})(\\d{2})Z$")
	}
}

private fun Instant.toBackupDate(): String {
	val dateTime = toLocalDateTime(TimeZone.UTC)
	val dateStr = dateTime.format("yyyy-MM-dd'T'HHmmss'Z'")

	return dateStr
}