import com.android.build.api.dsl.androidLibrary
import org.jetbrains.kotlin.gradle.ExperimentalKotlinGradlePluginApi
import org.jetbrains.kotlin.gradle.dsl.JvmTarget

plugins {
	alias(libs.plugins.kotlin.multiplatform)
	alias(libs.plugins.kotlin.serialization)
	alias(libs.plugins.kotlin.powerassert)
	alias(libs.plugins.android.kotlin.multiplatform.library)
	alias(libs.plugins.kotlin.parcelize)
	alias(libs.plugins.jetbrains.kover)
	alias(libs.plugins.jetbrains.compose)
	alias(libs.plugins.compose.compiler)
	alias(libs.plugins.buildconfig)
}

group = "com.darkrockstudios.apps.hammer"
version = libs.versions.app.get()

kotlin {
	androidLibrary {
		namespace = "com.darkrockstudios.apps.hammer.common"
		compileSdk = libs.versions.android.sdk.compile.get().toInt()
		minSdk = libs.versions.android.sdk.min.get().toInt()

		androidResources {
			enable = true
		}

		compilerOptions {
			jvmTarget.set(JvmTarget.fromTarget(libs.versions.jvm.get()))
		}
	}
	jvm("desktop") {
		compilerOptions {
			jvmTarget.set(JvmTarget.fromTarget(libs.versions.jvm.get()))
		}
	}

	listOf(
		iosX64(),
		iosArm64(),
		iosSimulatorArm64()
	).forEach { iosTarget ->
		iosTarget.binaries.framework {
			baseName = "Hammer"
			//isStatic = true
			//transitiveExport = true
			export(libs.decompose)
			export(libs.essenty.lifecycle)
			export(libs.coroutines.core)
			export(libs.napier)
		}
	}

	applyDefaultHierarchyTemplate()

	sourceSets {
		all {
			languageSettings {
				optIn("kotlin.io.encoding.ExperimentalEncodingApi")
				optIn("kotlin.uuid.ExperimentalUuidApi")
			}
		}

		val commonMain by getting {
			resources.srcDirs("resources")

			dependencies {
				api(project(":base"))

				api(libs.decompose)
				api(libs.napier)
				api(libs.coroutines.core)
				api(project.dependencies.platform(libs.koin.bom.get()))
				api(libs.koin.core)
				api(libs.okio)

				implementation(libs.bundles.ktor.client)
				implementation(libs.ktor.serialization.kotlinx.json)

				api(libs.serialization.core)
				api(libs.serialization.json)
				api(libs.kotlinx.datetime)
				implementation(libs.tomlkt)
				api(libs.bundles.essenty)
				implementation(libs.cache4k)
				implementation(compose.runtime)
				implementation(compose.components.resources)
				implementation(libs.kotlinx.atomicfu)
				implementation(libs.fluidsonic.locale)
				implementation(libs.aboutlibraries.core)
				implementation(libs.multiplatform.settings)
				implementation(libs.platform.spellcheckerkt)
				implementation(libs.kompress.core)
			}
		}
		val commonTest by getting {
			dependencies {
				implementation(kotlin("test"))
				implementation(libs.koin.test)
				implementation(libs.okio.fakefilesystem)
				implementation(libs.kotlin.reflect)
			}
		}
		val jvmMain by creating {
			dependsOn(commonMain)
		}
		val jvmTest by creating {
			dependsOn(commonTest)
		}
		val androidMain by getting {
			dependsOn(jvmMain)
			dependencies {
				api(libs.androidx.core.ktx)
				api(libs.coroutines.android)
				implementation(libs.koin.android)
				implementation(libs.ktor.client.okhttp)
				implementation(libs.moko.permissions)
				implementation(libs.moko.permissions.storage)
			}
		}
		val iosMain by getting {
			dependencies {
				api(libs.decompose)
				api(libs.bundles.essenty)
				api(libs.ktor.client.darwin)
			}
		}
		val iosTest by getting
		val desktopMain by getting {
			dependsOn(jvmMain)
			dependencies {
				implementation(libs.slf4j.simple)
				api(libs.serialization.jvm)
				api(libs.coroutines.swing)
				implementation(libs.appdirs)
				implementation(libs.ktor.client.java)
				implementation(libs.turbine)
			}
		}
		val desktopTest by getting {
			dependsOn(jvmTest)
			dependencies {
				implementation(libs.bundles.junit.jupiter)
				implementation(libs.coroutines.test)
				implementation(libs.mockk)
				implementation(libs.koin.test)
				implementation(libs.koin.test.junit5)
			}
		}
	}
}

compose.resources {
	publicResClass = true
	packageOfResClass = "com.darkrockstudios.apps.hammer"
}

buildConfig {
	useKotlinOutput { internalVisibility = false }
	packageName("com.darkrockstudios.apps.hammer.common")

	val isDebug = project.findProperty("hammer.debug")?.toString()?.toBoolean() ?: false
	buildConfigField("Boolean", "DEBUG", isDebug.toString())
}

kover {
	reports {
		filters {
			includes {
				packages("com.darkrockstudios.apps.hammer.*")
			}
			excludes {
				packages(
					"com.darkrockstudios.apps.hammer.util.*",
					"com.darkrockstudios.apps.hammer.parcelize.*",
					"com.darkrockstudios.apps.hammer.fileio.*",
				)
			}
		}
	}
}

@OptIn(ExperimentalKotlinGradlePluginApi::class)
powerAssert {
	functions = listOf(
		"kotlin.assert",
		"kotlin.test.assertTrue",
		"kotlin.test.assertEquals",
		"kotlin.test.assertNull"
	)
	includedSourceSets = listOf("commonTest", "desktopTest")
}