import { combine } from '../public-utils/combine';
import { once } from '../public-utils/once';
import { addAttribute } from '../util/add-attribute';
function copyReverse(array) {
  return array.slice(0).reverse();
}
export function makeDropTarget({
  typeKey,
  defaultDropEffect
}) {
  const registry = new WeakMap();
  const dropTargetDataAtt = `data-drop-target-for-${typeKey}`;
  const dropTargetSelector = `[${dropTargetDataAtt}]`;
  function addToRegistry(args) {
    registry.set(args.element, args);
    return () => registry.delete(args.element);
  }
  function dropTargetForConsumers(args) {
    // Guardrail: warn if the draggable element is already registered
    if (process.env.NODE_ENV !== 'production') {
      const existing = registry.get(args.element);
      if (existing) {
        // eslint-disable-next-line no-console
        console.warn(`You have already registered a [${typeKey}] dropTarget on the same element`, {
          existing,
          proposed: args
        });
      }
      if (args.element instanceof HTMLIFrameElement) {
        // eslint-disable-next-line no-console
        console.warn(`
            We recommend not registering <iframe> elements as drop targets
            as it can result in some strange browser event ordering.
          `
        // Removing newlines and excessive whitespace
        .replace(/\s{2,}/g, ' ').trim());
      }
    }
    const cleanup = combine(addAttribute(args.element, {
      attribute: dropTargetDataAtt,
      value: 'true'
    }), addToRegistry(args));

    // Wrapping in `once` to prevent unexpected side effects if consumers call
    // the clean up function multiple times.
    return once(cleanup);
  }
  function getActualDropTargets({
    source,
    target,
    input,
    result = []
  }) {
    var _args$getData, _args$getData2, _args$getDropEffect, _args$getDropEffect2;
    if (target == null) {
      return result;
    }
    if (!(target instanceof Element)) {
      // For "text-selection" drags, the original `target`
      // is not an `Element`, so we need to start looking from
      // the parent element
      if (target instanceof Node) {
        return getActualDropTargets({
          source,
          target: target.parentElement,
          input,
          result
        });
      }

      // not sure what we are working with,
      // so we can exit.
      return result;
    }
    const closest = target.closest(dropTargetSelector);

    // Cannot find anything else
    if (closest == null) {
      return result;
    }
    const args = registry.get(closest);

    // error: something had a dropTargetSelector but we could not
    // find a match. For now, failing silently
    if (args == null) {
      return result;
    }
    const feedback = {
      input,
      source,
      element: args.element
    };

    // if dropping is not allowed, skip this drop target
    // and continue looking up the DOM tree
    if (args.canDrop && !args.canDrop(feedback)) {
      return getActualDropTargets({
        source,
        target: args.element.parentElement,
        input,
        result
      });
    }

    // calculate our new record
    const data = (_args$getData = (_args$getData2 = args.getData) === null || _args$getData2 === void 0 ? void 0 : _args$getData2.call(args, feedback)) !== null && _args$getData !== void 0 ? _args$getData : {};
    const dropEffect = (_args$getDropEffect = (_args$getDropEffect2 = args.getDropEffect) === null || _args$getDropEffect2 === void 0 ? void 0 : _args$getDropEffect2.call(args, feedback)) !== null && _args$getDropEffect !== void 0 ? _args$getDropEffect : defaultDropEffect;
    const record = {
      data,
      element: args.element,
      dropEffect,
      // we are collecting _actual_ drop targets, so these are
      // being applied _not_ due to stickiness
      isActiveDueToStickiness: false
    };
    return getActualDropTargets({
      source,
      target: args.element.parentElement,
      input,
      // Using bubble ordering. Same ordering as `event.getPath()`
      result: [...result, record]
    });
  }
  function notifyCurrent({
    eventName,
    payload
  }) {
    for (const record of payload.location.current.dropTargets) {
      var _entry$eventName;
      const entry = registry.get(record.element);
      const args = {
        ...payload,
        self: record
      };
      entry === null || entry === void 0 ? void 0 : (_entry$eventName = entry[eventName]) === null || _entry$eventName === void 0 ? void 0 : _entry$eventName.call(entry,
      // I cannot seem to get the types right here.
      // TS doesn't seem to like that one event can need `nativeSetDragImage`
      // @ts-expect-error
      args);
    }
  }
  const actions = {
    onGenerateDragPreview: notifyCurrent,
    onDrag: notifyCurrent,
    onDragStart: notifyCurrent,
    onDrop: notifyCurrent,
    onDropTargetChange: ({
      payload
    }) => {
      const isCurrent = new Set(payload.location.current.dropTargets.map(record => record.element));
      const visited = new Set();
      for (const record of payload.location.previous.dropTargets) {
        var _entry$onDropTargetCh;
        visited.add(record.element);
        const entry = registry.get(record.element);
        const isOver = isCurrent.has(record.element);
        const args = {
          ...payload,
          self: record
        };
        entry === null || entry === void 0 ? void 0 : (_entry$onDropTargetCh = entry.onDropTargetChange) === null || _entry$onDropTargetCh === void 0 ? void 0 : _entry$onDropTargetCh.call(entry, args);

        // if we cannot find the drop target in the current array, then it has been left
        if (!isOver) {
          var _entry$onDragLeave;
          entry === null || entry === void 0 ? void 0 : (_entry$onDragLeave = entry.onDragLeave) === null || _entry$onDragLeave === void 0 ? void 0 : _entry$onDragLeave.call(entry, args);
        }
      }
      for (const record of payload.location.current.dropTargets) {
        var _entry$onDropTargetCh2, _entry$onDragEnter;
        // already published an update to this drop target
        if (visited.has(record.element)) {
          continue;
        }
        // at this point we have a new drop target that is being entered into
        const args = {
          ...payload,
          self: record
        };
        const entry = registry.get(record.element);
        entry === null || entry === void 0 ? void 0 : (_entry$onDropTargetCh2 = entry.onDropTargetChange) === null || _entry$onDropTargetCh2 === void 0 ? void 0 : _entry$onDropTargetCh2.call(entry, args);
        entry === null || entry === void 0 ? void 0 : (_entry$onDragEnter = entry.onDragEnter) === null || _entry$onDragEnter === void 0 ? void 0 : _entry$onDragEnter.call(entry, args);
      }
    }
  };
  function dispatchEvent(args) {
    actions[args.eventName](args);
  }
  function getIsOver({
    source,
    target,
    input,
    current
  }) {
    const actual = getActualDropTargets({
      source,
      target,
      input
    });

    // stickiness is only relevant when we have less
    // drop targets than we did before
    if (actual.length >= current.length) {
      return actual;
    }

    // less 'actual' drop targets than before,
    // we need to see if 'stickiness' applies

    // An old drop target will continue to be dropped on if:
    // 1. it has the same parent
    // 2. nothing exists in it's previous index

    const lastCaptureOrdered = copyReverse(current);
    const actualCaptureOrdered = copyReverse(actual);
    const resultCaptureOrdered = [];
    for (let index = 0; index < lastCaptureOrdered.length; index++) {
      var _argsForLast$getIsSti;
      const last = lastCaptureOrdered[index];
      const fresh = actualCaptureOrdered[index];

      // if a record is in the new index -> use that
      // it will have the latest data + dropEffect
      if (fresh != null) {
        resultCaptureOrdered.push(fresh);
        continue;
      }

      // At this point we have no drop target in the old spot
      // Check to see if we can use a previous sticky drop target

      // The "parent" is the one inside of `resultCaptureOrdered`
      // (the parent might be a drop target that was sticky)
      const parent = resultCaptureOrdered[index - 1];
      const lastParent = lastCaptureOrdered[index - 1];

      // Stickiness is based on parent relationships, so if the parent relationship has change
      // then we can stop our search
      if ((parent === null || parent === void 0 ? void 0 : parent.element) !== (lastParent === null || lastParent === void 0 ? void 0 : lastParent.element)) {
        break;
      }

      // We need to check whether the old drop target can still be dropped on

      const argsForLast = registry.get(last.element);

      // We cannot drop on a drop target that is no longer mounted
      if (!argsForLast) {
        break;
      }
      const feedback = {
        input,
        source,
        element: argsForLast.element
      };

      // We cannot drop on a drop target that no longer allows being dropped on
      if (argsForLast.canDrop && !argsForLast.canDrop(feedback)) {
        break;
      }

      // We cannot drop on a drop target that is no longer sticky
      if (!((_argsForLast$getIsSti = argsForLast.getIsSticky) !== null && _argsForLast$getIsSti !== void 0 && _argsForLast$getIsSti.call(argsForLast, feedback))) {
        break;
      }

      // Note: intentionally not recollecting `getData()` or `getDropEffect()`
      // Previous values for `data` and `dropEffect` will be borrowed
      // This is to prevent things like the 'closest edge' changing when
      // no longer over a drop target.
      // We could change our mind on this behaviour in the future

      resultCaptureOrdered.push({
        ...last,
        // making it clear to consumers this drop target is active due to stickiness
        isActiveDueToStickiness: true
      });
    }

    // return bubble ordered result
    return copyReverse(resultCaptureOrdered);
  }
  return {
    dropTargetForConsumers,
    getIsOver,
    dispatchEvent
  };
}