package eu.siacs.conversations.generator;

import eu.siacs.conversations.Config;
import eu.siacs.conversations.crypto.axolotl.AxolotlService;
import eu.siacs.conversations.crypto.axolotl.XmppAxolotlMessage;
import eu.siacs.conversations.entities.Account;
import eu.siacs.conversations.entities.Conversation;
import eu.siacs.conversations.entities.Conversational;
import eu.siacs.conversations.entities.Message;
import eu.siacs.conversations.services.XmppConnectionService;
import eu.siacs.conversations.xml.Element;
import eu.siacs.conversations.xml.Namespace;
import eu.siacs.conversations.xmpp.Jid;
import eu.siacs.conversations.xmpp.chatstate.ChatState;
import eu.siacs.conversations.xmpp.forms.Data;
import eu.siacs.conversations.xmpp.jingle.JingleConnectionManager;
import eu.siacs.conversations.xmpp.jingle.JingleRtpConnection;
import eu.siacs.conversations.xmpp.jingle.Media;
import eu.siacs.conversations.xmpp.jingle.stanzas.Reason;
import im.conversations.android.xmpp.model.correction.Replace;
import im.conversations.android.xmpp.model.hints.NoStore;
import im.conversations.android.xmpp.model.hints.Store;
import im.conversations.android.xmpp.model.reactions.Reaction;
import im.conversations.android.xmpp.model.reactions.Reactions;
import im.conversations.android.xmpp.model.receipts.Received;
import im.conversations.android.xmpp.model.unique.OriginId;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

public class MessageGenerator extends AbstractGenerator {
    private static final String OMEMO_FALLBACK_MESSAGE =
            "I sent you an OMEMO encrypted message but your client doesn’t seem to support that."
                    + " Find more information on https://conversations.im/omemo";
    private static final String PGP_FALLBACK_MESSAGE =
            "I sent you a PGP encrypted message but your client doesn’t seem to support that.";

    public MessageGenerator(XmppConnectionService service) {
        super(service);
    }

    private im.conversations.android.xmpp.model.stanza.Message preparePacket(
            final Message message, final boolean legacyEncryption) {
        Conversation conversation = (Conversation) message.getConversation();
        Account account = conversation.getAccount();
        im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        final boolean isWithSelf = conversation.getContact().isSelf();
        if (conversation.getMode() == Conversation.MODE_SINGLE) {
            packet.setTo(message.getCounterpart());
            packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
            if (!isWithSelf) {
                packet.addChild("request", "urn:xmpp:receipts");
            }
        } else if (message.isPrivateMessage()) {
            packet.setTo(message.getCounterpart());
            packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
            packet.addChild("x", "http://jabber.org/protocol/muc#user");
            packet.addChild("request", "urn:xmpp:receipts");
        } else {
            packet.setTo(message.getCounterpart().asBareJid());
            packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.GROUPCHAT);
        }
        if (conversation.isSingleOrPrivateAndNonAnonymous() && !message.isPrivateMessage()) {
            packet.addChild("markable", "urn:xmpp:chat-markers:0");
        }
        packet.setFrom(account.getJid());
        packet.setId(message.getUuid());
        if (conversation.getMode() == Conversational.MODE_MULTI
                && !message.isPrivateMessage()
                && !conversation.getMucOptions().stableId()) {
            packet.addExtension(new OriginId(message.getUuid()));
        }
        if (message.edited()) {
            packet.addExtension(new Replace(message.getEditedIdWireFormat()));
        }
        if (!legacyEncryption) {
            if (message.getSubject() != null && message.getSubject().length() > 0) packet.addChild("subject").setContent(message.getSubject());
            // Legacy encryption can't handle advanced payloads
            for (Element el : message.getPayloads()) {
                packet.addChild(el);
            }
        } else {
            for (Element el : message.getPayloads()) {
                if ("thread".equals(el.getName())) packet.addChild(el);
            }
        }
        return packet;
    }

    public void addDelay(
            im.conversations.android.xmpp.model.stanza.Message packet, long timestamp) {
        final SimpleDateFormat mDateFormat =
                new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.US);
        mDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        Element delay = packet.addChild("delay", "urn:xmpp:delay");
        Date date = new Date(timestamp);
        delay.setAttribute("stamp", mDateFormat.format(date));
    }

    public im.conversations.android.xmpp.model.stanza.Message generateAxolotlChat(
            Message message, XmppAxolotlMessage axolotlMessage) {
        im.conversations.android.xmpp.model.stanza.Message packet = preparePacket(message, true);
        if (axolotlMessage == null) {
            return null;
        }
        packet.setAxolotlMessage(axolotlMessage.toElement());
        packet.setBody(OMEMO_FALLBACK_MESSAGE);
        packet.addExtension(new Store());
        packet.addChild("encryption", "urn:xmpp:eme:0")
                .setAttribute("name", "OMEMO")
                .setAttribute("namespace", AxolotlService.PEP_PREFIX);
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message generateKeyTransportMessage(
            Jid to, XmppAxolotlMessage axolotlMessage) {
        im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
        packet.setTo(to);
        packet.setAxolotlMessage(axolotlMessage.toElement());
        packet.addChild(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message generateChat(Message message) {
        im.conversations.android.xmpp.model.stanza.Message packet = preparePacket(message, false);
        String content;
        if (message.hasFileOnRemoteHost()) {
            final Message.FileParams fileParams = message.getFileParams();

            if (message.getFallbacks(Namespace.OOB).isEmpty()) {
                if (message.getBody().equals("")) {
                    message.setBody(fileParams.url);
                    final var fallback = new Element("fallback", "urn:xmpp:fallback:0").setAttribute("for", Namespace.OOB);
                    fallback.addChild("body", "urn:xmpp:fallback:0");
                    message.addPayload(fallback);
                } else {
                    long start = message.getRawBody().codePointCount(0, message.getRawBody().length());
                    message.appendBody(fileParams.url);
                    final var fallback = new Element("fallback", "urn:xmpp:fallback:0").setAttribute("for", Namespace.OOB);
                    fallback.addChild("body", "urn:xmpp:fallback:0")
                        .setAttribute("start", String.valueOf(start))
                        .setAttribute("end", String.valueOf(start + fileParams.url.length()));
                    message.addPayload(fallback);
                }
            }

            packet = preparePacket(message, false);
            packet.addChild("x", Namespace.OOB).addChild("url").setContent(fileParams.url);
        }
        if (message.getRawBody() != null) packet.setBody(message.getRawBody());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message generatePgpChat(Message message) {
        final im.conversations.android.xmpp.model.stanza.Message packet = preparePacket(message, true);
        if (message.hasFileOnRemoteHost()) {
            Message.FileParams fileParams = message.getFileParams();
            final String url = fileParams.url;
            packet.setBody(url);
            packet.addChild("x", Namespace.OOB).addChild("url").setContent(url);
            packet.addChild("fallback", "urn:xmpp:fallback:0").setAttribute("for", Namespace.OOB)
                  .addChild("body", "urn:xmpp:fallback:0");
        } else {
            if (Config.supportUnencrypted()) {
                packet.setBody(PGP_FALLBACK_MESSAGE);
            }
            if (message.getEncryption() == Message.ENCRYPTION_DECRYPTED) {
                packet.addChild("x", "jabber:x:encrypted").setContent(message.getEncryptedBody());
            } else if (message.getEncryption() == Message.ENCRYPTION_PGP) {
                packet.addChild("x", "jabber:x:encrypted").setContent(message.getBody());
            }
            packet.addChild("encryption", "urn:xmpp:eme:0")
                    .setAttribute("namespace", "jabber:x:encrypted");
        }
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message generateChatState(
            Conversation conversation) {
        final Account account = conversation.getAccount();
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                conversation.getMode() == Conversation.MODE_MULTI
                        ? im.conversations.android.xmpp.model.stanza.Message.Type.GROUPCHAT
                        : im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
        packet.setTo(conversation.getJid().asBareJid());
        packet.setFrom(account.getJid());
        packet.addChild(ChatState.toElement(conversation.getOutgoingChatState()));
        packet.addExtension(new NoStore());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message confirm(final Message message) {
        final boolean groupChat = message.getConversation().getMode() == Conversational.MODE_MULTI;
        final Jid to = message.getCounterpart();
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                groupChat
                        ? im.conversations.android.xmpp.model.stanza.Message.Type.GROUPCHAT
                        : im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
        packet.setTo(groupChat ? to.asBareJid() : to);
        final Element displayed = packet.addChild("displayed", "urn:xmpp:chat-markers:0");
        if (groupChat) {
            final String stanzaId = message.getServerMsgId();
            if (stanzaId != null) {
                displayed.setAttribute("id", stanzaId);
            } else {
                displayed.setAttribute("sender", to.toString());
                displayed.setAttribute("id", message.getRemoteMsgId());
            }
        } else {
            displayed.setAttribute("id", message.getRemoteMsgId());
        }
        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message reaction(
            final Jid to,
            final boolean groupChat,
            final Message inReplyTo,
            final String reactingTo,
            final Collection<String> ourReactions) {
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                groupChat
                        ? im.conversations.android.xmpp.model.stanza.Message.Type.GROUPCHAT
                        : im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
        packet.setTo(to);
        final var reactions = packet.addExtension(new Reactions());
        reactions.setId(reactingTo);
        for (final String ourReaction : ourReactions) {
            reactions.addExtension(new Reaction(ourReaction));
        }

        final var thread = inReplyTo.getThread();
        if (thread != null) packet.addChild(thread);

        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message requestVoice(Jid jid) {
        final var packet = new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.NORMAL);
        packet.setTo(jid.asBareJid());
        final var form = new Data();
        form.setFormType("http://jabber.org/protocol/muc#request");
        form.put("muc#role", "participant");
        form.submit();
        packet.addChild(form);
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message received(
            final Jid from,
            final String id,
            final im.conversations.android.xmpp.model.stanza.Message.Type type) {
        final var receivedPacket = new im.conversations.android.xmpp.model.stanza.Message();
        receivedPacket.setType(type);
        receivedPacket.setTo(from);
        receivedPacket.addExtension(new Received(id));
        receivedPacket.addExtension(new Store());
        return receivedPacket;
    }

    public im.conversations.android.xmpp.model.stanza.Message received(
            Account account, Jid to, String id) {
        im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setFrom(account.getJid());
        packet.setTo(to);
        packet.addChild("received", "urn:xmpp:receipts").setAttribute("id", id);
        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message sessionFinish(
            final Jid with, final String sessionId, final Reason reason) {
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(im.conversations.android.xmpp.model.stanza.Message.Type.CHAT);
        packet.setTo(with);
        final Element finish = packet.addChild("finish", Namespace.JINGLE_MESSAGE);
        finish.setAttribute("id", sessionId);
        final Element reasonElement = finish.addChild("reason", Namespace.JINGLE);
        reasonElement.addChild(reason.toString());
        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message sessionProposal(
            final JingleConnectionManager.RtpSessionProposal proposal) {
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                im.conversations.android.xmpp.model.stanza.Message.Type
                        .CHAT); // we want to carbon copy those
        packet.setTo(proposal.with);
        packet.setId(JingleRtpConnection.JINGLE_MESSAGE_PROPOSE_ID_PREFIX + proposal.sessionId);
        final Element propose = packet.addChild("propose", Namespace.JINGLE_MESSAGE);
        propose.setAttribute("id", proposal.sessionId);
        for (final Media media : proposal.media) {
            propose.addChild("description", Namespace.JINGLE_APPS_RTP)
                    .setAttribute("media", media.toString());
        }
        packet.addChild("request", "urn:xmpp:receipts");
        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message sessionRetract(
            final JingleConnectionManager.RtpSessionProposal proposal) {
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                im.conversations.android.xmpp.model.stanza.Message.Type
                        .CHAT); // we want to carbon copy those
        packet.setTo(proposal.with);
        final Element propose = packet.addChild("retract", Namespace.JINGLE_MESSAGE);
        propose.setAttribute("id", proposal.sessionId);
        propose.addChild("description", Namespace.JINGLE_APPS_RTP);
        packet.addExtension(new Store());
        return packet;
    }

    public im.conversations.android.xmpp.model.stanza.Message sessionReject(
            final Jid with, final String sessionId) {
        final im.conversations.android.xmpp.model.stanza.Message packet =
                new im.conversations.android.xmpp.model.stanza.Message();
        packet.setType(
                im.conversations.android.xmpp.model.stanza.Message.Type
                        .CHAT); // we want to carbon copy those
        packet.setTo(with);
        final Element propose = packet.addChild("reject", Namespace.JINGLE_MESSAGE);
        propose.setAttribute("id", sessionId);
        propose.addChild("description", Namespace.JINGLE_APPS_RTP);
        packet.addExtension(new Store());
        return packet;
    }
}
