// File: src/gui/mod.rs
pub mod async_ops;
pub mod icon;
pub mod message;
pub mod state;
pub mod subscription;
pub mod update;
pub mod view;

use crate::config::{AppTheme, Config};
use crate::gui::message::Message;
use crate::gui::state::GuiApp;
use crate::system::spawn_alarm_actor;
use iced::futures::SinkExt;
use iced::futures::channel::mpsc::Sender;
use iced::stream;
use iced::{Element, Subscription, Task, Theme, font, window};
use std::sync::Arc;

pub fn run() -> iced::Result {
    async_ops::init_runtime();

    iced::application(GuiApp::new, GuiApp::update, GuiApp::view)
        .title(GuiApp::title)
        .subscription(GuiApp::subscription)
        .theme(GuiApp::theme)
        .window(window::Settings {
            decorations: false,
            platform_specific: window::settings::PlatformSpecific {
                #[cfg(target_os = "linux")]
                application_id: String::from("cfait"),
                ..Default::default()
            },
            ..Default::default()
        })
        .run()
}

// Helper function to satisfy Subscription::run fn pointer requirement
fn alarm_stream() -> impl iced::futures::Stream<Item = Message> {
    stream::channel(100, |mut output: Sender<Message>| async move {
        let (gui_tx, mut gui_rx) = tokio::sync::mpsc::channel(10);
        let actor_tx = spawn_alarm_actor(Some(gui_tx));
        let _ = output.send(Message::InitAlarmActor(actor_tx)).await;

        while let Some(msg) = gui_rx.recv().await {
            let _ = output
                .send(Message::AlarmSignalReceived(Arc::new(msg)))
                .await;
        }

        std::future::pending::<()>().await;
    })
}

impl GuiApp {
    fn new() -> (Self, Task<Message>) {
        (
            Self::default(),
            Task::batch(vec![
                Task::perform(
                    async { Config::load().map_err(|e| e.to_string()) },
                    Message::ConfigLoaded,
                ),
                font::load(icon::FONT_BYTES).map(|_| Message::FontLoaded(Ok(()))),
            ]),
        )
    }

    fn view(&self) -> Element<'_, Message> {
        view::root_view(self)
    }

    fn title(&self) -> String {
        "Cfait | 🗹 Take control of your TODO list".to_string()
    }

    fn theme(&self) -> Theme {
        match self.current_theme {
            AppTheme::Dark => Theme::Dark,
            AppTheme::RustyDark => {
                let dark_palette = iced::Theme::Dark.palette();
                Theme::custom(
                    "Rusty Dark",
                    iced::theme::Palette {
                        background: iced::Color::from_rgb8(0x21, 0x1e, 0x1e),
                        primary: iced::Color::from_rgb(1.0, 0.6, 0.0),
                        ..dark_palette
                    },
                )
            }
        }
    }

    fn subscription(&self) -> Subscription<Message> {
        let subs = subscription::subscription(self);
        let alarm_sub = Subscription::run(alarm_stream);
        Subscription::batch(vec![subs, alarm_sub])
    }

    fn update(&mut self, message: Message) -> Task<Message> {
        update::update(self, message)
    }
}
