import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:hive_ce/hive.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/hive/hive_registrar.g.dart';
import 'package:miniature_painting_companion/injector.dart';
import 'package:path_provider_platform_interface/path_provider_platform_interface.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:test/test.dart';
import 'package:uuid/uuid.dart';

import 'mocks/mock_path_provider_platform.dart';
import 'test_utils.dart';

void main() {
  group('Test model exports and import', () {
    late Directory tempDir;

    setUpAll(() async {
      WidgetsFlutterBinding.ensureInitialized();

      Hive.init((await Directory.systemTemp.createTemp()).path);
      Hive.registerAdapters();

      SharedPreferences.setMockInitialValues({});

      await setupInjector();
    });

    setUp(() async {
      await paintRepo.add(Paint(
          type: "type",
          name: "name",
          image: generateTestImage().path,
          h: 0,
          s: 0,
          v: 0));
      tempDir = await Directory.systemTemp.createTemp('file_storage_test');
      PathProviderPlatform.instance = MockPathProviderPlatform(tempDir.path);
    });

    tearDown(() async {
      await miniatureRepo.clear();
      await historyImagesRepo.clear();
      await paintJobRepo.clear();
      await iDrawableRepo.clear();
      await paintRepo.clear();
      await positionedRepo.clear();

      await tempDir.delete(recursive: true);
    });

    tearDownAll(() {
      clearTestImages();
    });

    group('Miniature export', () {
      test('Simple miniature should export and import Correctly', () async {
        var miniature = Miniature(
            name: "name",
            imagePath: generateTestImage().path,
            paintJobs: paintJobRepo.createList(),
            historyImages: historyImagesRepo.createList(),
            importTime: DateTime.now());
        var export = miniature.toJson();
        await Miniature.fromJson(export);

        final captured = miniatureRepo.values().elementAt(0);

        expect(captured.name, equals(miniature.name));
        expect(captured.imagePath, isNot(miniature.imagePath));
        expect(captured.imagePath, isNot(miniature.imagePath));
        expect(File(captured.imagePath!).readAsBytesSync(),
            File(miniature.imagePath!).readAsBytesSync());
      });
    });

    group('Paint job export', () {
      test('Exporting miniature with paint jobs should bind them on import',
          () async {
        var holder = Miniature(
            name: "name",
            imagePath: null,
            paintJobs: paintJobRepo.createList(),
            historyImages: historyImagesRepo.createList(),
            importTime: DateTime.now());
        await miniatureRepo.add(holder);

        var paintJob = PaintJob(
            imagePath: generateTestImage().path,
            name: "name",
            paintLayers: iDrawableRepo.createList(),
            positioned: positionedRepo.createList(),
            importTime: DateTime.now());
        await holder.addPaintJob(paintJob);

        var export = paintJob.toJson();
        await PaintJob.fromJson(export);

        final captured = miniatureRepo.values().elementAt(0);
        expect(captured.paintJobs.length, 1);
        expect(paintRepo.values().length, greaterThan(0));

        expect(captured.paintJobs.first, paintJob);
      });

      test('Simple PaintJob should export and importCorrectly', () async {
        var paintJob = PaintJob(
            imagePath: generateTestImage().path,
            name: "name",
            paintLayers: iDrawableRepo.createList(),
            positioned: positionedRepo.createList(),
            importTime: DateTime.now());
        var export = paintJob.toJson();
        await PaintJob.fromJson(export);

        final captured = paintJobRepo.values().elementAt(0);

        expect(captured.name, equals(paintJob.name));
        expect(captured.imagePath, isNot(paintJob.imagePath));
        expect(captured.imagePath, isNot(paintJob.imagePath));
        expect(File(captured.imagePath!).readAsBytesSync(),
            File(paintJob.imagePath!).readAsBytesSync());
      });

      test('PaintJob with layers should export and importCorrectly', () async {
        var holder = Miniature(
            name: "name",
            imagePath: generateTestImage().path,
            paintJobs: paintJobRepo.createList(),
            historyImages: historyImagesRepo.createList(),
            importTime: DateTime.now());
        await miniatureRepo.add(holder);

        var paintJob = PaintJob(
            name: "name",
            paintLayers: iDrawableRepo.createList(),
            positioned: positionedRepo.createList(),
            importTime: DateTime.now());
        holder.addPaintJob(paintJob);
        var paint = Paint(
            type: "type",
            name: "name",
            image: generateTestImage().path,
            h: 0,
            s: 0,
            v: 0);

        var paintLayer = PaintLayer(
            name: "paintLayer",
            paint: paint,
            note: "note",
            applicationType: ApplicationType.dryBrush,
            dilution: 100);
        await paintJob.addPaintLayer(paintLayer);

        var separatorLayer = LayerSeparator("separatorLayer");
        await paintJob.addPaintLayer(separatorLayer);

        var stickerLayer =
            StickerLayer("stickerLayer", generateTestImage().path);
        await paintJob.addPaintLayer(stickerLayer);

        var combinedLayer = CombinedLayer(
            name: "combinedLayer", layers: iDrawableRepo.createList());
        await paintJob.addPaintLayer(combinedLayer);
        var paintLayer2 = PaintLayer(
            name: "paintLayer2",
            paint: paint,
            note: "note",
            applicationType: ApplicationType.dryBrush,
            dilution: 100);
        await combinedLayer.addLayer(paintLayer2);

        var export = paintJob.toJson();
        await PaintJob.fromJson(export);

        final captured = miniatureRepo.values().elementAt(0);

        expect(holder, captured);
        expect(holder.paintJobs[0].paintLayers.length, 4);
        var paintList = holder.paintJobs[0].paintLayers;
        expect((paintList[0] as PaintLayer).name, paintLayer.name);
        expect((paintList[1] as LayerSeparator).text, separatorLayer.text);
        expect((paintList[2] as StickerLayer).name, stickerLayer.name);
        expect((paintList[3] as CombinedLayer).name, combinedLayer.name);
        expect((paintList[3] as CombinedLayer).layers.length, 1);
        expect(((paintList[3] as CombinedLayer).layers[0] as PaintLayer).name,
            paintLayer2.name);
      });
    });

    group('History image export', () {
      test('Simple History image should export and importCorrectly', () async {
        var historyImage =
            HistoryImage(generateTestImage().path, DateTime.now());
        var export = historyImage.toJson();
        await HistoryImage.fromJson(export);

        final captured = historyImagesRepo.values().elementAt(0);
        expect(historyImagesRepo.values().length, 1);

        expect(captured.importTime, equals(historyImage.importTime));
        expect(captured.imagePath, isNot(historyImage.imagePath));
        expect(captured.imagePath, isNot(historyImage.imagePath));
        expect(File(captured.imagePath).readAsBytesSync(),
            File(historyImage.imagePath).readAsBytesSync());
      });

      test('Exporting miniature with history image should bind them on import',
          () async {
        var holder = Miniature(
            name: "name",
            imagePath: generateTestImage().path,
            paintJobs: paintJobRepo.createList(),
            historyImages: historyImagesRepo.createList(),
            importTime: DateTime.now());
        await miniatureRepo.add(holder);

        var historyImage =
            HistoryImage(generateTestImage().path, DateTime.now());
        await holder.addHistoryImage(historyImage);

        var export = holder.toJson();
        await Miniature.fromJson(export);

        final captured = miniatureRepo.values().elementAt(0);
        expect(captured.historyImages.length, 1);
        expect(captured.historyImages.first, historyImage);
      });
    });

    group('Layers export', () {
      test('PaintLayer export should be correct', () async {
        var paint = Paint(
            type: "type",
            name: "name",
            image: generateTestImage().path,
            h: 0,
            s: 0,
            v: 0);

        var paintLayer = PaintLayer(
            name: "paintLayer",
            paint: paint,
            note: "note",
            applicationType: ApplicationType.dryBrush,
            dilution: 100);

        var export = paintLayer.toJson();
        var imported = await PaintLayer.fromJson(export);
        expect("paintLayer", imported.name);
        expect(paint.id, imported.paint.id);
        expect(paint.name, imported.paint.name);
        expect("note", imported.note);
        expect(ApplicationType.dryBrush, imported.applicationType);
        expect(100, imported.dilution);
      });

      test('PaintLayer export should be correct with UserPaint', () async {
        var paint = Paint(
            type: "type",
            name: "name",
            image: generateTestImage().path,
            h: 0,
            s: 0,
            v: 0);
        var compo = UserPaintComponent(paint: paint, concentration: 75);
        var userPaint = UserPaint(
            id: Uuid().v4(),
            type: "otherType",
            name: "otherName",
            components: [compo]);

        var paintLayer = PaintLayer(
            name: "paintLayer",
            paint: userPaint,
            note: "note",
            applicationType: ApplicationType.dryBrush,
            dilution: 100);

        // Put's it in the repo the be able to recover it for the layer
        await paintRepo.add(userPaint);

        var export = paintLayer.toJson();
        var imported = await PaintLayer.fromJson(export);
        expect("paintLayer", imported.name);
        expect(userPaint.id, imported.paint.id);
        expect(userPaint.name, imported.paint.name);
        expect(userPaint.type, imported.paint.type);
        expect(true, imported.paint is UserPaint);
        var importedUser = imported.paint as UserPaint;
        expect(1, importedUser.components.length);
        expect(paint, importedUser.components[0].paint);
        expect(75, importedUser.components[0].concentration);

        expect("note", imported.note);
        expect(ApplicationType.dryBrush, imported.applicationType);
        expect(100, imported.dilution);
      });

      test('SeparatorLayer export should be correct', () async {
        var separatorLayer = LayerSeparator("separatorLayer");

        var export = separatorLayer.toJson();
        var import = await LayerSeparator.fromJson(export);
        expect("separatorLayer", import.text);
      });

      test('StickerLayer export should be correct', () async {
        var stickerLayer =
            StickerLayer("stickerLayer", generateTestImage().path);
        var export = stickerLayer.toJson();
        var import = await StickerLayer.fromJson(export);
        expect("stickerLayer", import.name);
        expect(stickerLayer.imagePath, isNot(import.imagePath));
        expect(File(stickerLayer.imagePath!).readAsBytesSync(),
            File(import.imagePath!).readAsBytesSync());
      });

      test('CombinedLayer export should be correct', () async {
        var paint = Paint(
            type: "type",
            name: "name",
            image: generateTestImage().path,
            h: 0,
            s: 0,
            v: 0);

        var paintLayer = PaintLayer(
            name: "paintLayer",
            paint: paint,
            note: "note",
            applicationType: ApplicationType.dryBrush,
            dilution: 100);
        iDrawableRepo.add(paintLayer);
        var combinedLayer = CombinedLayer(
            name: "combinedLayer",
            layers: iDrawableRepo.createList(objects: [paintLayer]));
        var export = combinedLayer.toJson();
        var import = await CombinedLayer.fromJson(export);
        expect("combinedLayer", import.name);
        expect(1, import.layers.length);
        expect("paintLayer", (import.layers[0] as PaintLayer).name);
      });
    });

    group('Paint export', () {
      test('Paint export should be correct', () async {
        var paint = Paint(
            type: "type",
            name: "name",
            image: generateTestImage().path,
            h: 0,
            s: 1,
            v: 2);

        var export = paint.toJson();
        var imported = Paint.fromJson(export);
        expect(paint.name, imported.name);
        expect(paint.type, imported.type);
        expect(paint.image, imported.image);
        expect(paint.h, imported.h);
        expect(paint.s, imported.s);
        expect(paint.v, imported.v);
      });

      test('Paint export should be correct with componenent', () async {
        var paint2 = Paint(type: "type2", name: "name2");

        var paint3 = Paint(type: "type3", name: "name3");

        var paint =
            UserPaint(id: Uuid().v4(), type: "type", name: "name", components: [
          UserPaintComponent(paint: paint2, concentration: 100),
          UserPaintComponent(paint: paint3, concentration: 50)
        ]);

        var export = paint.toJson();
        var imported = await UserPaint.fromJson(export);
        expect(imported.id, paint.id);
        expect(imported.components[0].paint.name, paint2.name);
        expect(imported.components[0].concentration, 100);
        expect(imported.components[1].paint.name, paint3.name);
        expect(imported.components[1].concentration, 50);
      });
    });
  });
}
