import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:hive_ce/hive.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/hive/hive_registrar.g.dart';
import 'package:miniature_painting_companion/injector.dart';
import 'package:path_provider_platform_interface/path_provider_platform_interface.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'mocks/mock_path_provider_platform.dart';
import 'test_utils.dart';

void main() {
  group('Test models and DB', () {
    late Directory tempDir;

    setUpAll(() async {
      WidgetsFlutterBinding.ensureInitialized();

      Hive.init((await Directory.systemTemp.createTemp()).path);
      Hive.registerAdapters();

      SharedPreferences.setMockInitialValues({});

      await setupInjector();
    });

    setUp(() async {
      await paintRepo
          .add(Paint(type: "type", name: "testing_paint", h: 0, s: 0, v: 0));

      tempDir = await Directory.systemTemp.createTemp('file_storage_test');
      PathProviderPlatform.instance = MockPathProviderPlatform(tempDir.path);
    });

    tearDown(() async {
      await miniatureRepo.clear();
      await historyImagesRepo.clear();
      await paintJobRepo.clear();
      await iDrawableRepo.clear();
      await paintRepo.clear();
      await positionedRepo.clear();

      await tempDir.delete(recursive: true);
    });

    tearDownAll(() {
      clearTestImages();
    });

    group('Deletion tests', () {
      test('Models should clear their file image on delete', () async {
        var miniImage = generateTestImage().path;
        var paintJobImage = generateTestImage().path;
        var stickerImage = generateTestImage().path;
        var historyImage = generateTestImage().path;

        Miniature miniature = await addMiniature(miniImage);

        PaintJob paintJob = await addPaintJob(paintJobImage, miniature);

        await addHistoryImage(historyImage, miniature);

        await addStickerLayer(stickerImage, paintJob);

        PaintLayer paintLayer = addpaintLayer();

        PositionedLayer positioned =
            await addPositionedLayer(paintJob, paintLayer);

        PaintLayer secondPaintLayer = addpaintLayer();
        await positioned.addLayer(secondPaintLayer);

        UserPaint customPaint = await addUserPaint();

        await miniatureRepo.delete(miniature);

        expect(File(miniImage).existsSync(), false);
        expect(File(paintJobImage).existsSync(), false);
        expect(File(historyImage).existsSync(), false);
        expect(File(stickerImage).existsSync(), false);

        expect(miniatureRepo.values().isEmpty, true);
        expect(historyImagesRepo.values().isEmpty, true);
        expect(paintJobRepo.values().isEmpty, true);
        expect(iDrawableRepo.values().isEmpty, true);
        expect(positionedRepo.values().isEmpty, true);

        expect(
            paintRepo
                .values()
                .toList()
                .any((act) => act.key == customPaint.key),
            true);
        expect(
            paintRepo
                .values()
                .toList()
                .any((act) => act.name == "testing_paint"),
            true);
      });

      test('Cloned models should not reuse the same files', () async {
        var miniImage = generateTestImage().path;
        var paintJobImage = generateTestImage().path;
        var stickerImage = generateTestImage().path;
        var historyImage = generateTestImage().path;

        Miniature miniature = await addMiniature(miniImage);

        PaintJob paintJob = await addPaintJob(paintJobImage, miniature);

        HistoryImage history = await addHistoryImage(historyImage, miniature);

        StickerLayer sticker = await addStickerLayer(stickerImage, paintJob);

        expect((await miniature.clone()).imagePath, isNot(miniature.imagePath));
        expect((await paintJob.clone()).imagePath, isNot(paintJob.imagePath));
        expect((await history.clone()).imagePath, isNot(history.imagePath));
        expect((await sticker.clone()).imagePath, isNot(sticker.imagePath));
      });

      test('Original should work correctly after clone deletion', () async {
        var miniImage = generateTestImage().path;
        var paintJobImage = generateTestImage().path;
        var stickerImage = generateTestImage().path;
        var historyImage = generateTestImage().path;

        Miniature miniature = await addMiniature(miniImage);
        (await miniature.clone()).delete();

        (await miniature.clone()).delete();
        miniature.delete();

        PaintJob paintJob =
            await addPaintJob(paintJobImage, await addMiniature(miniImage));

        (await paintJob.clone()).delete();
        paintJob.delete();

        HistoryImage history =
            await addHistoryImage(historyImage, await addMiniature(miniImage));

        (await history.clone()).delete();
        history.delete();

        StickerLayer sticker = await addStickerLayer(stickerImage,
            await addPaintJob(paintJobImage, await addMiniature(miniImage)));

        (await sticker.clone()).delete();
        sticker.delete();
      });
    });
  });
}
