import 'package:flutter/material.dart';
import 'package:hive_ce/hive.dart';
import 'package:receive_sharing_intent/receive_sharing_intent.dart';

import '../../Models/hive_models.dart';
import '../../injector.dart';
import '../../main.dart';
import '../PaintJob/paint_jobs_screen.dart';
import '../design/nothing_created_widget.dart';
import '../design/tittle_card.dart';
import '../miniature/new_miniature_dialog.dart';

class SharingIntentContent extends StatefulWidget {
  const SharingIntentContent({
    super.key,
    required List<SharedMediaFile> sharedFiles,
  }) : _sharedFiles = sharedFiles;

  final List<SharedMediaFile> _sharedFiles;

  @override
  State<StatefulWidget> createState() {
    return SharingIntentContentState();
  }
}

class SharingIntentContentState extends State<SharingIntentContent> {
  bool showMiniaturePicker = false;

  @override
  Widget build(BuildContext context) {
    return showMiniaturePicker
        ? miniaturePickerContent()
        : buildDefaultContent();
  }

  Widget miniaturePickerContent() {
    return Column(
      mainAxisSize: MainAxisSize.min,
      children: [
        Align(
          alignment: Alignment.centerLeft,
          child: FilledButton.icon(
            icon: Icon(Icons.arrow_back),
            onPressed: () {
              setState(() {
                showMiniaturePicker = false;
              });
            },
            label: const Text('Back'),
          ),
        ),
        Expanded(
          child: ValueListenableBuilder(
              valueListenable: miniatureRepo.listenable(),
              builder: (context, Box box, _) {
                if (box.length > 0) {
                  return GridView.builder(
                    shrinkWrap: true,
                    padding: const EdgeInsets.fromLTRB(0, 0, 0, 100),
                    itemCount: box.length,
                    itemBuilder: (context, listIndex) {
                      var miniature = box.getAt(listIndex);
                      return TittleCard(
                        imagePath: miniature.imagePath,
                        title: miniature.name,
                        onTap: () {
                          toPaintJob(miniature);
                          Navigator.pop(context);
                          Navigator.push(
                            context,
                            MaterialPageRoute(
                              builder: (context) => PaintJobsScreen(
                                miniature: miniature,
                              ),
                            ),
                          );
                        },
                      );
                    },
                    gridDelegate:
                        const SliverGridDelegateWithFixedCrossAxisCount(
                      crossAxisCount: 2,
                    ),
                  );
                } else {
                  return NothingCreatedWidget(
                      title: localizations.noMiniatureCreated,
                      onCreateClick: () {
                        NewMiniatureDialog.show(context, null);
                      });
                }
              }),
        ),
        TextButton(
          style: ButtonStyle(visualDensity: VisualDensity.compact),
          onPressed: () {
            Navigator.pop(context);
          },
          child: const Text('Close'),
        )
      ],
    );
  }

  Widget buildDefaultContent() {
    return Column(
      mainAxisSize: MainAxisSize.min,
      mainAxisAlignment: MainAxisAlignment.center,
      // spacing: 15, // This property does not exist on Column
      children: <Widget>[
        Padding(
          padding: const EdgeInsets.only(top: 8),
          child: Text(
            "Shared ${widget._sharedFiles.length.toString()} image(s)",
            style: const TextStyle(fontFamily: 'Gothic', fontSize: 24),
          ),
        ),
        FilledButton(
          onPressed: () {
            toMiniature();
            Navigator.pop(context);
          },
          child: const Text('Import as miniature'),
        ),
        FilledButton(
          onPressed: () {
            setState(() {
              showMiniaturePicker = true;
            });
          },
          child: const Text('Import as details in miniature'),
        ),
        TextButton(
          style: ButtonStyle(visualDensity: VisualDensity.compact),
          onPressed: () {
            Navigator.pop(context);
          },
          child: const Text('Close'),
        ),
      ],
    );
  }

  void toMiniature() {
    widget._sharedFiles.map((added) {
      return Miniature(
          imagePath: added.path,
          name: null,
          paintJobs: paintJobRepo.createList(),
          historyImages: historyImagesRepo.createList(),
          importTime: DateTime.now());
    }).forEach((created) => miniatureRepo.add(created));
  }

  void toPaintJob(Miniature holder) {
    widget._sharedFiles.map((added) {
      return PaintJob(
        imagePath: added.path,
        name: null,
        paintLayers: iDrawableRepo.createList(),
        positioned: positionedRepo.createList(),
        importTime: DateTime.now(),
      );
    }).forEach((created) => holder.addPaintJob(created));
  }
}
