import 'package:flutter/material.dart';
import 'package:image_picker/image_picker.dart';
import 'package:miniature_painting_companion/injector.dart';

import 'package:miniature_painting_companion/views/design/app_sizes.dart';
import 'package:miniature_painting_companion/views/design/textual_button.dart';
import 'package:miniature_painting_companion/views/design/updatable_image.dart';
import 'package:reactive_forms/reactive_forms.dart';

import '../../main.dart';
import '../../utils/app_icons.dart';

class ReactiveImagePicker extends StatefulWidget {
  final FormControl<String> formControl;
  final String? initialValue;
  final String? holderImage;

  const ReactiveImagePicker(
      {super.key,
      required this.formControl,
      this.initialValue,
      this.holderImage});

  @override
  _ReactiveImagePickerState createState() => _ReactiveImagePickerState();
}

class _ReactiveImagePickerState extends State<ReactiveImagePicker> {
  late final GlobalKey<UpdatableImageState> _imageKey;

  @override
  void initState() {
    super.initState();
    _imageKey = GlobalKey<UpdatableImageState>();
    widget.formControl.value = widget.initialValue;
  }

  Future<void> _pickImage(ImageSource source) async {
    final pickedFile = await imagePickerService.pickImage(source);
    if (pickedFile?.path != null) {
      _imageKey.currentState?.changeImage(pickedFile!.path);
      widget.formControl.value = pickedFile!.path;
    }
  }

  Future<void> _useHolderImage() async {
    if (widget.holderImage != null) {
      _imageKey.currentState?.changeImage(widget.holderImage);
      widget.formControl.value = widget.holderImage;
    }
  }

  @override
  Widget build(BuildContext context) {
    return Column(
      children: [
        IconButton(
          icon: ClipOval(
            child: SizedBox.fromSize(
              size: const Size.fromRadius(64),
              child: UpdatableImage(
                key: _imageKey,
                initialValue: widget.initialValue,
              ),
            ),
          ),
          tooltip: localizations.takePicture,
          onPressed: () async {
            if (_imageKey.currentState?.imagePath == null) {
              await _pickImage(ImageSource.camera);
            } else {
              imagePickerService.showImageEditor(
                  _imageKey.currentState!.imagePath!, context);
            }
          },
        ),
        const SizedBox(height: AppSizes.size16),
        if (widget.holderImage != null)
          TextualButton(
            onPressed: () => _useHolderImage(),
            text: localizations.useMiniatureImage,
            iconName: AppIcons.plus.path,
          ),
        if (widget.holderImage != null)
          SizedBox(
            height: AppSizes.size16,
          ),
        Row(
          children: [
            Expanded(
              child: TextualButton(
                onPressed: () => _pickImage(ImageSource.gallery),
                text: localizations.choosePicture,
                iconName: AppIcons.uploadFile.path,
              ),
            ),
            SizedBox(
              width: AppSizes.size8,
            ),
            Expanded(
              child: TextualButton(
                onPressed: () => _pickImage(ImageSource.camera),
                text: localizations.takePicture,
                iconName: AppIcons.takePicture.path,
              ),
            )
          ],
        ),
        const SizedBox(height: AppSizes.size16),
      ],
    );
  }
}
