import 'dart:io';

import 'package:collection/collection.dart';
import 'package:flutter/material.dart';
import 'package:miniature_painting_companion/main.dart';
import 'package:miniature_painting_companion/views/design/app_paddings.dart';
import 'package:miniature_painting_companion/views/design/app_sizes.dart';

import '../../Models/hive_models.dart';
import '../../injector.dart';
import '../../utils/app_icons.dart';
import '../miniature/new_miniature_dialog.dart';
import 'history_image_list.dart';
import 'new_paint_job_dialog.dart';
import 'paint_job_widget.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

class PaintJobsScreen extends StatefulWidget {
  final Miniature miniature;

  const PaintJobsScreen({super.key, required this.miniature});

  @override
  State<PaintJobsScreen> createState() => _PaintJobList();
}

class _PaintJobList extends State<PaintJobsScreen> {
  bool sortByName = false;

  _PaintJobList();

  List<PaintJob> getSortedContent() {
    var toSort = widget.miniature.paintJobs.toList();
    if (sortByName) {
      toSort.sortBy((a) => a.name ?? "z");
    }
    return toSort;
  }

  @override
  Widget build(BuildContext context) {
    var miniature = widget.miniature;

    return Scaffold(
      appBar: AppBar(
        title: Text(miniature.name ?? "Miniature"),
        actions: [
          SizedBox(
            width: 8,
          ),
          InkWell(
            onTap: () async {
              await NewMiniatureDialog.show(context, widget.miniature);
              setState(() {});
            },
            child: SizedBox(
                height: AppSizes.size50,
                width: AppSizes.size50,
                child: CircleAvatar(
                  radius: AppSizes.size48,
                  backgroundImage: miniature.imagePath != null
                      ? FileImage(File(widget.miniature.imagePath!))
                      : null,
                  backgroundColor: Colors.grey.withAlpha(50),
                  child: miniature.imagePath == null
                      ? Text(
                          (miniature.name == null || miniature.name!.isEmpty)
                              ? "?"
                              : miniature.name![0],
                          style: const TextStyle(
                              fontFamily: 'Gothic', fontSize: 24))
                      : null,
                )),
          ),
          SizedBox(
            width: AppSizes.size8,
          )
        ],
      ),
      body: ListenableBuilder(
        listenable: Listenable.merge([
          paintJobRepo.listenable(),
          miniatureRepo.listenable(),
          historyImagesRepo.listenable(),
        ]),
        builder: (context, _) {
          return Column(
            children: [
              Material(
                color: Colors.grey.withAlpha(30),
                child: MiniatureHistoryList(
                  miniature: miniature,
                ),
              ),
              Padding(
                padding: AppPaddings.symmetricHorizontal8,
                child: Row(
                  children: [
                    Row(
                      children: [
                        SizedBox(
                          width: AppSizes.size8,
                          height: AppSizes.size60,
                        ),
                        Text(
                          "Details",
                          style: TextStyle(fontSize: 18),
                        )
                      ],
                    ),
                    Spacer(),
                    Row(
                      mainAxisAlignment: MainAxisAlignment.end,
                      children: [
                        if (miniature.paintJobs.length > 1)
                          IconButton(
                            visualDensity: VisualDensity.comfortable,
                            onPressed: () => setState(() {
                              sortByName = !sortByName;
                            }),
                            icon: sortByName
                                ? const Icon(Icons.sort_by_alpha)
                                : const Icon(Icons.sort),
                          ),
                      ],
                    ),
                  ],
                ),
              ),
              // Can't rely solely on this as the refresh might happen before the new object is add to the layer of our current paintJob
              Expanded(
                child: Builder(builder: (context) {
                  if (miniature.paintJobs.isNotEmpty) {
                    return GridView.builder(
                      padding: AppPaddings.bottom200,
                      itemCount: miniature.paintJobs.length,
                      shrinkWrap: true,
                      itemBuilder: (context, listIndex) {
                        return PaintJobWidget(
                          paintJob: getSortedContent().elementAt(listIndex),
                          allPaintJobs: getSortedContent(),
                        );
                      },
                      gridDelegate:
                          const SliverGridDelegateWithFixedCrossAxisCount(
                        crossAxisCount: 2,
                      ),
                    );
                  } else {
                    return Center(
                      child: Column(
                        mainAxisAlignment: MainAxisAlignment.center,
                        children: [
                          Text("No PaintJob created",
                              style: const TextStyle(
                                  fontSize: 22, fontWeight: FontWeight.normal)),
                          SizedBox(
                            height: AppSizes.size8,
                          ),
                          FilledButton(
                              onPressed: () async =>
                                  await NewPaintJobDialog.show(
                                      context, miniature, null),
                              child: Text('Create'))
                        ],
                      ),
                    );
                  }
                }),
              )
            ],
          );
        },
      ),
      floatingActionButton: FloatingActionButton(
        onPressed: () async {
          await NewPaintJobDialog.show(context, miniature, null);
          setState(() {});
        },
        tooltip: localizations.addPaintJob,
        child: ImageAssetService.getSvgIconSized(
            AppSizes.size40, AppSizes.size40, AppIcons.plus),
      ),
    );
  }
}
