import 'package:flutter/material.dart';
import 'package:image_picker/image_picker.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/injector.dart';
import 'package:miniature_painting_companion/main.dart';
import 'package:persistent_bottom_nav_bar_v2/persistent_bottom_nav_bar_v2.dart';
import 'package:reactive_forms/reactive_forms.dart';

import '../../validators/at_least_one_validator.dart';
import '../Layers/layers_screen.dart';
import '../design/reactive_crud_dialog.dart';
import '../design/reactive_image_picker.dart';

final PAINT_JOB_NAME = "miniName";
final PAINT_JOB_IMAGE_PATH = "miniImagePath";

class NewPaintJobDialog {
  static Future<bool?> show(context, Miniature? miniature, PaintJob? toUpdate) {
    final form = FormGroup(validators: [
      AtLeastOneValidator(
          controllerNames: [PAINT_JOB_NAME, PAINT_JOB_IMAGE_PATH])
    ], {
      PAINT_JOB_NAME: FormControl<String>(value: toUpdate?.name),
      PAINT_JOB_IMAGE_PATH: FormControl<String>(
        value: toUpdate?.imagePath,
      ),
    });

    var content = [
      ReactiveImagePicker(
          formControl:
              form.control(PAINT_JOB_IMAGE_PATH) as FormControl<String>,
          holderImage: miniature?.imagePath,
          initialValue: toUpdate?.imagePath),
      ReactiveTextField(
        formControlName: PAINT_JOB_NAME,
        textCapitalization: TextCapitalization.sentences,
        decoration: InputDecoration(
          border: const OutlineInputBorder(),
          hintText: localizations.name,
          labelText: localizations.name,
        ),
      ),
    ];

    onUpdate(BuildContext context) async => {
          toUpdate?.name = form.control(PAINT_JOB_NAME).value,
          if (form.control(PAINT_JOB_IMAGE_PATH).value != null)
            {
              toUpdate?.imagePath =
                  (await fileStorageService.moveFileToAppStorage(
                          XFile(form.control(PAINT_JOB_IMAGE_PATH).value)))
                      ?.path,
            },
          await paintJobRepo.save(toUpdate)
        };

    onSave(BuildContext saveContext) async {
      var paintJob = PaintJob(
          imagePath: form.control(PAINT_JOB_IMAGE_PATH).value == null
              ? null
              : (await fileStorageService.moveFileToAppStorage(
                      XFile(form.control(PAINT_JOB_IMAGE_PATH).value)))
                  ?.path,
          name: form.control(PAINT_JOB_NAME).value,
          paintLayers: iDrawableRepo.createList(),
          positioned: positionedRepo.createList(),
          importTime: DateTime.now());
      await miniature!.addPaintJob(paintJob);
      pushWithNavBar(
        context,
        MaterialPageRoute(
          builder: (context) => LayersScreen(
            paintJob: paintJob,
          ),
        ),
      );
    }

    return ReactiveCRUDDialog.show(
        context,
        toUpdate,
        content,
        form,
        localizations.createPaintJob,
        localizations.updatePaintJob,
        onSave,
        onUpdate);
  }
}
