import 'dart:io';

import 'package:collection/collection.dart';
import 'package:exif/exif.dart';
import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:hive_ce_flutter/hive_flutter.dart';
import 'package:image_picker/image_picker.dart';
import 'package:intl/intl.dart';
import 'package:logger/logger.dart';

import 'package:miniature_painting_companion/views/design/app_paddings.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

import '../../Models/hive_models.dart';
import '../../injector.dart';
import '../../main.dart';
import '../../utils/app_icons.dart';
import '../design/app_sizes.dart';
import '../design/custom_pop_up_menu.dart';
import '../design/image_views.dart';
import '../design/updatable_image.dart';

final logger = Logger();

class MiniatureHistoryList extends StatefulWidget {
  final Miniature miniature;

  const MiniatureHistoryList({super.key, required this.miniature});

  @override
  State<MiniatureHistoryList> createState() => _MiniatureHistoryListState();
}

class _MiniatureHistoryListState extends State<MiniatureHistoryList> {
  @override
  Widget build(BuildContext context) {
    return ValueListenableBuilder(
        valueListenable: miniatureRepo.listenable(),
        builder: (context, Box box, _) {
          List<Widget> content = [
            AddMiniatureHistoryCard(
                index: 0,
                miniature: widget.miniature,
                onAdded: () {
                  setState(() {});
                })
          ];

          // Will put the most recent first
          List<HistoryImage>? images = widget.miniature.historyImages
              .toList()
              .sorted((a, b) => b.importTime.compareTo(a.importTime));

          images
              .asMap()
              .forEach((index, element) => content.add(MiniatureHistoryCard(
                    index: index,
                    image: images,
                    holder: widget.miniature,
                  )));

          if (images.isEmpty) {
            content.add(SizedBox(
              width: AppSizes.size24,
            ));
            content.add(
              Center(
                child: Text("No history image",
                    style: const TextStyle(
                        fontSize: AppSizes.size22,
                        fontWeight: FontWeight.normal)),
              ),
            );
          }

          return Padding(
            padding: AppPaddings.symmetricHorizontal8,
            child: ConstrainedBox(
                constraints: const BoxConstraints(
                    maxHeight: 150, minWidth: double.infinity),
                child: ListView.builder(
                    scrollDirection: Axis.horizontal,
                    itemCount: content.length,
                    shrinkWrap: true,
                    itemBuilder: (BuildContext context, int index) {
                      return content[index];
                    })),
          );
        });
  }
}

class MiniatureHistoryCard extends StatelessWidget {
  const MiniatureHistoryCard({
    super.key,
    required this.index,
    required this.image,
    required this.holder,
  });

  final int index;
  final List<HistoryImage> image;
  final Miniature holder;

  @override
  Widget build(BuildContext context) {
    final GlobalKey<UpdatableImageState> imageKey = GlobalKey();
    var displayed =
        UpdatableImage(key: imageKey, initialValue: image[index].imagePath);

    return SizedBox(
      height: AppSizes.size150,
      width: AppSizes.size150,
      child: Card(
        child: CustomPopUpMenu(
          displayOptionals: true,
          items: [
            (
              key: "view",
              value: localizations.view,
              action: () {
                Navigator.push(
                  context,
                  MaterialPageRoute(
                    builder: (context) => ImageViews(
                      image.map((image) => image.imagePath).toList(),
                      name: image[index].importTime.toString(),
                      initialPage: index,
                    ),
                  ),
                );
              }
            ),
            (
              key: "edit",
              value: localizations.edit,
              action: () {
                imagePickerService
                    .showImageEditor(image[index].imagePath, context,
                        additionalCallback: () async {
                  await miniatureRepo.save(holder);
                  imageKey.currentState?.changeImage(image[index].imagePath);
                });
              }
            ),
            (
              key: "date",
              value: localizations.date,
              action: () async {
                final DateTime? picked = await showDatePicker(
                    context: context,
                    initialDate: image[index].importTime,
                    firstDate: DateTime(2015, 8),
                    lastDate: DateTime(2101));
                if (picked != null) {
                  image[index].importTime = picked;
                  await historyImagesRepo.save(image[index]);
                }
              }
            ),
            (
              key: "delete",
              value: localizations.delete,
              action: () async {
                await historyImagesRepo.delete(image[index]);
              }
            ),
            (
              key: "use",
              value: localizations.convertToDetail,
              action: () async {
                var paintJob = PaintJob(
                    imagePath: (await fileStorageService.moveFileToAppStorage(
                            XFile(image[index].imagePath)))
                        ?.path,
                    name: null,
                    paintLayers: iDrawableRepo.createList(),
                    positioned: positionedRepo.createList(),
                    importTime: DateTime.now());
                await holder.addPaintJob(paintJob);
              }
            )
          ],
          child: displayed,
        ),
      ),
    );
  }
}

class AddMiniatureHistoryCard extends StatelessWidget {
  const AddMiniatureHistoryCard({
    super.key,
    required this.index,
    required this.miniature,
    required this.onAdded,
  });

  final int index;
  final Miniature miniature;
  final VoidCallback onAdded;

  @override
  Widget build(BuildContext context) {
    return Padding(
      padding: AppPaddings.symmetricVertical24,
      child: SizedBox(
        width: AppSizes.size100,
        child: Card(
          shape: RoundedRectangleBorder(
              borderRadius: BorderRadius.circular(AppSizes.size100)),
          shadowColor: Colors.transparent,
          child: FilledButton(
            onLongPress: () async {
              (await imagePickerService.pickImages(ImageSource.gallery))
                  ?.forEach((path) {
                createHistoryImage(path, context);
                onAdded();
              });
            },
            onPressed: () async {
              var path = await imagePickerService.pickImage(ImageSource.camera);
              await createHistoryImage(path, context);
              onAdded();
            },
            child:
                ImageAssetService.getSvgIconSized(40, 40, AppIcons.takePicture),
          ),
        ),
      ),
    );
  }

  Future<DateTime?> getImageTimeTaken(XFile file, BuildContext context) async {
    var exif = await readExifFromFile(File(file.path));

    try {
      var timeTaken = exif.entries
          .firstWhere((elem) => elem.key == 'Image DateTime')
          .value
          .printable;
      DateFormat format = DateFormat("yyyy:MM:dd hh:mm:ss");

      return format.parse(timeTaken);
    } catch (e) {
      logger.w("Could not parser dateTime", error: e);
      Fluttertoast.showToast(
          // TODO will need to check this
          msg: localizations.errorWhileParsingDate,
          toastLength: Toast.LENGTH_LONG);
      return null;
    }
  }

  Future<void> createHistoryImage(
      XFile? imageFile, BuildContext context) async {
    if (imageFile != null) {
      var date = await getImageTimeTaken(imageFile, context) ?? DateTime.now();

      var historyImage = HistoryImage(
          (await fileStorageService.moveFileToAppStorage(imageFile))!.path,
          date);
      await miniature.addHistoryImage(historyImage);
      //TODO setState
    }
  }
}
