import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/extensions/string_extension.dart';
import 'package:miniature_painting_companion/utils.dart';
import 'package:miniature_painting_companion/views/design/app_sizes.dart';

import '../../../injector.dart';
import '../../../main.dart';
import '../../../utils/app_icons.dart';
import '../../design/paint_computed_color.dart';
import '../../design/slidable_card.dart';
import 'new_paint_layer_dialog.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

class PaintLayerWidget extends StatelessWidget {
  final PaintLayer layer;

  const PaintLayerWidget({super.key, required this.layer});

  Future<void> _showNoteDialog(BuildContext context) async {
    return showDialog<void>(
      context: context,
      builder: (BuildContext context) {
        return AlertDialog(
          title: Text(localizations.note),
          content: SingleChildScrollView(
            child: ListBody(
              children: <Widget>[
                Text(layer.note),
              ],
            ),
          ),
          actions: <Widget>[
            TextButton(
              child: Text(localizations.close),
              onPressed: () {
                Navigator.of(context).pop();
              },
            ),
          ],
        );
      },
    );
  }

  Widget _getPaintImage() {
    switch (layer.paint) {
      case Paint paint:
        if (paint.image == null) {
          if (paint.getHsvColor() != null) {
            return SizedBox(
              height: 50,
              width: 50,
              child: CircleAvatar(
                backgroundColor: paint.getHsvColor()!.toColor(),
              ),
            );
          }

          return ImageAssetService.getSvgIconSized(
              AppSizes.size50, AppSizes.size50, AppIcons.placeholderFilled);
        } else {
          return SvgPicture.string(paint.image!);
        }
      case UserPaint userpaint:
        return PaintComputedColor(
          paint: userpaint,
        );
    }
  }

  @override
  Widget build(BuildContext context) {
    var cardContent = Row(
      children: [
        const SizedBox(width: AppSizes.size8),
        Visibility(
            visible: layer.note.isNotEmpty,
            child: IconButton(
              icon: ImageAssetService.getSvgIconSized(
                  AppSizes.size34, AppSizes.size34, AppIcons.note),
              onPressed: () {
                _showNoteDialog(context);
              },
            )),
        Expanded(
            child: Container(
          alignment: Alignment.center,
          child: Column(
            children: [
              const SizedBox(height: AppSizes.size8),
              Utils.gothicText(layer.name),
              const SizedBox(height: AppSizes.size8),
              Text(layer.applicationType.name.camelToSentence().capitalize()),
              Text("${layer.dilution}%"),
              const SizedBox(height: AppSizes.size8),
            ],
          ),
        )),
        Row(
          children: [
            const SizedBox(width: AppSizes.size18),
            Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                const SizedBox(height: AppSizes.size8),
                _getPaintImage(),
                const SizedBox(height: AppSizes.size8),
                Visibility(
                    visible: layer.paint.type.isNotEmpty,
                    child: Row(children: [
                      Utils.gothicText(layer.paint.type.toUpperCase()),
                      const SizedBox(height: AppSizes.size8),
                    ])),
                const SizedBox(height: AppSizes.size8),
              ],
            ),
            const SizedBox(width: AppSizes.size18),
          ],
        ),
      ],
    );

    return SlidableCard(
        startAction: (context) async =>
            {await iDrawableRepo.delete(layer)},
        endAction: (otherContext) => NewPaintLayerDialog.show(
            context,
            false,
            layer,
            // we are only updating so no need for a real saveFunction
            (created) {}),
        child: cardContent);
  }
}
