import 'dart:convert';
import 'dart:io';

import 'package:file_picker/file_picker.dart';
import 'package:flutter/material.dart';
import 'package:intl/intl.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/injector.dart';
import 'package:permission_handler/permission_handler.dart';

import 'main.dart';

class DataExporter {
  static const String customPaintSeparator = "--customPaintSeparator--";

  static void _checkStoragePermission(dynamic context) async {
    var status = await Permission.manageExternalStorage.status;
    if (status.isRestricted) {
      status = await Permission.manageExternalStorage.request();
    }

    if (status.isDenied) {
      status = await Permission.manageExternalStorage.request();
    }
    if (status.isPermanentlyDenied) {
      ScaffoldMessenger.of(context).showSnackBar(SnackBar(
        backgroundColor: Colors.green,
        content: Text(localizations.accessExternalStorageRequired),
      ));
    }
  }

  static Future<void> exportData(context) async {
    _checkStoragePermission(context);

    var destination = await FilePicker.platform.getDirectoryPath();
    if (destination != null) {
      var formattedTime =
          DateFormat("yyyy_MM_dd_hh_mm_ss").format(DateTime.now());
      final File file =
          File("$destination/miniature_companion_export$formattedTime.json");
      final miniatures = miniatureRepo.values().toList();
      var mapped = miniatures.map((e) => e.toJson()).toList();
      await file.writeAsString(jsonEncode(mapped));

      await file.writeAsString(customPaintSeparator, mode: FileMode.append);

      final paints = paintRepo
          .values()
          .whereType<UserPaint>()
          .toList();
      var mappedCustomPaints = paints.map((e) => e.toJson()).toList();
      await file.writeAsString(jsonEncode(mappedCustomPaints),
          mode: FileMode.append);
    }
  }

  static Future<void> importData(context) async {
    _checkStoragePermission(context);
    FilePickerResult? result = await FilePicker.platform.pickFiles();
    if (result != null) {
      File file = File(result.files.single.path!);
      var content = file.readAsStringSync().split(customPaintSeparator);

      // Might be false for old export files (made before custom paint creation)
      if (content.length > 1) {
        // Start with custom paints, to use them in minis
        List<dynamic> importedCustomPaints = jsonDecode(content[1]);
        for (var element in importedCustomPaints) {
          await UserPaint.fromJson(element);
        }
      }

      List<dynamic> imported = jsonDecode(content[0]);

      for (var element in imported) {
        await Miniature.fromJson(element);
      }
    }
  }
}
