import colorsys
import csv
import os
import tempfile

import cairosvg
import numpy as np
from PIL import Image
from PIL import UnidentifiedImageError
from sklearn.cluster import KMeans

WHITE_TOLERANCE = 250  # ignore pixels where R,G,B are all above this value


def ensure_images_folder():
    os.makedirs('images', exist_ok=True)


def svg_to_png(svg_content, png_path, size=(64, 64)):
    with tempfile.NamedTemporaryFile(delete=False, suffix='.svg', mode='w',
                                     encoding='utf-8') as tmp_svg:
        tmp_svg.write(svg_content)
        tmp_svg_path = tmp_svg.name
    cairosvg.svg2png(url=tmp_svg_path, write_to=png_path, output_width=size[0],
                     output_height=size[1])
    os.remove(tmp_svg_path)


def get_average_color(image_path: str) -> tuple[int, int, int] | None:
    """
    Compute the average color of a PNG image, ignoring fully transparent pixels.

    Args:
        image_path (str): Path to the PNG image.

    Returns:
        tuple[int, int, int] | None: Average color as (R, G, B),
        or None if the image cannot be processed.
    """

    try:
        # Open image and ensure RGBA to handle transparency
        img = Image.open(image_path).convert("RGBA")

        # Resize to reduce computation (optional)
        img = img.resize((100, 100))

        # Get pixel data
        pixels = list(img.getdata())

        # Keep only non-transparent pixels
        opaque_pixels = [p for p in pixels if p[3] > 0]

        if not opaque_pixels:
            print("Warning: Image has no opaque pixels to compute an average color.")
            return None

        # Compute average per channel
        num_pixels = len(opaque_pixels)
        avg_r = sum(p[0] for p in opaque_pixels) // num_pixels
        avg_g = sum(p[1] for p in opaque_pixels) // num_pixels
        avg_b = sum(p[2] for p in opaque_pixels) // num_pixels

        value = (avg_r, avg_g, avg_b)
        print("parsing: " + image_path + " for value: " + ' '.join(str(val) for val in value))
        return value

    except FileNotFoundError:
        print(f"Error: File not found -> {image_path}")
    except UnidentifiedImageError:
        print(f"Error: Cannot identify or open image -> {image_path}")
    except Exception as e:
        print(f"Unexpected error: {e}")

    return None




def rgb_to_hsv(rgb):
    r, g, b = [x / 255.0 for x in rgb]
    h, s, v = colorsys.rgb_to_hsv(r, g, b)
    return (h * 360, s , v )


# --- Main script ---

input_csv = 'paints_images.csv'
output_csv = 'sorted_paints_images.csv'
ensure_images_folder()

results = []

with open(input_csv, newline='\n', encoding='utf-8') as f:
    reader = csv.reader(f)
    # next(reader)  # Skip header

    for row in reader:
        if len(row) < 3:
            continue
        type_val = row[0].strip('"')
        name = row[1].strip('"')
        svg_content = row[2].strip('"')

        try:
            png_path = os.path.join('images', f'{name}.png')
            svg_to_png(svg_content, png_path)
            rgb = get_average_color(png_path)
            h, s, v = rgb_to_hsv(rgb)
            print(f"path: {png_path} result: {h} {s} {v}")

            results.append({
                'Type': type_val,
                'Name': name,
                'h': h,
                's': s,
                'v': v,
                'svg': svg_content
            })

        except Exception as e:
            print(f'Error processing {name}: {e}')

# Sort results by HSV
results.sort(key=lambda x: (x['h'], x['s'], x['v']))

# Write output CSV with numeric HSV columns
with open(output_csv, mode='w', newline='\n', encoding='utf-8') as f:
    writer = csv.writer(f)
    writer.writerow(['Type', 'Name', 'h', 's', 'v', 'svg'])
    for row in results:
        writer.writerow([
            row['Type'],
            row['Name'],
            row['h'],
            row['s'],
            row['v'],
            row['svg']
        ])
