import 'package:flutter/material.dart';
import 'package:flutter_svg/flutter_svg.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';
import 'package:miniature_painting_companion/utils/app_icons.dart';

void main() {
  group('ImageAssetService', () {
    testWidgets('getImageOrPlaceholder returns placeholder when path is null',
        (tester) async {
      final widget = ImageAssetService.getImageOrPlaceholder(null);
      expect(widget, isA<SvgPicture>());
      // We can't easily verify the asset path inside SvgPicture without more complex setup or finding by type/key if set.
      // But we know it returns the result of getSvgIcon(AppIcons.placeholder)
    });

    testWidgets('getImageOrPlaceholder returns Ink.image when path is provided',
        (tester) async {
      // We can't easily test File images in widget tests without IO overrides,
      // but we can check the widget structure.
      // However, Image.file creates an Image widget.
      // The service returns Ink.image which is an Ink widget with decoration image.
      // Let's just verify it returns an Ink widget.
      final widget = ImageAssetService.getImageOrPlaceholder('/tmp/test.jpg');
      expect(widget, isA<Ink>());
    });

    testWidgets(
        'getImageOrPlaceholderSized returns sized placeholder when path is null',
        (tester) async {
      final widget =
          ImageAssetService.getImageOrPlaceholderSized(100, 100, null);
      expect(widget, isA<SizedBox>());
      final sizedBox = widget as SizedBox;
      expect(sizedBox.height, 100);
      expect(sizedBox.width, 100);
      expect(sizedBox.child, isA<SvgPicture>());
    });

    testWidgets(
        'getImageOrPlaceholderSized returns SizedBox with Ink when path is provided',
        (tester) async {
      final widget = ImageAssetService.getImageOrPlaceholderSized(
          100, 100, '/tmp/test.jpg');
      expect(widget, isA<SizedBox>());
      final sizedBox = widget as SizedBox;
      expect(sizedBox.height, 100);
      expect(sizedBox.width, 100);
      expect(sizedBox.child, isA<Ink>());
    });

    testWidgets('getSvgIconSized returns correctly sized SvgPicture',
        (tester) async {
      final widget =
          ImageAssetService.getSvgIconSized(50, 50, AppIcons.placeholder);
      expect(widget, isA<SizedBox>());
      final sizedBox = widget as SizedBox;
      expect(sizedBox.height, 50);
      expect(sizedBox.width, 50);
      expect(sizedBox.child, isA<SvgPicture>());
    });

    testWidgets('getSvgIcon returns SvgPicture', (tester) async {
      final widget = ImageAssetService.getSvgIcon(AppIcons.placeholder);
      expect(widget, isA<SvgPicture>());
    });

    testWidgets('getSvgIconOrPlaceholder returns placeholder when path is null',
        (tester) async {
      final widget = ImageAssetService.getSvgIconOrPlaceholder(null);
      expect(widget, isA<SvgPicture>());
    });

    testWidgets(
        'getSvgIconOrPlaceholder returns SvgPicture.string when path is provided',
        (tester) async {
      final widget = ImageAssetService.getSvgIconOrPlaceholder('<svg></svg>');
      expect(widget, isA<SvgPicture>());
    });

    testWidgets('getPlaceHolderSvg returns SvgPicture', (tester) async {
      final widget = ImageAssetService.getPlaceHolderSvg();
      expect(widget, isA<SvgPicture>());
    });

    testWidgets('getPngIcon returns correctly sized Image', (tester) async {
      final widget = ImageAssetService.getPngIcon(40, 40,
          AppPng.appIcon); // Assuming AppPng.logo exists, or pick one that does
      expect(widget, isA<SizedBox>());
      final sizedBox = widget as SizedBox;
      expect(sizedBox.height, 40);
      expect(sizedBox.width, 40);
      expect(sizedBox.child, isA<Image>());
    });
  });
}
