import 'dart:convert';
import 'dart:io';

import 'package:flutter_test/flutter_test.dart';
import 'package:image_picker/image_picker.dart';
import 'package:miniature_painting_companion/services/file_storage_service.dart';
import 'package:path_provider_platform_interface/path_provider_platform_interface.dart';
import 'package:plugin_platform_interface/plugin_platform_interface.dart';

class MockPathProviderPlatform extends Fake
    with MockPlatformInterfaceMixin
    implements PathProviderPlatform {
  final String tmpPath;

  MockPathProviderPlatform(this.tmpPath);

  @override
  Future<String?> getApplicationDocumentsPath() async {
    return tmpPath;
  }
}

void main() {
  group('FileStorageService', () {
    late FileStorageService service;
    late Directory tempDir;

    setUp(() async {
      service = FileStorageService();
      tempDir = await Directory.systemTemp.createTemp('file_storage_test');
      PathProviderPlatform.instance = MockPathProviderPlatform(tempDir.path);
    });

    tearDown(() async {
      await tempDir.delete(recursive: true);
    });

    test('moveFileToAppStorage moves file correctly', () async {
      final sourceFile = File('${tempDir.path}/source.txt');
      await sourceFile.writeAsString('test content');
      final xFile = XFile(sourceFile.path);

      final result = await service.moveFileToAppStorage(xFile);

      expect(result, isNotNull);
      expect(File(result!.path).existsSync(), isTrue);
      expect(File(result.path).readAsStringSync(), 'test content');
      // Verify it is in the "app documents" directory (which is our tempDir)
      expect(result.path.startsWith(tempDir.path), isTrue);
    });

    test('moveFileToAppStorage generates unique paths for same file', () async {
      final sourceFile = File('${tempDir.path}/source_unique.txt');
      await sourceFile.writeAsString('test content');
      final xFile = XFile(sourceFile.path);

      final result1 = await service.moveFileToAppStorage(xFile);
      final result2 = await service.moveFileToAppStorage(xFile);

      expect(result1, isNotNull);
      expect(result2, isNotNull);
      expect(result1!.path, isNot(equals(result2!.path)));
      expect(File(result1.path).existsSync(), isTrue);
      expect(File(result2.path).existsSync(), isTrue);
    });

    test('moveFileToAppStorage returns null when input is null', () async {
      final result = await service.moveFileToAppStorage(null);
      expect(result, isNull);
    });

    test('filePathToBase64 encodes file correctly', () async {
      final file = File('${tempDir.path}/test_base64.txt');
      final content = 'hello world';
      await file.writeAsBytes(utf8.encode(content));

      final base64String = service.filePathToBase64(file.path);

      final expectedBase64 = base64Encode(utf8.encode(content));
      expect(base64String, expectedBase64);
    });

    test('filePathToBase64 returns null when path is null', () {
      final result = service.filePathToBase64(null);
      expect(result, isNull);
    });

    test('base64toFile decodes and writes file correctly', () async {
      final targetPath = '${tempDir.path}/decoded.txt';
      final content = 'decoded content';
      final base64String = base64Encode(utf8.encode(content));

      final result = service.base64toFile(targetPath, base64String);

      expect(result, targetPath);
      expect(File(targetPath).existsSync(), isTrue);
      expect(File(targetPath).readAsStringSync(), content);
    });

    test('base64toFile does nothing if file already exists', () async {
      final targetPath = '${tempDir.path}/existing.txt';
      await File(targetPath).writeAsString('original content');
      final base64String = base64Encode(utf8.encode('new content'));

      final result = service.base64toFile(targetPath, base64String);

      expect(result, targetPath);
      expect(File(targetPath).readAsStringSync(), 'original content');
    });
  });
}
