import 'dart:io';

import 'package:flutter/cupertino.dart';
import 'package:hive_ce/hive.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/hive/hive_registrar.g.dart';
import 'package:miniature_painting_companion/injector.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:test/test.dart';

import 'test_utils.dart';

void main() {
  group('Test model exports and import', () {
    setUpAll(() async {
      WidgetsFlutterBinding.ensureInitialized();

      Hive.init((await Directory.systemTemp.createTemp()).path);
      Hive.registerAdapters();

      SharedPreferences.setMockInitialValues({});

      await setupInjector();
    });

    setUp(() async {
      await paintRepo
          .add(Paint(type: "type", name: "testing_paint", h: 0, s: 0, v: 0));
    });

    tearDown(() async {
      await miniatureRepo.clear();
      await historyImagesRepo.clear();
      await paintJobRepo.clear();
      await iDrawableRepo.clear();
      await paintRepo.clear();
      await positionedRepo.clear();
    });

    tearDownAll(() {
      clearTestImages();
    });

    group('Deletion tests', () {
      test('Models should clear their file image on delete', () async {
        var miniImage = createTestImage().path;
        var paintJobImage = createTestImage().path;
        var stickerImage = createTestImage().path;
        var historyImage = createTestImage().path;

        var miniature = Miniature(
            name: "name",
            imagePath: miniImage,
            paintJobs: paintJobRepo.createList(),
            historyImages: historyImagesRepo.createList(),
            importTime: DateTime.now());
        miniatureRepo.add(miniature);

        var paintJob = PaintJob(
            name: "name",
            imagePath: paintJobImage,
            paintLayers: iDrawableRepo.createList(),
            positioned: positionedRepo.createList(),
            importTime: DateTime.now());
        await miniature.addPaintJob(paintJob);

        var history = HistoryImage(historyImage, DateTime.now());
        await miniature.addHistoryImage(history);

        var stickerLayer = StickerLayer("stickerLayer", stickerImage);
        await paintJob.addPaintLayer(stickerLayer);

        var paintLayer = PaintLayer(
            name: "fake",
            paint: paintRepo.values().first,
            note: "",
            applicationType: ApplicationType.dryBrush,
            dilution: 0);

        var positioned =
            PositionedLayer(dx: 0, dy: 0, layers: iDrawableRepo.createList());
        await paintJob.addPositioned(positioned);
        await positioned.addLayer(paintLayer);

        var secondPaintLayer = PaintLayer(
            name: "fake",
            paint: paintRepo.values().first,
            note: "",
            applicationType: ApplicationType.dryBrush,
            dilution: 0);
        await positioned.addLayer(secondPaintLayer);

        var customPaint = UserPaint(
            id: 'fake', type: 'custom', name: 'custom', components: []);
        await paintRepo.add(customPaint);

        await miniatureRepo.delete(miniature);

        expect(File(miniImage).existsSync(), false);
        expect(File(paintJobImage).existsSync(), false);
        expect(File(historyImage).existsSync(), false);
        expect(File(stickerImage).existsSync(), false);

        expect(miniatureRepo.values().isEmpty, true);
        expect(historyImagesRepo.values().isEmpty, true);
        expect(paintJobRepo.values().isEmpty, true);
        expect(iDrawableRepo.values().isEmpty, true);
        expect(positionedRepo.values().isEmpty, true);

        expect(
            paintRepo
                .values()
                .toList()
                .any((act) => act.key == customPaint.key),
            true);
        expect(
            paintRepo
                .values()
                .toList()
                .any((act) => act.name == "testing_paint"),
            true);
      });
    });
  });
}
